/*
    File:       NameAndAddress.m

    Contains:   The private implementation for the NameAndAddress class.
                NameAndAddress is a cover for the functionality of the the NSHost class.
                NameAndAddress acts as a controller that allows hostname or IP address lookups.
                In this interface, the IPAddressField is used by the hostNameButton,
                and the hostnameField is used by the IPAddressButton. The interface
                disables a button if its corresponding field is empty.

    Written by: Guy @Werk

    Created:    June 1997

    Copyright:  (c)1997 by Apple Computer, Inc., all rights reserved.

    Change History (most recent first):
                 April 1997 -- Created for the initial Rhapsody training.

    You may incorporate this sample code into your applications without
    restriction, though the sample code has been provided "AS IS" and the
    responsibility for its operation is 100% yours.  However, what you are
    not permitted to do is to redistribute the source as "DSC Sample Code"
    after having made changes. If you're going to re-distribute the source,
    we require that you make it clear in the source that the code was
    descended from Apple Sample Code, but that you've made changes.
*/

#import "NameAndAddress.h"

@implementation NameAndAddress

// D E L E G A T A T I O N   A N D   N O T I F I C A T I O N

// NSApplication notification method
// gets the host name and IP address for the local host
// places the window onscreen
- (void)applicationDidFinishLaunching:(NSNotification *)notification
{
    NSHost *host;

    host = [NSHost currentHost];
    if (host) {
        [hostNameField setStringValue:[host name]];
        [self getIPAddress:nil];
    }    
    [[hostNameField window] makeKeyAndOrderFront:nil];
}

// NSControl notification method
// checks as each character is entered in the text field
// if the result is an empty text field, disables the corresponding button
- (void)controlTextDidChange:(NSNotification *)notification
{
    NSControl *textField;
    NSButton *button;

    // get the field that is sending the notification
    textField = [notification object];
    // get the button that corresponds to the field
    button = (textField == IPAddressField) ? hostNameButton : IPAddressButton;
    // disable that button if the field is empty
    [[textField stringValue] isEqualToString:@""] ? [button setEnabled:NO] : [button setEnabled:YES];
}

// A C T I O N S

// gets a host name for the IP adress in the IPAddressField
- (void)getHostName:(id)sender
{
    NSHost *host;
    NSString *IPAddress;

    // get the IP address
    IPAddress = [IPAddressField stringValue];
    // get a host for the IP address
    host = [NSHost hostWithAddress:IPAddress];

    // if there is no host for that IP address,
    // empty the host name and disable the corresponding button
    if (!host) {
        [hostNameField setStringValue:@""];
        [IPAddressButton setEnabled:NO];
        NSBeep();
        return;
    }

    // get and display the host name
    [hostNameField setStringValue:[host name]];
    // enable the corresponding button
    [IPAddressButton setEnabled:YES];
}

// get an IP address for the host name in the hostNameField
- (void)getIPAddress:(id)sender
{
    NSHost *host;
    NSString *hostName;

    // get the host name
    hostName = [hostNameField stringValue];
    // get the host for the host name
    host = [NSHost hostWithName:hostName];

    // if there is no host for that host name,
    // empty the IP address and disable the corresponding button
    if (!host) {
        [IPAddressField setStringValue:@""];
        [hostNameButton setEnabled:NO];
        NSBeep();
        return;
    }

    // get and display the IP address
    [IPAddressField setStringValue:[host address]];
    // enable the corresponding button
    [hostNameButton setEnabled:YES];
}

@end
