//----------------------------------------------------------------------------
// KILL.c
// Copyright (C)1998 by Eric Sunshine <sunshine@sunshineco.com>
//
// A Windows NT program which provides a command-line interface to task
// manipulation.  With this program, Windows NT processes can be listed and
// killed (via PID or process name).
//
// The list of running applications is gleaned from the Windows "performance
// data" which is stored under the dynamic Registry key HKEY_PERFORMANCE_DATA.
// See the header <winperf.h> for details about the data stored therein.
//
// The comments in this program do not necessarily attempt to demystify the
// structure of the performance data, nor the methods by which it is accessed.
// A prior understanding of this particular area of Windows NT will help in
// understanding the inner workings of this program.
//----------------------------------------------------------------------------
//----------------------------------------------------------------------------
// Revision History
// 22 May 1998 Eric Sunshine - Created.
//----------------------------------------------------------------------------
#include <stdio.h>
#include <stdlib.h>
#include <windows.h>
#define __stdcall
#include <winperf.h>
#undef __stdcall
typedef unsigned long ulong;

#define VERSION "1.0"
#define PERFLIB_KEY "Software\\Microsoft\\Windows NT\\CurrentVersion\\PerfLib"
#define PERFLIB_COUNTER "Counter"
#define PROCESS_KEY "process"
#define PROCESS_ID_KEY "id process"

#define ERR_CHECK(X) if (X != ERROR_SUCCESS) fatal(handle)

typedef int (*KL_IterFunc)( char const* name, ulong pid, void* );

//----------------------------------------------------------------------------
// Command-line options
//----------------------------------------------------------------------------
enum { C_ERROR, C_USAGE, C_LIST, C_PID, C_PREFIX };
static struct { char const* cmd; int tag; } const CMDS[] =
{
    { "?",    C_USAGE },
    { "h",    C_USAGE },
    { "help", C_USAGE },
    { "l",    C_LIST  },
    { "list", C_LIST  }
};
#define NUM_CMDS (sizeof(CMDS) / sizeof(CMDS[0]))


//----------------------------------------------------------------------------
// fatal
//----------------------------------------------------------------------------
static void fatal( HANDLE handle )
{
    if (handle != 0)
        RegCloseKey( handle );
    fprintf( stderr, "Fatal error\n" );
    exit(1);
}


//----------------------------------------------------------------------------
// usage
//----------------------------------------------------------------------------
static void usage( void )
{
    printf(
        "KILL version " VERSION " for Windows NT\n"
        "Copyright (C) 1998 by Eric Sunshine <sunshine@sunshineco.com>\n\n"
        "Usage: KILL -list\n"
        "       KILL <pid>\n"
        "       KILL <prefix>\n\n"
        "-list    Lists the name and process-id of each running program.\n"
        "<pid>    The process-id of the task to be killed.\n"
        "<prefix> A pattern which may be an entire process name, or a\n"
        "         prefix.  Kills each process which matches prefix.\n" );
}


//----------------------------------------------------------------------------
// parse_command
//----------------------------------------------------------------------------
static int parse_command( char const* s )
{
    int i;
    for (i = 0; i < NUM_CMDS; i++)
        if (stricmp( s, CMDS[i].cmd ) == 0)
            return CMDS[i].tag;
    return C_ERROR;
}


//----------------------------------------------------------------------------
// classify -- Classify input argument (command vs. pid vs. prefix vs. error)
//----------------------------------------------------------------------------
static int classify( char const* s )
{
    if (*s == '-' || *s == '/')
        return parse_command( s + 1 );
    else if (isdigit(*s))
        return C_PID;
    else
        return C_PREFIX;
}


//----------------------------------------------------------------------------
// extract_name -- Extract process name from performance instance.
//----------------------------------------------------------------------------
static void extract_name( PPERF_INSTANCE_DEFINITION instance,
                      char* buff, int size )
{
    if (WideCharToMultiByte( CP_ACP, 0, (LPCWSTR)
        ((char*)instance + instance->NameOffset), -1, buff, size, 0, 0) == 0)
        strcpy( buff, "<unknown>" );
}


//----------------------------------------------------------------------------
// extract_pid -- Extract pid from performance counter.
//----------------------------------------------------------------------------
static ulong extract_pid( PPERF_COUNTER_BLOCK counter, ulong offset )
{
    return *((ulong*)((char*)counter + offset));
}


//----------------------------------------------------------------------------
// report_killing
//----------------------------------------------------------------------------
static void report_killing( char const* name, ulong pid, int killed )
{
    printf( "process #%lu ", pid );
    if (name != 0)
        printf( "[%s] ", name );
    if (!killed)
        printf( "could not be " );
    printf( "killed\n" );
}


//----------------------------------------------------------------------------
// kill_process
//----------------------------------------------------------------------------
static void kill_process( char const* name, ulong pid )
{
    int killed = 0;
    HANDLE const handle = OpenProcess( PROCESS_ALL_ACCESS, 0, pid );
    if (handle != 0)
        {
        killed = TerminateProcess( handle, 1 );
        CloseHandle( handle );
        }
    report_killing( name, pid, killed );
}


//----------------------------------------------------------------------------
// kill_prefix -- Kill processes which match 'prefix'.
//----------------------------------------------------------------------------
static int kill_prefix( char const* name, ulong pid, void* data )
{
    char const* const s = (char const*)data;
    if (strnicmp( name, s, strlen(s) ) == 0)
        kill_process( name, pid );
    return 1;
}


//----------------------------------------------------------------------------
// list_process
//----------------------------------------------------------------------------
static int list_process( char const* name, ulong pid, void* data )
{
    printf( "%5lu %s\n", pid, name );    
    return 1;
}


//----------------------------------------------------------------------------
// parse_counter
//	The "Counter" value in the PerfLib from HKEY_LOCAL_MACHINE consists
//	of a batch of null terminated strings.  The batch itself is also 
//	terminated by a null byte.  The counters always come in pairs of
//	strings making up the 2-tuple (id,name).  Since 'pattern' is matched
//	against 'name', it is necessary to keep a back-pointer to the 'id'.
//----------------------------------------------------------------------------
static ulong parse_counter( char const* p, char const* pattern )
{
    char const* prev = "";
    while (*p != 0)
        {
        if (stricmp( p, pattern ) == 0)
            return atol( prev );		/*** RETURN ***/
	prev = p;
	while (*p != 0) p++;
        p++;
        }
    return 0;
}


//----------------------------------------------------------------------------
// get_process_values -- Get the counter IDs of "process" and "id process".
//----------------------------------------------------------------------------
static void get_process_values( ulong* process_value, ulong* process_id_value )
{
    int const BUFF_SIZE = 1024;
    ulong len;
    HKEY handle;
    char* buff = (char*)malloc( BUFF_SIZE );

    sprintf( buff, "%s\\%03x", PERFLIB_KEY,
        MAKELANGID(LANG_ENGLISH, SUBLANG_NEUTRAL) );

    ERR_CHECK(RegOpenKeyEx( HKEY_LOCAL_MACHINE, buff, 0, KEY_READ, &handle ));
    ERR_CHECK(RegQueryValueEx( handle, PERFLIB_COUNTER, 0, 0, 0, &len ));

    if (len > BUFF_SIZE) buff = (char*)realloc( buff, len );
    memset( buff, 0, len );

    ERR_CHECK(RegQueryValueEx(handle,PERFLIB_COUNTER,0,0,(LPBYTE)buff,&len));
    *process_value = parse_counter( buff, PROCESS_KEY );
    *process_id_value = parse_counter( buff, PROCESS_ID_KEY );

    RegCloseKey( handle );
    free( buff );
}


//----------------------------------------------------------------------------
// read_perf_data -- Most Registry data, can be accessed by first calling
// RegQueryValueEx() to determine the data size, and then a second time to
// actually read the data.  With the dynamic performance data, however, this
// does not work, and one must actually try reading it without knowing its
// size ahead of time.  Consequently, it is necessary to place the read
// attempt in a loop which increases the buffer size upon each iteration until
// the data is succesfully read.
//----------------------------------------------------------------------------
static PPERF_DATA_BLOCK read_perf_data( ulong process_value,
                                        ulong process_id_value )
{
    int const BUFF_SIZE = 8 * 1024;
    ulong len = BUFF_SIZE;
    char* buff = (char*)malloc( len );
    HKEY const handle = HKEY_PERFORMANCE_DATA;
    char pkey[32];
    sprintf( pkey, "%ld", process_value );

    while (1)
        {
        ulong const rc = RegQueryValueEx( handle,pkey,0,0,(LPBYTE)buff,&len );
        if (rc == ERROR_SUCCESS)
            break;
        else if (rc != ERROR_MORE_DATA)
            fatal( handle );
        else
            {
            len += BUFF_SIZE;
            buff = (char*)realloc( buff, len );
            }
        }
              
    RegCloseKey( handle );
    return (PPERF_DATA_BLOCK)buff;
}


//----------------------------------------------------------------------------
// validate_perf_data
//----------------------------------------------------------------------------
static void validate_perf_data( PPERF_DATA_BLOCK perf_data )
{
    if (perf_data->Signature[0] != 'P' ||
        perf_data->Signature[1] != 'E' ||
        perf_data->Signature[2] != 'R' ||
        perf_data->Signature[3] != 'F')
        fatal(0);
}


//----------------------------------------------------------------------------
// get_process_id_offset -- Get the offset of the "id process" counter within
// the instance counter block in order to quickly access the value in each
// instance.
//----------------------------------------------------------------------------
static ulong get_process_id_offset( PPERF_OBJECT_TYPE perf_obj, ulong id_val )
{
    ulong n = perf_obj->NumCounters;
    PPERF_COUNTER_DEFINITION def = (PPERF_COUNTER_DEFINITION)
        ((char*)perf_obj + perf_obj->HeaderLength);
    for (; n-- > 0; def=(PPERF_COUNTER_DEFINITION)((char*)def+def->ByteLength))
        if (id_val == def->CounterNameTitleIndex)
            return def->CounterOffset;
    fatal(0);
    return 0;
}


//----------------------------------------------------------------------------
// iterate_perf_data -- Iterate over each instance of performance data (an
// instance consists of performance data from a single running task) and call
// a user-supplied function each time.
//----------------------------------------------------------------------------
static void iterate_perf_data( PPERF_OBJECT_TYPE perf_obj,
    ulong process_id_offset, KL_IterFunc func, void* data )
{
    ulong n = perf_obj->NumInstances;
    PPERF_INSTANCE_DEFINITION instance = (PPERF_INSTANCE_DEFINITION)
	((char*)perf_obj + perf_obj->DefinitionLength);

    while (n-- > 0)
        {
        char name[ MAX_PATH ];
        PPERF_COUNTER_BLOCK counter = (PPERF_COUNTER_BLOCK)
            ((char*)instance + instance->ByteLength);
        ulong const pid = extract_pid( counter, process_id_offset );
        extract_name( instance, name, sizeof(name) );

	if (!func( name, pid, data ))
            break;

        instance = (PPERF_INSTANCE_DEFINITION)
            ((char*)counter + counter->ByteLength);
        }
}


//----------------------------------------------------------------------------
// perform -- Load the performance data, and apply the appropriate "action"
// over each counter instance.
//----------------------------------------------------------------------------
static void perform( int tag, char const* v )
{
    PPERF_DATA_BLOCK perf_data;
    PPERF_OBJECT_TYPE perf_obj;
    ulong process_value, process_id_value, process_id_offset;

    get_process_values( &process_value, &process_id_value );
    perf_data = read_perf_data( process_value, process_id_value );
    validate_perf_data( perf_data );

    perf_obj = (PPERF_OBJECT_TYPE)((char*)perf_data + perf_data->HeaderLength);
    process_id_offset = get_process_id_offset( perf_obj, process_id_value );

    if (tag == C_PREFIX)
        iterate_perf_data( perf_obj, process_id_offset, kill_prefix, (void*)v);
    else // (tag == C_LIST)
        iterate_perf_data( perf_obj, process_id_offset, list_process, 0 );
    
    free( perf_data );
}


//----------------------------------------------------------------------------
// main
//----------------------------------------------------------------------------
int main ( int argc, char const* const argv[] )
{
    int ok = 0;
    if (argc == 2)
        {
        char const* const v = argv[1];
        int const tag = classify(v);
        if (tag == C_ERROR)
            fprintf( stderr, "Unrecognized command: %s\n\n", v );
        else if (tag != C_USAGE)
            {
            ok = 1;
            if (tag == C_PID)
                kill_process( 0, atol(v) );
            else
                perform( tag, v );
            }
        }

    if (!ok)
        usage();
    return !ok;
}
