/* Expense.m
 *
 * You may freely copy, distribute and reuse the code in this example.
 * Apple Computer, Inc. disclaims any warranty of any kind, expressed or implied, as to
 * its fitness for any particular use.
 */

#import "Expense.h"

NSString *defaultNewCategory = @"defaultNewCategory";

@implementation Expense

- (id)init
{
    NSUserDefaults *defaults;
    NSString *newCategory;

    [super init];

    defaults = [NSUserDefaults standardUserDefaults];
    newCategory = [defaults stringForKey:defaultNewCategory];
    // default attribute values
    if (!newCategory) newCategory = @"New";

    // default attribute values
    [self setCategory: newCategory];
    [self setDate: [NSCalendarDate calendarDate]];	// today
    [self setAmount: [NSNumber numberWithFloat:0]];
    return self;
}

- (void)dealloc
{
    [category release];
    [date release];
    [amount release];
    [super dealloc];
}

// This is one way to write a set accessor method:
// Since NSMutableString is a subclass of NSString, we have no guarantee
// that the caller will not decide to change the category at some point in
// the future, so we make a copy, which returns an immutable object. This is
// simply setting our attribute to a snapshot of whatever the string is at
// the time our set method is invoked. Efficiency afficionados need not
// worry - if the object is, in reality, an immutable string, then the copy
// method is implemented in a lightweight manner - it almost certainly
// simply sends a retain message and returns the original.
- (void)setCategory:(NSString *)newCategory
{
    newCategory = [newCategory copy];
    [category release];
    category = newCategory;
}

- (NSString *)category
{
    return category;
}

// This is another way ...
// NB: Since the original object is released before we make the new copy,
// under certain circumstances the old object may actually be the new object.
// If we used release in this case, and we were the only owner of the object
// (and thus the only ones retaining it), the second line which does the
// copy would actually be messaging a freed object. So we use autorelease
// to preserve the lifetime of the original object past the end of the method.
- (void)setDate:(NSCalendarDate *)newDate
{
    [date autorelease];
    date = [newDate copy];
}

- (NSCalendarDate *)date
{
    return date;
}

// And this is a third variation:
// Here we test for the special case mentioned earlier. In cases where
// a lot of unique immutable strings are being repeatedly passed around
// as attributes, or attributes are often reset to the same value, this
// approach can be a little more efficient, at the cost of looking a little
// odd.
- (void)setAmount:(NSNumber *)newAmount
{
    if (newAmount == amount) return;
    [amount release];
    amount = [newAmount copy];
}

- (NSNumber *)amount
{
    return amount;
}

- (BOOL)isBillable
{
    return billable;
}

- (void)setIsBillable:(BOOL)isBillable
{
    billable = isBillable;
}

// The initWithCoder: method invokes its own set methods here to ensure the
// retain/release policy is correctly maintained. It also allows specialised
// subclasses to intervene in a single place if they wish to modify the way
// these attributes are handled.
- (id)initWithCoder:(NSCoder *)coder
{
    BOOL temp;
    [self setCategory: [coder decodeObject]];
    [self setDate: [coder decodeObject]];
    [self setAmount: [coder decodeObject]];
    [coder decodeValuesOfObjCTypes:"c", &temp];
    [self setIsBillable:temp];
    return self;
}

// For similar reasons, encodeWithCoder: uses its own accessor methods
// rather than directly accessing its ivars.
- (void)encodeWithCoder:(NSCoder *)coder
{
    BOOL temp = [self isBillable];
    [coder encodeObject: [self category]];
    [coder encodeObject: [self date]];
    [coder encodeObject: [self amount]];
    [coder encodeValuesOfObjCTypes:"c", &temp];
}

@end
