/* SortingDataSource.m
 *
 * You may freely copy, distribute and reuse the code in this example.
 * Apple Computer, Inc. disclaims any warranty of any kind, expressed or implied, as to
 * its fitness for any particular use.
 */

#import "SortingDataSource.h"
#import "SortableTableRow.h"

NSString *sortByDefaultName = @"sortByColumn";

@implementation SortingDataSource

- (id)initWithDataSource:(id)theDataSource
{
    self = [super init];
    rows = [[NSMutableArray alloc] init];
    [self setDataSource:theDataSource];
    return self;
}

- (id)init
{
    return [self initWithDataSource:nil];
}

- (void)dealloc
{
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    [rows release];
    [dataSource release];
    [super dealloc];
}

- (void)defaultsChanged:(NSUserDefaults *)defaults
{
    // Check if the default sort column is set, and if so, use it
    NSString *columnName = [defaults stringForKey:sortByDefaultName];

    if (columnName && [columnName length] > 0)
    {
        NSArray *columns = [tableView tableColumns];
        if (columns)
        {
            NSTableColumn *column;
            NSEnumerator *colEnum = [columns objectEnumerator];
            while (column = [colEnum nextObject])
            {
                id ident = [column identifier];
                if ([ident caseInsensitiveCompare:columnName] == NSOrderedSame)
                {
                    break;
                }
            }

            // No column match implies sorting off. Skip resort if the
            // column hasn't really changed
            if (column != sortByColumn)
            {
                [self setSortByColumn:column];
                [self reloadData];
            }
        }
    }

}

- (void)defaultsDidChangeNotification:(NSNotification *)notification
{
    NSUserDefaults *defaults = [notification object];
    [self defaultsChanged:defaults];
}

- awakeFromNib
{
    // Set up the sortBy column if there is a default one
    NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
    [self defaultsChanged:defaults];

    if ([tableView target] == self)
    {
        // We have to disable the single action as it's confusing
        [tableView setDoubleAction:@selector(sortBy:)];
        [tableView setAction:NULL];
    }

    // Register for defaults updates
    [[NSNotificationCenter defaultCenter]
               addObserver:self
               selector:@selector(defaultsDidChangeNotification:)
               name:NSUserDefaultsDidChangeNotification
               object:defaults];
    return self;
}

- (void)setDataSource:(id)theDataSource
{
    [theDataSource retain];
    [dataSource release];
    dataSource = theDataSource;
    isSortNeeded = YES;
}

- (id)dataSource
{
    return dataSource;
}

- (void)setSortByColumn:(NSTableColumn *)aColumn;
{
    [aColumn retain];
    [sortByColumn release];
    sortByColumn = aColumn;
    isSortNeeded = YES;
}

- (NSTableColumn *)sortByColumn
{
    return sortByColumn;
}

- (void)sortBy:(id)sender
{
    NSArray *columns;
    unsigned index;
    NSTableColumn *column;
    columns = [sender tableColumns];
    index = [sender clickedColumn];
    column = [columns objectAtIndex:index];
    [self setSortByColumn:column];
    [self sort:sender];
}

- (void)sort:(id)sender
{
    isSortNeeded = YES;
    [self reloadData];
}

- (NSTableView *)tableView
{
    return tableView;
}

- (void)noteNumberOfRowsChanged
{
    isSortNeeded = YES;
    [tableView noteNumberOfRowsChanged];
}

- (void)fetchData
{
    int i, nrows;

    [rows removeAllObjects];
    nrows = [dataSource numberOfRowsInTableView:tableView];

    // Re-creating all the row objects is probably not optimal -
    // this is only strictly needed when the number of rows changes
    
    for (i = 0; i < nrows; i++)
    {
        id row = [[SortableTableRow alloc] initWithRow:i];
        [rows addObject:row];
        if (sortByColumn)
        {
            id	cellKey = [dataSource tableView:(NSTableView *)self
                                      objectValueForTableColumn:sortByColumn
                                      row:i];
            [row setSortKey:cellKey];
        }
        [row release];	// lose our ownership - passed to rows array
    }
}

- (void)reloadData
{
    if (isSortNeeded)
    {
        // We have an array of rows, which contain some EOs
        // that need to be sorted according to the selected column
        // So they are referenced via our row objects that keep the
        // mapping between the original rows and the new (sorted) ones.
        [self fetchData];
        [rows sortUsingSelector:@selector(compare:)];
        isSortNeeded = NO;
    }
    [tableView reloadData];
}

- (int)numberOfRowsInTableView:(NSTableView *)aTableView
{
    return [dataSource numberOfRowsInTableView:(NSTableView *)self];
}

- (unsigned)originalRow:(unsigned)sortedRow
{
    id sortedObject = [rows objectAtIndex:sortedRow];
    return [sortedObject originalRow];
}

- (id)tableView:(NSTableView *)tableView objectValueForTableColumn:(NSTableColumn *)tableColumn row:(int)row
{
    id	object = [dataSource tableView:(NSTableView *)self
                       objectValueForTableColumn:tableColumn
                       row:[self originalRow:row]];
    return object;
}

- (void)tableView:(NSTableView *)tableView setObjectValue:(id)object forTableColumn:(NSTableColumn *)tableColumn row:(int)row
{
    id sortedRow = [rows objectAtIndex:row];
    [dataSource tableView:(NSTableView *)self setObjectValue:object
           forTableColumn:tableColumn row:[sortedRow originalRow]];

    // We assume that the columns are always the same object being passed
    // around - it seems to be the case
    if (tableColumn == sortByColumn)
    {
        [sortedRow setSortKey:object];
        isSortNeeded = YES;	// Could post a delayed msg for this
        [self reloadData];
    }
}

// A bunch of delegation forwarding methods ...
- (void)tableViewColumnDidMove:(NSNotification *)notification
{
    if ([dataSource respondsToSelector:@selector(tableViewColumnDidMove:)])
    {
        [dataSource tableViewColumnDidMove:notification];
    }
}

- (void)tableViewColumnDidResize:(NSNotification *)notification
{
    if ([dataSource respondsToSelector:@selector(tableViewColumnDidResize:)])
    {
        [dataSource tableViewColumnDidResize:notification];
    }
}

- (void)tableViewSelectionDidChange:(NSNotification *)notification
{
    if ([dataSource respondsToSelector:@selector(tableViewSelectionDidChange:)])
    {
        [dataSource tableViewSelectionDidChange:notification];
    }
}

- (void)tableView:(NSTableView *)tableView willDisplayCell:(id)cell forTableColumn:(NSTableColumn *)tableColumn row:(int)row
{
    if ([dataSource respondsToSelector:@selector(tableView:willDisplayCell:forTableColumn:row:)])
    {
        [dataSource tableView:(NSTableView *)self willDisplayCell:cell forTableColumn:tableColumn row:row];
    }
}

- (BOOL)tableView:(NSTableView *)tableView shouldEditTableColumn:(NSTableColumn *)tableColumn row:(int)row
{
    if ([dataSource respondsToSelector:@selector(tableView:shouldEditTableColumn:row:)])
    {
        return [dataSource tableView:(NSTableView *)self shouldEditTableColumn:tableColumn row:row];
    }
    else return YES;
}

- (BOOL)selectionShouldChangeInTableView:(NSTableView *)aTableView
{
    if ([dataSource respondsToSelector:@selector(selectionShouldChangeInTableView:)])
    {
        return [dataSource selectionShouldChangeInTableView:aTableView];
    }
    else return YES;
}

- (BOOL)tableView:(NSTableView *)tableView shouldSelectRow:(int)row
{
    if ([dataSource respondsToSelector:@selector(tableView:shouldSelectRow:)])
    {
        return [dataSource tableView:(NSTableView *)self shouldSelectRow:row];
    }
    else return YES;
}

- (BOOL)tableView:(NSTableView *)aTableView shouldSelectTableColumn:(NSTableColumn *)tableColumn
{
    if ([dataSource respondsToSelector:@selector(tableView:shouldSelectTableColumn:)])
    {
        return [dataSource tableView:(NSTableView *)self shouldSelectTableColumn:tableColumn];
    }
    else return NO;
}

- (int)selectedRow
{
    int selectedSortedRow = [tableView selectedRow];
    if (selectedSortedRow < 0) return selectedSortedRow;
    return [self originalRow:selectedSortedRow];
}

- (void)selectRow:(int)row byExtendingSelection:(BOOL)extend
{
    int new_row, count = [rows count];
    id rowObject;

    // Row may have been added or deleted, so check
    if ([dataSource numberOfRowsInTableView:tableView] != count)
    {
        [self fetchData];
        count = [rows count];
    }

    for (new_row = 0; new_row < count; new_row++)
    {
        rowObject = [rows objectAtIndex:new_row];
        if ([rowObject originalRow] == row)
        {
            [tableView selectRow:new_row byExtendingSelection:extend];
            break;
        }
    }
}

/*
 * To finish this filter properly one should implement these methods too
 *
- (void)deselectRow:(int)row;
- (BOOL)isRowSelected:(int)row;
- (void)scrollRowToVisible:(int)row;
- (int)editedRow;
- (int)clickedRow;
 *
 */

// Anything we don't recognise, we send on to the table view, as that is
// what we are pretending to be, as far as the dataSource object is concerned.
// In terms of faking being the tableView delegate, we explicitly forward on
// all the relevant methods.
// N.B. You can't do this in Java (yet) ;-)
- (void)forwardInvocation:(NSInvocation *)invocation
{
    [invocation invokeWithTarget:tableView];
}

// We have to reimplement this as the runtime otherwise cannot find out
// about all our forwarded methods, and would assume we can't do them.
- (NSMethodSignature *)methodSignatureForSelector:(SEL)sel
{
    NSMethodSignature *sig = [super methodSignatureForSelector:sel];

    if (nil == sig)
    {
        // must not respond to the selector, ask our tableView
        sig = [tableView methodSignatureForSelector:sel];
    }
    return sig;
}

@end
