/* Logger.m
 *
 * You may freely copy, distribute and reuse the code in this example.
 * Apple Computer, Inc. disclaims any warranty of any kind, expressed or implied, as to
 * its fitness for any particular use.
 */

#import "Logger.h"

@implementation Logger

static NSString *logNotif = @"Log";

// Convenience method for logging messages
+ (void)log:(NSString *)msg
{
    // notification created retains msg ...
    [[NSNotificationCenter defaultCenter]
          postNotificationName:logNotif
                        object:msg
                      userInfo:nil];
}

- (id)init
{
    self = [super init];

    [self setIsEnabled:YES];
    [self setIsTimeStamping:NO];
    [self setIsAutoScrolling:YES];
    [self setAppendWithNewline:YES];

    // An example of assertions ...
    NSAssert([NSBundle loadNibNamed: @"Logger" owner:self],
             @"Logger nib not found");

    // auto save frame stuff
    [panel setFrameAutosaveName:@"Logger"];
    [panel setFrameUsingName:[panel frameAutosaveName]];

    // Hide on deactivate ? Design decision: putting it here is easier
    // to see than buried in the nib if using indexing or grep ...
    [panel setHidesOnDeactivate: NO];

    // Test if the panel is a window (for the find panel)
    if ([panel isKindOfClass:[NSPanel class]])
    {
        [panel setBecomesKeyOnlyIfNeeded: YES];
    }

    return self;
}

- (void)dealloc
{
    // This is probably never called but for completeness ...
    [self setIsEnabled:NO];
    [super dealloc];
}

- (void)awakeFromNib
{
    [enabledCheck setState:[self isEnabled]];
    [timeStampingCheck setState:[self isTimeStamping]];
    [autoScrollingCheck setState:[self isAutoScrolling]];
    [textView setRichText: NO];
    font = [[textView font] retain];
}

// TextView handling methods:

- (NSRange)textViewEnd
{
    NSRange range;

    // Note: to obtain the text view length we must convert it into a
    // string first
    range.location = [[textView string] length];
    range.length = 0;
    return range;	// Ok to return automatic variable as return by value
}

- (void)textViewScrollToEnd
{
    // scroll the TextView
    [textView scrollRangeToVisible:[self textViewEnd]];
}

- (void)textViewAppend:(NSString *)string
{
    if ([self appendWithNewline])
    {
        string = [string stringByAppendingString:@"\n"];
    }
    [textView replaceCharactersInRange:[self textViewEnd] withString:string];
}

- (void)clear
{
    [textView setString:@""];
}

- (BOOL)textViewLoadFromFile:(NSString *)path
{
    NSString *extension = [path pathExtension];

    [textView setFont: font];
    
    // rtfd
    if ([extension isEqual: @"rtfd"])
    {
        [textView setImportsGraphics: YES];
        return [textView readRTFDFromFile:path];
    }
    else if ([extension isEqual: @"rtf"])	// RTF
    {
        NSData *data;
        NSRange allText = NSMakeRange(0, [[textView string] length]);

        [textView setRichText: YES];
        [textView setImportsGraphics: NO];
        data = [NSData dataWithContentsOfFile:path];
        if (data)
            [textView replaceCharactersInRange:allText withRTF:data];
    }
    else
    {
        // ASCII
        NSString *string;
        [textView setRichText: NO];
        if (string = [[NSString alloc] initWithContentsOfFile:path])
        {
            [textView setString: string];
            [string release];
            return YES;
        }
    }
    return NO;
}

- (BOOL)textViewSaveToFile:(NSString *)path
{
    NSString *extension = [path pathExtension];

    // rtfd
    if ([extension isEqual: @"rtfd"])
    {
        return [textView writeRTFDToFile:path atomically:NO];
    }
    else if ([extension isEqual: @"rtf"])
    {
        NSData *data;
        NSRange allText = NSMakeRange(0, [[textView string] length]);
        data = [textView RTFFromRange:allText];
        if (data)
        {
            return [data writeToFile:path atomically:YES];
        }
    }
    else
    {
        // ASCII
        return [[textView string] writeToFile:path atomically: YES];
    }
    return NO;
}

// Logger Interface
- (void)timeStamp
{
    // append to TextView
    [self textViewAppend: [[NSCalendarDate calendarDate] description]];
}

- (NSString *)title
{
    return [panel title];
}

- (void)setTitle:(NSString *)string
{
    [panel setTitle:string];
}

- (void)logNotification:(NSNotification *)notification
{
    [self log:[notification object]];
}

- (void)log:(NSString *)string
{
    if ([self isEnabled])
    {
        if ([self isTimeStamping]) [self timeStamp];

        // append to TextView
        [self textViewAppend:string];

        if ([self isAutoScrolling]) [self textViewScrollToEnd];
    }
}

- (void)setIsEnabled:(BOOL)state
{
    // Ensure we only add ourselves as observer the one time
    // so that we are definitely removed when we go away ...
    if (isEnabled && !state)
    {
        [[NSNotificationCenter defaultCenter] removeObserver:self name:logNotif object:nil];
    }
    else if (!isEnabled && state)
    {
        [[NSNotificationCenter defaultCenter]
               addObserver:self
                  selector:@selector(logNotification:)
                      name:logNotif
                    object:nil];
    }
    isEnabled = state;
}

- (BOOL)isEnabled
{
    return isEnabled;
}

- (void)setIsTimeStamping:(BOOL)state
{
    isTimeStamping = state;
}

- (BOOL)isTimeStamping
{
    return isTimeStamping;
}

- (void)setIsAutoScrolling:(BOOL)state
{
    if (isAutoScrolling = state) [self textViewScrollToEnd];
}

- (BOOL)isAutoScrolling
{
    return isAutoScrolling;
}

- (void)setAppendWithNewline:(BOOL)state
{
    appendWithNewline = state;
}

- (BOOL)appendWithNewline
{
    return appendWithNewline;
}

// GUI access to logger features: target/action
- (void)show:(id)sender
{
    [panel orderFront:self];
}

- (void)clear:(id)sender
{
    [self clear];
}

- (void)save:(id)sender
{
    NSSavePanel *savePanel = [NSSavePanel savePanel];

    if ([textView importsGraphics])
    {
        [savePanel setRequiredFileType:@"rtfd"];
    }
    else if ([textView isRichText])
    {
        [savePanel setRequiredFileType:@"rtf"];
    }

    if ([savePanel runModal] == NSOKButton)
    {
        [self textViewSaveToFile:[savePanel filename]];
    }
}

- (void)load:(id)sender
{
    NSOpenPanel *openPanel = [NSOpenPanel openPanel];

    if ([openPanel runModal] == NSOKButton)
    {
        [self textViewLoadFromFile:[openPanel filename]];
    }
}

- (void)changeIsEnabled:(id)sender
{
    [self setIsEnabled:[sender state]];
}

- (void)changeIsTimeStamping:(id)sender
{
    [self setIsTimeStamping:[sender state]];
}

- (void)changeIsAutoScrolling:(id)sender
{
    [self setIsAutoScrolling:[sender state]];
}

@end
