#import "AlphaFormatter.h"

@implementation AlphaFormatter

// Basic "Alpha" Formatter
// - allows only letters, i.e. no white space, numbers etc.
// - does not allow empty value
// - configurable attribute: capitalize or not
//
// Two different implementations:
// 1) using just NSString and NSCharacterSet
// 2) using NSScanner
// Uncomment the next line for the NSScanner implementation
// #define SCANNER

- (void)_setCharacterSet
{
#ifdef SCANNER
    // don't need to retain common character sets
    set = [NSCharacterSet letterCharacterSet];
#else
    // Build the inverse of the allowed character set
    set = [[[NSCharacterSet letterCharacterSet] invertedSet] retain];
#endif

}

- (id)init
{
    [super init];
    capitalize = NO;
    [self _setCharacterSet];
    return self;
}

- (void)setCapitalize:(BOOL)flag
{
    capitalize = flag;
}

- (BOOL)capitalize
{
    return capitalize;
}

- (NSString *)stringForObjectValue:(id)anObject
{
    if ([anObject isKindOfClass:[NSString class]])
    {
	if (capitalize)
		return [anObject capitalizedString];
	else
        	return anObject;
    }
    return nil;
}

- (BOOL)getObjectValue:(id *)anObject
             forString:(NSString *)string
      errorDescription:(NSString **)error
{
    // don't allow an empty string
    if (![string length]) {
        *error = @"Zero length string";
	return NO;
    }

    // if caller actually wants the objectValue
    if (anObject)
    {
        if (capitalize)
            *anObject = [string capitalizedString];
        else
            *anObject = [[string copy] autorelease];
    }
    return YES;
}

- (BOOL)isPartialStringValid:(NSString *)partialString
            newEditingString:(NSString **)newString
            errorDescription:(NSString **)error
{    
    // Zero-length strings are always ok
    if ([partialString length] == 0) return YES;

    // We have to check the entire (partial) string since the user's
    // edit event can replace more than one character at any point in the
    // string

#ifdef SCANNER
    {
        NSScanner *scanner;
        NSString *tmp;

        scanner = [NSScanner scannerWithString:partialString];
        [scanner scanCharactersFromSet:set intoString:&tmp];
        if ([tmp length] != [partialString length])
        {
            *error = @"bad string";
            return NO;
        }
    }
#else
    {
        NSRange range;
        range = [partialString rangeOfCharacterFromSet: set];
        if (range.length > 0)
        {
            *error = @"bad string";
            return NO;
        }
    }
#endif
    
    // To capitalize now, while the user types
    {
        *newString = [partialString capitalizedString];
        // the partialString is OK but we want caller to use newString
        return NO;
    } 

    return YES;
}

- (void)encodeWithCoder:(NSCoder *)coder
{
    [super encodeWithCoder:coder];
    [coder encodeValuesOfObjCTypes:"c",&capitalize];
    // no need to encode the character set; regenerate in initWithCoder:
}

- (id)initWithCoder:(NSCoder *)coder
{
    self = [super initWithCoder:coder];
    [coder decodeValuesOfObjCTypes:"c",&capitalize];
    // need to build the character set
    [self _setCharacterSet];
    return self;
}

@end