

#import "ExpenseDataSource.h"
#import "Expense.h"
#import "AppController.h" // for logger
#import "Logger.h"
#import "AlphaFormatter.h"

@implementation ExpenseDataSource

// Logic to determine when the delete button should be enabled
- (void)_updateDeleteButton
{
    [deleteButton setEnabled: [array count] && [tableView numberOfSelectedRows]];
}

- (void)_update
{
    [self _updateDeleteButton];
    [tableView noteNumberOfRowsChanged];
    [tableView reloadData];
}

// private helper method: generate an array of test expense objects
- (void)_generateTestData
{
       int i;
       for (i = 0; i < 10; i++)
       {
               id expense = [[Expense alloc] init];
               [expense setCategory:[NSString stringWithFormat:@"Cat%d", i]];
               [expense setAmount:[NSNumber numberWithInt:(10 - i)]];
               [expense setDate:
                       [NSCalendarDate dateWithTimeIntervalSinceNow: 60*60*24*i]];
               [expense setIsBillable: i % 2];
               [array addObject:expense];
               [expense release];
       }
}

- (id)init
{
    [super init];
    [self setArray: [NSMutableArray array]];
    return self;
}

- (void)dealloc
{
    [array release];
    [tableView setDelegate: nil];
    [tableView setDataSource: nil];
    [super dealloc];
}

- (void)awakeFromNib
{
    {
    NSArray *columns = [tableView tableColumns];
    id column = [columns objectAtIndex:1];
    // The data cell is the one we want, not the header cell
    id cell = [column dataCell];
    id formatter = [[AlphaFormatter alloc] init];
    [formatter setCapitalize: YES];
    [cell setFormatter:formatter];
    [formatter release]; // I'm finished with it ...
    }
    
    [self _update];
}

- (void)setArray:(NSMutableArray *)theArray
{
    [theArray retain];
    [array release];
    array = theArray;
    if (!array)
	array = [[NSMutableArray alloc] init];
    [self _update];
}

- (NSMutableArray *)array
{
    return array;
}

- (id)selectedObject
{
    int index = [tableView selectedRow];
    id object = nil;
    if (index != -1)
        object = [array objectAtIndex: index];
    return object;
}

- (SEL)accessorFromIdentifier:(NSString *)identifier
{
        return NSSelectorFromString(identifier);
}

- (SEL)setAccessorFromIdentifier:(NSString *)identifier
{
        NSString *string = [@"set" stringByAppendingString:
                                [[identifier capitalizedString]
                                        stringByAppendingString:@":"]];
        return NSSelectorFromString(string);
}

- (int)numberOfRowsInTableView:(NSTableView *)tableView
{
    return [array count];
}

- (id)tableView:(NSTableView *)tableView objectValueForTableColumn:(NSTableColumn *)tableColumn row:(int)row
{
    id item = [array objectAtIndex:row];
    NSString *identifier = [tableColumn identifier];
    SEL sel = [self accessorFromIdentifier:identifier];

    if ([item respondsToSelector:sel])
        return [item performSelector:sel];
    else
        return @"";
}

- (void)tableView:(NSTableView *)tv setObjectValue:(id)object forTableColumn:(NSTableColumn *)tableColumn row:(int)row
{
    id item = [array objectAtIndex:row];
    NSString *identifier = [tableColumn identifier];
    SEL sel = [self setAccessorFromIdentifier:identifier];

    if ([item respondsToSelector:sel])
    {
        [item performSelector:sel withObject:object];
        // Make a note that we changed the document
        [[tv window] setDocumentEdited: YES];
    }
}

- (void)add:(id)sender
{
    int index;
    id new  = [[Expense alloc] init];
    [array addObject:new];
    [new release];
    [[tableView window] setDocumentEdited: YES];
    // select new row
    [self _update]; // first update table view
    index = [array indexOfObject: new];
    [tableView selectRow:index byExtendingSelection:NO];
    [tableView scrollRowToVisible:index];
    [[[NSApp delegate] logger] log: @"add"];
}

- (void)delete:(id)sender
{
    int row = [tableView selectedRow];
    if (row != -1)
    {
        [array removeObjectAtIndex:row];
        [[tableView window] setDocumentEdited: YES];
        // automatically update the selection
        [self _update];  // first update table view
        if ([array count])
        {
            if (row != 0) row--;
            [tableView selectRow:row byExtendingSelection:NO];
        }
    }
    [Logger log: @"delete"];
}

- (void)tableViewSelectionDidChange:(NSNotification *)notification
{
    [self _updateDeleteButton];
}

@end
