/*
 BreakView.m, view to implement the "BreakApp" game.
 Author: Ali Ozer
 Written for 0.8 October 88. 
 Modified for 0.9 March 89.
 Modified for 1.0 July 89.
 Removed use of Bitmap and threw away some classes May 90.
 Final 2.0 fixes/enhancements Sept 90.
 3.0 update March 92.
 Sound-related changes for 3.1 March 92.
 Modified for 4.0 Jan 96
 Modified for Rhapsody June 97

 BreakView implements an interactive custom view that allows the user
 to play "BreakApp," a game similar to a popular arcade classic.

 BreakView's main control methods are based on the target-action
 paradigm; thus you can include BreakView in an Interface-Builder based
 application. Please refer to BreakView.h for a list of "public" methods
 that you should provide links to in Interface Builder.
 
 Copyright (c) 1988-1997 by Apple Computer, Inc., all rights reserved.

 You may incorporate this sample code into your applications without
 restriction, though the sample code has been provided "AS IS" and the
 responsibility for its operation is 100% yours.  However, what you are
 not permitted to do is to redistribute the source as "DSC Sample Code"
 after having made changes. If you're going to re-distribute the source,
 we require that you make it clear in the source that the code was
 descended from Apple Sample Code, but that you've made changes.
*/

#import <AppKit/AppKit.h>
#import "BreakView.h"
#import "SoundEffect.h"

// Max absolute x and y velocities of the ball, in base coordinates per msec.

#define MAXXV     ((level > 6) ? 0.3 : 0.2) 
#define MAXYV     (0.4)

// Maximum amount of time that is allowed to pass between two calls to the
// step method. If the time is greater than MAXTIMEDIFFERENCE, then this
// value is used instead. MAXTIMEDIFFERENCE should be no greater
// than the time it takes for the ball to go the height of a tile
// or the height of the ball + height of paddle. The units
// are in milliseconds.

#define MAXTIMEDIFFERENCE (TILEHEIGHT * 0.8 / MAXYV)
#define MINTIMEDIFFERENCE 1

// Max revolution speed of the ball; this is the maximum
// number of radians it will turn per millisecond when rotating...

#define APPROX_PI (3.14)
#define MAXREVOLUTIONSPEED (APPROX_PI / 250.0)	// Max is 2 revs/sec

// The following values are the default sizes for the various pieces. 

#define RADIUS		8.0 			// Ball radius
#define PADDLEWIDTH	(TILEWIDTH * 1.8)	// Paddle width
#define PADDLEHEIGHT	(TILEHEIGHT * 0.6)	// Paddle height
#define BALLWIDTH	(RADIUS * 2.0)		// Ball width
#define BALLHEIGHT	(RADIUS * 2.0)		// Ball height

// SHADOWOFFSET defines the amount the shadow is offset from the piece. 
#define SHADOWOFFSET 3.0

// Number of lives per game
#define LIVES     5		

// Number of loops through the game after all tiles die
#define STOPGAMEAT (-10)

// Bonus at the end of a level
#define LEVELBONUS 50

// Starting locations...			
#define PADDLEX ((gameSize.width - paddleSize.width) / 2.0)
#define PADDLEY 1.0
#define BALLX ((gameSize.width - ballSize.width) / 2.0)
#define BALLY (paddleY + paddleSize.height)

// Accelaration & score values of the different tile types.
static const float tileAccs[NUMTILETYPES] = {1.0, 1.3};
static const int tileScores[NUMTILETYPES] = {5, 25};

#define NOTILE -1

#define RANDINT(n) (rand() % ((n)+1))		// Random integer 0..n
#define ONEIN(n)   ((rand() % (n)) == 0)	// TRUE one in n times 
#define INITRAND   srand(time(0))		// Randomizer

#define gameSize  [self bounds].size

// Restrict a value to the range -max .. max.

inline float restrictValue(float val, float max) {
    if (val > max) return max;
    else if (val < -max) return -max;
    else return val;
}

// Convert x-location to left/right pan for playing sounds

@implementation BreakView

- (id)initWithFrame:(NSRect)frm {
    [super initWithFrame:frm];
    
    [self allocateGState];	// For faster lock/unlockFocus
    
    [(ball = [[NSImage allocWithZone:[self zone]] init]) setScalesWhenResized:NO];
    [ball addRepresentation:[[[NSCustomImageRep alloc] initWithDrawSelector:@selector(drawBall:) delegate:self] autorelease]];

    [(paddle = [[NSImage allocWithZone:[self zone]] init]) setScalesWhenResized:NO];
    [paddle addRepresentation:[[[NSCustomImageRep alloc] initWithDrawSelector:@selector(drawPaddle:) delegate:self] autorelease]];

    [(tile[0] = [[NSImage allocWithZone:[self zone]] init]) setScalesWhenResized:NO];
    [(tile[1] = [[NSImage allocWithZone:[self zone]] init]) setScalesWhenResized:NO];
    [tile[0] addRepresentation:[[[NSCustomImageRep alloc] initWithDrawSelector:@selector(drawNormalTile:) delegate:self] autorelease]];
    [tile[1] addRepresentation:[[[NSCustomImageRep alloc] initWithDrawSelector:@selector(drawToughTile:) delegate:self] autorelease]];

    wallSound = [[SoundEffect allocWithZone:[self zone]] initWithSoundResource:@"Wall"];
    tileSound = [[SoundEffect allocWithZone:[self zone]] initWithSoundResource:@"Tile"];
    missSound = [[SoundEffect allocWithZone:[self zone]] initWithSoundResource:@"Miss"];
    paddleSound = [[SoundEffect allocWithZone:[self zone]] initWithSoundResource:@"Paddle"];

    [self setBackgroundFile:[[NSUserDefaults standardUserDefaults] stringForKey:@"BackGround"] andRemember:NO];
	    
    [self resizePieces];
    
    [self getHighScore];
    
    demoMode = NO;
    
    INITRAND;
    
    return self;
}

// free simply gets rid of everything we created for BreakView, including
// the instance of BreakView itself. This is how nice objects clean up.

- (void)dealloc {
    int cnt;

    if (gameRunning) {
	[timer invalidate];
        [timer release];
    }
    for (cnt = 0; cnt < NUMTILETYPES; cnt++) {
	[tile[cnt] release];
    }

    [ball release];    
    [paddle release];
    [backGround release];
    [wallSound release];
    [tileSound release];
    [missSound release];
    [paddleSound release];

    [super dealloc];
}

// resizePieces calculates the new sizes of all the pieces after the game is
// started or the playing field (the BreakView) is resized.

- (void)resizePieces {
    int cnt;
    float xRatio = gameSize.width / GAMEWIDTH;
    float yRatio = gameSize.height / GAMEHEIGHT;

    [backGround setSize:gameSize];

    tileSize.width = floor(xRatio * TILEWIDTH);
    tileSize.height = floor(yRatio * TILEHEIGHT);
    for (cnt = 0; cnt < NUMTILETYPES; cnt++) {
	[tile[cnt] setSize:tileSize];
    }
    leftMargin = floor((gameSize.width - (tileSize.width + INTERTILE) * NUMTILESX) / 2.0 + 1.0);

    paddleSize.width = floor(xRatio * PADDLEWIDTH);
    paddleSize.height = floor(yRatio * PADDLEHEIGHT);
    [paddle setSize:paddleSize];

    ballSize.width = floor(xRatio * BALLWIDTH);
    ballSize.height = floor(yRatio * BALLHEIGHT);
    [ball setSize:ballSize];
}

// The following allows BreakView to grab the mousedown event that activates
// the window. By default, the View's acceptsFirstMouse returns NO.

- (BOOL)acceptsFirstMouse:(NSEvent *)theEvent {
    return YES;
}

// This methods allows changing the file used to paint the background of the
// playing field. Set fileName to NULL to revert to the default. Set
// remember to YES if you wish the write the value out in the defaults.

- (void)setBackgroundFile:(NSString *)fileName andRemember:(BOOL)remember {
    [backGround release];
    if (fileName) {
	backGround = [[NSImage allocWithZone:[self zone]] initByReferencingFile:fileName];
	[backGround setScalesWhenResized:YES];
	[backGround setSize:gameSize];
	if (remember) {
	    [[NSUserDefaults standardUserDefaults] setObject:fileName forKey:@"BackGround"];
	}
    } else {
	backGround = [[NSImage allocWithZone:[self zone]] initWithSize:gameSize];
	[backGround addRepresentation:[[[NSCustomImageRep alloc] initWithDrawSelector:@selector(drawDefaultBackground:) delegate:self] autorelease]];
	[backGround setScalesWhenResized:NO];
	if (remember) {
	    [[NSUserDefaults standardUserDefaults] removeObjectForKey:@"BackGround"];
	}
    }
    [backGround setBackgroundColor:[NSColor whiteColor]];
    [self setNeedsDisplay:YES];
}

// The following two methods allow changing the background image from
// menu items or buttons.

- (void)changeBackground:sender {
    NSOpenPanel *openPanel = [NSOpenPanel openPanel];
    if ([openPanel runModalForTypes:[NSImage imageFileTypes]]) {
	[self setBackgroundFile:[openPanel filename] andRemember:YES];
        [self setNeedsDisplay:YES];
    }
}

- (void)revertBackground:sender {
    [self setBackgroundFile:NULL andRemember:YES];
    [self setNeedsDisplay:YES];
}

// getHighScore reads the previous high score from the user's defaults file.
// If no such default is found, then the high score is set to zero.

- (void)getHighScore {
    highScore = [[NSUserDefaults standardUserDefaults] integerForKey:@"HighScore"];
}

// setHighScore should be called when the user score for a game is above 
// the current high score. setHighScore sets the high score and 
// writes it out the defaults file so that it can be remembered for eternity.

- (void)setHighScore:(int)hScore {
    highScore = hScore;
    [[NSUserDefaults standardUserDefaults] setInteger:highScore forKey:@"HighScore"];
}

- (int)score {
    return score;
}

- (int)level {
    return level;
}

- (int)lives {
    return lives;
}

// gotoFirstLevel: sets everything up for a new game.

- (void)gotoFirstLevel:sender {
    score = 0;
    level = 0;
    lives = LIVES;
    [self gotoNextLevel:sender];
}

// gotoNextLevel: sets everything up for the next level of the game; the level
// count is incremented and the pieces are set up on the field. The ball and
// the paddle are also brought to the starting locations.
//
// This routine can of course be made infinitely more complicated in
// determining where the tiles go. Left as an exercise to the reader. 8-)

- (void)gotoNextLevel:sender {
    int xcnt, ycnt, yFrom, yTo, xFrom, xTo;
    
    // We are at the next level... Stop the game and increment the level.
    
    [self stop:sender];
    
    level++;
    
    // Now place the tiles. Here's where we could do some fancy tile layout,
    // depending on the game level. yFrom, yTo, xFrom, and xTo define the "box"
    // in which we will lay the tiles out. These values are inclusive.
    
    switch (level % 6) {
	case 0: yTo = NUMTILESY-2; break;
	case 4: yTo = NUMTILESY-4; break;
	case 5: yTo = 2 * (NUMTILESY / 3); break;
	default: yTo = 3 * (NUMTILESY / 4); break;
    }
    
    xFrom = 0; xTo = NUMTILESX-1; yFrom = yTo - 3;
    
    switch (level % 10) {
	case 1: yFrom++; break;   
	case 2: yFrom--; xFrom++; xTo--; break;
	case 4: xFrom += 2; xTo -= 2; break;
	case 6: yFrom = MIN(yFrom, NUMTILESY / 4); xFrom++; xTo--; break;
	case 7: xTo -= 3; break;
	case 8: yFrom -= 2; xFrom += 2; xTo -= 2; break;
	case 9: yFrom = MIN(yFrom, NUMTILESY / 4);
		yTo = MAX(yTo, yFrom+4);
		break;
	case 0: yFrom = MIN(yFrom, NUMTILESY / 5);
		xFrom += (NUMTILESX / 2);
		break;
	default: break;
    }    
    
    // The area in the playing field where we place tiles is at least 3 tiles 
    // high and at least NUMTILESX-4 tiles wide.
    
    // Empty out the whole playing field.
    for (xcnt = 0; xcnt < NUMTILESX; xcnt++) {
	for (ycnt = 0; ycnt < NUMTILESY; ycnt++) {
	    tiles[xcnt][ycnt] = NOTILE;
	}
    }

    // Fill up the tile area with wimpy tiles
    for (ycnt = yFrom; ycnt <= yTo; ycnt++) {
	for (xcnt = xFrom; xcnt <= xTo; xcnt++) {
	    tiles[xcnt][ycnt] = 0;
	}
    }

    // Erase or change some of the tiles, depending on the level.
    // Assumption is that we have at least 3 rows of tiles, yFrom..yTo.
    
    switch (level % 7) {
	case 2: // clear two rows in the middle	      
	    for (xcnt = xFrom; xcnt <= xTo; xcnt++) {
		tiles[xcnt][yFrom+1] = tiles[xcnt][yTo-1] = NOTILE;
	    }
	    break;
	case 3: // randomly clear out some tiles
	    for (xcnt = 0; xcnt < 5; xcnt++) {
		tiles[xFrom+RANDINT(xTo-xFrom)][yFrom+1+RANDINT(yTo-yFrom-2)] = NOTILE;
	    }
	    break;
	case 4: // clear middle columns
	    for (xcnt = xFrom +  2; xcnt <= xTo - 2; xcnt++) {
		for (ycnt = yFrom; ycnt <= yTo; ycnt++) {
		    tiles[xcnt][ycnt] = NOTILE;
		}
	    }
	    break;
	case 6: // clear out the insides
	    for (ycnt = yFrom+1; ycnt < yTo; ycnt++) {
		for (xcnt = xFrom+1; xcnt < xTo; xcnt++) {
		    tiles[xcnt][ycnt] = NOTILE;
		}
	    }
	    break;
	default:
	    break;    
    }
    
    // Drop in some tough tiles in all rows except the first one
    for (xcnt = 0; xcnt < 5; xcnt++) {    	
	tiles[xFrom+RANDINT(xTo-xFrom)][yFrom+1+RANDINT(yTo-yFrom-1)] = 1;
    }
    
    // Compute the number of tiles we actually ended up putting down...
    numTilesLeft = 0;
    for (ycnt = yFrom; ycnt <= yTo; ycnt++) {
	for (xcnt = xFrom; xcnt <= xTo; xcnt++) {
	    if (tiles[xcnt][ycnt] != NOTILE) numTilesLeft++;
	}
    }

    // Of course you might think there are too many braces in the above code,
    // where probably none would've sufficed. Too many braces never hurt, & it
    // will save you from some bozo bug some day. So use them! They're cheap!
    
    [self resetBallAndPaddle];
    
    [levelView setIntValue:level];
    [scoreView setIntValue:score];
    [livesView setIntValue:lives];
    [hscoreView setIntValue:highScore];
    [statusView setStringValue:NSLocalizedString(@"Game Ready", "Message indicating the game is ready to run")];
    
    killerBall = ((level % 12) == 6);	// Every 12 turns (starting on 6th), the ball loses its ability to bounce off tiles
    niceBall = (level % 5 == 0);	// Every 5 turns, make the ball bounce towards the paddle

    // For backgrounds from files, this is a bit wasteful...

    [backGround recache];
    
    [self setNeedsDisplay:YES];			// Display the new arrangement
    
    if (demoMode) {
	[self go:sender];	// If in demo mode, start rolling
    }
}      

// setDemoMode: allows the user to put the game in a demo mode.
// In the demo mode, the paddle constantly follows the ball.

- (void)setDemoMode:sender {
    if (demoMode = ([sender state] == 0 ? NO : YES)) {
	[self go:sender];
    } else {
	[self stop:sender];
    }
}

// This method should be called when a new level or game is started or the
// player misses the ball. It resets the ball & paddle locations back to
// default.

- (void)resetBallAndPaddle {
    paddleX = PADDLEX;
    paddleY = PADDLEY;
    ballX = BALLX;
    ballY = BALLY;

    ballXVel = 0.0;
    ballYVel = 0.0;

    // The ball shouldn't start out rotating...
    revolutionsLeft = 0;	
}
        
// The directBallAt: initializes the velocity vector of the ball so that
// the ball will go from its current location to the specified destination  
// point. The speed of the ball is determined by the current level. If ballYVel
// is already set, then only the x velocity & y direction is changed.

- (void)directBallAt:(NSPoint)dest {
    float desiredYVel = dest.y - (ballY + ballSize.height / 2.0);
    float desiredXVel = dest.x - (ballX + ballSize.width / 2.0);

    // Transform back to original game coords (velocity values are measured
    // in these).

    desiredYVel /= (gameSize.height / GAMEHEIGHT);
    desiredXVel /= (gameSize.width / GAMEWIDTH);

    if (fabs(desiredYVel) < 1.0) {
	desiredYVel = desiredYVel < 0.0 ? -1.0 : 1.0;
    }
    if (ballYVel == 0.0) {
	// Come up with a value between 60 and 100% of MAXYV.
	ballYVel = restrictValue(((RANDINT(level * 8) + 60.0) / 100.0) * MAXYV, MAXYV);
    }
    ballYVel = fabs(ballYVel) * (desiredYVel < 0.0 ? -1.0 : 1.0);
    ballXVel = restrictValue(ballYVel * (desiredXVel / desiredYVel) ,MAXXV);
}    

// The stop method will pause a running game. The go method will start it up
// again. They can be assigned to buttons or other appkit objects through IB.

- (void)go:sender {
    if (lives && !gameRunning) {
	// If the ball velocity wasn't initialized, start it rolling
	// towards the mouse location...
	if (ballXVel == 0.0 && ballYVel == 0.0) {
	    NSPoint mouseLoc = [self convertPoint:[[self window] mouseLocationOutsideOfEventStream] fromView:nil];
	    [self directBallAt:mouseLoc];
	    ballYVel = fabs(ballYVel);
	}
	gameRunning = YES;
	timer = [[NSTimer scheduledTimerWithTimeInterval:0.03 target:self selector:@selector(step:) userInfo:self repeats:YES] retain];
	[statusView setStringValue:NSLocalizedString(@"Running", "Message indicating that the game is running")];
    }
}

- (void)stop:(id)sender {
    if (gameRunning) {
	gameRunning = NO;
	[timer invalidate]; [timer release];;
	[statusView setStringValue:NSLocalizedString(@"Paused", "Message indicating that the game is paused")]; 
    }
}

- (void)setFrameSize:(NSSize)_newSize {
    NSSize oldSize = [self bounds].size;
    
    [super setFrameSize:_newSize];
    
    ballX = (ballX * _newSize.width / oldSize.width);
    ballY = (ballY * _newSize.height / oldSize.height);
    paddleX = (paddleX * _newSize.width / oldSize.width);
    paddleY = (paddleY * _newSize.height / oldSize.height);
    
    [self resizePieces];
    
    [self setNeedsDisplay:YES];
}

// A mousedown effectively allows pausing and unpausing the game by
// alternately calling one of the above two functions (stop/go).

- (void)mouseDown:(NSEvent *)event {
    if (gameRunning) {
	[self stop:self]; 
    } else if (lives) {
	[self go:self];   
    }
}

// The following few methods draw the pieces.

- (void)drawBall:imageRep {
    NSSize circleSize = NSMakeSize(BALLWIDTH - SHADOWOFFSET, BALLHEIGHT - SHADOWOFFSET);
    
    // Drawing the ball is tricky; if the view is resized, we want to draw an ellipse
    // (not circle), so we have to scale. When we scale, we want the distance 
    // between the shadow and the ball to remain the same, so we scale and draw 
    // the shadow and the ball separately...

    PSgsave();
    // First draw the shadow under the ball.
    PStranslate(SHADOWOFFSET, 0);
    PSscale ((ballSize.width - SHADOWOFFSET) / circleSize.width, (ballSize.height - SHADOWOFFSET) / circleSize.height);
    [[[NSColor blackColor] colorWithAlphaComponent:0.333] set];
    PSarc (RADIUS-SHADOWOFFSET, RADIUS-SHADOWOFFSET, RADIUS-SHADOWOFFSET, 0.0, 360.0);
    PSfill ();
    PSgrestore();

    PSgsave();
    
    // Then the ball.
    PStranslate(0, SHADOWOFFSET);
    PSscale ((ballSize.width - SHADOWOFFSET) / circleSize.width, (ballSize.height - SHADOWOFFSET) / circleSize.height);
    [[NSColor colorWithCalibratedWhite:0.875 alpha:1.0] set];
    PSarc (RADIUS-SHADOWOFFSET, RADIUS-SHADOWOFFSET, RADIUS-SHADOWOFFSET, 0.0, 360.0);
    PSfill ();

    // And the lighter & darker spots on the ball...
    [[NSColor whiteColor] set];
    PSarc (RADIUS-SHADOWOFFSET, RADIUS-SHADOWOFFSET, RADIUS-SHADOWOFFSET-1.0, 100.0, 170.0);
    PSstroke ();
    [[NSColor colorWithCalibratedWhite:0.666 alpha:1.0] set];
    PSarc (RADIUS-SHADOWOFFSET, RADIUS-SHADOWOFFSET, RADIUS-SHADOWOFFSET-1.0, 280.0, 350.0);
    PSstroke ();

    PSgrestore();
}

// Function to draw a shadow under the given rectangle.

static void drawRectangularShadowUnder (NSRect rect, float offset) {
    NSRect shadeRect = NSOffsetRect(rect, offset, -offset);
    [[[NSColor blackColor] colorWithAlphaComponent:0.333] set];
    NSRectFill(shadeRect);
}

// Draws a 3d box; if inset == YES, the box is inset

static void draw3DBox (NSRect rect, BOOL inset) {
    static NSColor *colors[9] = {nil};
    NSRect rects[9];

    // Convenience macros...
    #define RX rect.origin.x
    #define RY rect.origin.y
    #define RW rect.size.width
    #define RH rect.size.height
    #define RMX (RX + RW - 1)
    #define RMY (RY + RH - 1)

    if (colors[0] == nil) {	// Fill these in first time through
        colors[0] = colors[1] = [[NSColor colorWithCalibratedWhite:0.666 alpha:1.0] retain];
        colors[2] = colors[3] = [[NSColor colorWithCalibratedWhite:1.0 alpha:1.0] retain];
        colors[4] = colors[5] = [[NSColor colorWithCalibratedWhite:0.0 alpha:1.0] retain];
        colors[6] = colors[7] = [[NSColor colorWithCalibratedWhite:0.5 alpha:1.0] retain];
        colors[8] = [[NSColor colorWithCalibratedWhite:0.875 alpha:1.0] retain];
    }
    rects[inset ? 6 : 0] = NSMakeRect(RX, RY + 1, 1, RH - 2);
    rects[inset ? 7 : 1] = NSMakeRect(RX + 1, RMY, RW - 2, 1);
    rects[inset ? 4 : 2] = NSMakeRect(RX + 1, RY + 1, 1, RH - 2);
    rects[inset ? 5 : 3] = NSMakeRect(RX + 1, RMY - 1, RW - 2, 1);
    rects[inset ? 2 : 4] = NSMakeRect(RMX, RY + 1, 1, RH - 2);
    rects[inset ? 3 : 5] = NSMakeRect(RX + 1, RY, RW - 2, 1);
    rects[inset ? 0 : 6] = NSMakeRect(RMX - 1, RY + 1, 1, RH - 3);
    rects[inset ? 1 : 7] = NSMakeRect(RX + 2, RY + 1, RW - 3, 1);
    rects[8] = NSMakeRect(RX + 2, RY + 2, RW - 4, RH - 4);
    NSRectFillListWithColors(rects, colors, 9);
}

- (void)drawPaddle:imageRep {
    NSRect pieceRect = {{0.0, SHADOWOFFSET}, {(paddleSize.width-SHADOWOFFSET)-1, (paddleSize.height-SHADOWOFFSET)-1}};
    drawRectangularShadowUnder (pieceRect, SHADOWOFFSET);
    draw3DBox(pieceRect, NO);
}

- (void)drawToughTile:imageRep {
    NSRect pieceRect = {{0.0, SHADOWOFFSET}, {(tileSize.width-SHADOWOFFSET)-1, (tileSize.height-SHADOWOFFSET)-1}};
    drawRectangularShadowUnder (pieceRect, SHADOWOFFSET);
    draw3DBox(pieceRect, NO);
    pieceRect = NSInsetRect(pieceRect , 3.0 , 3.0);
    draw3DBox(pieceRect, YES);
}

- (void)drawNormalTile:imageRep {
    NSRect pieceRect = {{0.0, SHADOWOFFSET}, {(tileSize.width-SHADOWOFFSET)-1, (tileSize.height-SHADOWOFFSET)-1}};
    drawRectangularShadowUnder (pieceRect, SHADOWOFFSET);
    draw3DBox(pieceRect, NO);
}

#define NUMYBOXES 10
#define NUMXBOXES 6

// This method draws the default background. The default background consists of
// NUMXBOXES x NUMYBOXES raised boxes. Each box is drawn as four triangles to
// provide a raised effect. Boxes near the top left corner are lighter in color
// than the ones near the bottom right.

- (void)drawDefaultBackground:imageRep {
#define NOTFOUND ((id)-1)
    static NSColorList *colorList = nil;	// Static because it's shared
    NSSize boxSize = {gameSize.width / NUMXBOXES, gameSize.height / NUMYBOXES};
    int xCnt, yCnt;
    NSColor * color;

    // The first time we're here, we load and cache the color list. If we don't find
    // it, we remember that fact so that we don't go through the search again.
    if (colorList == nil) {
	NSString *colorListPath;
	if (colorListPath = [[NSBundle mainBundle] pathForResource:@"BreakApp" ofType:@"clr"]) {
	    colorList = [[NSColorList allocWithZone:NSDefaultMallocZone()] initWithName:@"" fromFile:colorListPath];
	}
	if (!colorList) {
	    NSLog(@"Can't find color list for backgrounds.");
	    colorList = NOTFOUND;
	}
    }

    // Now get the color. If the color list wasn't found, we use some default random color.
    // Note that because colors in different color spaces might look different on
    // different devices (although they might look identical on screen), it's
    // important to always use colors from the same color space when creating
    // a wash. Below we assure that our colors always start off in HSB color space.

    if (colorList != NOTFOUND) {
	color = [colorList colorWithKey:[[colorList allKeys] objectAtIndex:(level % [[colorList allKeys] count])]];
	color = [NSColor colorWithCalibratedHue:[[color colorUsingColorSpaceName:NSCalibratedRGBColorSpace] hueComponent] saturation:[[color colorUsingColorSpaceName:NSCalibratedRGBColorSpace] saturationComponent] brightness:1.0 alpha:1.0];
    } else {
	color = [NSColor colorWithCalibratedHue:(level % 8) / 7.0 saturation:0.8 brightness:1.0 alpha:1.0];
    }

    for (yCnt = 0; yCnt < NUMYBOXES; yCnt++) {
	for (xCnt = 0; xCnt < NUMXBOXES; xCnt++) {
	    float h, s, b, a;
            // Determine brightness (each box has a different brightness); we also use the color from the previous time
	    [[color colorUsingColorSpaceName:NSCalibratedRGBColorSpace] getHue:&h saturation:&s brightness:NULL alpha:&a];
	    b = 0.4 + (yCnt + (4 - xCnt)) * (0.2 / (NUMYBOXES + NUMXBOXES));
	    // The bottom triangle
	    [[NSColor colorWithCalibratedHue:h saturation:s brightness:b alpha:a] set];
	    PSmoveto (xCnt * boxSize.width + boxSize.width / 2.0, yCnt * boxSize.height + boxSize.height / 2.0);
	    PSrlineto (-boxSize.width / 2.0, -boxSize.height / 2.0);
	    PSrlineto (boxSize.width, 0);
	    PSfill ();
	    // The right triangle
            [color = [NSColor colorWithCalibratedHue:h saturation:s brightness:b * 1.2 alpha:a] set];
	    PSmoveto (xCnt * boxSize.width + boxSize.width / 2.0, yCnt * boxSize.height + boxSize.height / 2.0);
	    PSrlineto (boxSize.width / 2.0, boxSize.height / 2.0);
	    PSrlineto (0, -boxSize.height);
	    PSfill ();
	    // The left triangle
            [color = [NSColor colorWithCalibratedHue:h saturation:s brightness:b * 1.2 * 1.2 alpha:a] set];
	    PSmoveto (xCnt * boxSize.width + boxSize.width / 2.0, yCnt * boxSize.height + boxSize.height / 2.0);
	    PSrlineto (-boxSize.width / 2.0, boxSize.height / 2.0);
	    PSrlineto (0, -boxSize.height);
	    PSfill ();
	    // The right triangle
            [color = [NSColor colorWithCalibratedHue:h saturation:s brightness:b * 1.2 * 1.2 * 1.2 alpha:a] set];
	    PSmoveto (xCnt * boxSize.width + boxSize.width / 2.0, yCnt * boxSize.height + boxSize.height / 2.0);
	    PSrlineto (boxSize.width / 2.0, boxSize.height / 2.0);
	    PSrlineto (-boxSize.width, 0.0);
	    PSfill ();
	}
    }
}

// The following methods show or erase the ball and the paddle from the field.

- (void)showBall {
    NSRect tmpRect = {{floor(ballX), floor(ballY)}, {ballSize.width, ballSize.height}};
    [ball compositeToPoint:tmpRect.origin operation:NSCompositeSourceOver];
}

- (void)showPaddle {
    NSRect tmpRect = {{floor(paddleX), floor(paddleY)}, {paddleSize.width, paddleSize.height}};
    [paddle compositeToPoint:tmpRect.origin operation:NSCompositeSourceOver];
}

- (void)eraseBall {
    NSRect rect = {{ballX, ballY}, {ballSize.width, ballSize.height}};
    [self drawBackground:rect];
}

- (void)erasePaddle {
    NSRect rect = {{paddleX, paddleY}, paddleSize};
    [self drawBackground:rect];
}

// drawBackground: just draws the specified piece of the background by
// compositing from the background image.

- (void)drawBackground:(NSRect)rect {
    rect.origin.x = floor(NSMinX(rect));
    rect.origin.y = floor(NSMinY(rect));
    if ([[NSDPSContext currentContext] isDrawingToScreen]) {
	[[NSColor whiteColor] set];
        NSRectFill (rect);
    }
    [backGround compositeToPoint:rect.origin fromRect:rect operation:NSCompositeSourceOver];
}

// drawRect:, a method every decent View should have, redraws the game
// in its current state. This allows us to print the game very easily as well.

- (void)drawRect:(NSRect)rect {
    int xcnt, ycnt;

    [self drawBackground:rect];

    for (xcnt = 0; xcnt < NUMTILESX; xcnt++) { 
	for (ycnt = 0; ycnt < NUMTILESY; ycnt++) {
	    if (tiles[xcnt][ycnt] != NOTILE) {
		NSPoint tileLoc = {floor(leftMargin + (tileSize.width + INTERTILE) * xcnt), floor((tileSize.height + INTERTILE) * ycnt)};
		[tile[tiles[xcnt][ycnt]] compositeToPoint:tileLoc operation:NSCompositeSourceOver];
	    }
	}
    }

    if (lives) {
	[self showBall];
	[self showPaddle];
    }
}

// incrementGameScore: adds the value of the argument to the score if the game
// is not in demo mode.

- (void)incrementGameScore:(int)scoreIncrement {
    if (demoMode == NO) {
	score += scoreIncrement;
    }
}

// hitTileAt:: checks to see if there's a tile at tile location x, y;
// if so, it is considered hit by the ball and cleared. hitTileTile:: also
// updates the score and the ball velocity. hitTileAt:: returns YES if there
// was a tile, NO otherwise.

-(BOOL) hitTileAt:(int)x :(int)y {
    NSRect rect = {{floor(leftMargin + (tileSize.width + INTERTILE) * x), 
		    floor((tileSize.height + INTERTILE) * y)},
		   {tileSize.width, tileSize.height}};

    if (x < NUMTILESX && y < NUMTILESY && x >= 0 && y >= 0 && 
	(tiles[x][y] != NOTILE)) {
	[self incrementGameScore:tileScores[tiles[x][y]]];
	ballYVel = restrictValue(ballYVel * tileAccs[tiles[x][y]], MAXYV);
	[self drawBackground:rect];
	tiles[x][y] = NOTILE;
	numTilesLeft--;
	return YES;
    } else {
	return NO;
    }
}


// The paddleHit method is called whenever the ball hits the paddle.
// This method bounces the ball back at an angle depending on what part of
//  the paddle was hit.

- paddleHit {
    float whereHit = ((ballX + RADIUS) - paddleX) / paddleSize.width;

    ballYVel = -ballYVel;
    ballY = paddleSize.height;

    [self playSound:paddleSound atXLoc:paddleX];

    // Alter the x-velocity and make sure it is in the valid range.
    // If the ball hits the edges of the paddle, bounce it back at some angle.
    
    if (whereHit < 0.1) {
	ballXVel = - MAXXV;
    } else if (whereHit > 0.9) {
	ballXVel = MAXXV;
    } else {
	// Now whereHit is in the range 0.1 .. 0.9, with 0.5 indicating middle
	// of the paddle.  Convert to a number in the range 0.2 to 1, with 0.2
	// indicating the middle and 1 either end.
	whereHit = (fabs(whereHit - 0.5) + 0.1) * 2.0;  
	ballXVel = ((ballXVel > 0.0) ? 1.0 : -1.0) * MAXXV * whereHit;
    }

    return self;
}

// If upon launch we discover that there's no sound, then we fail
// silently. Note that although the SoundEffect class has the ability
// to enable/disable sounds, because we might have multiple
// BreakViews each with its own sound state, we keep a local state in
// addition to the one in SoundEffect.

// Note that although this is an outlet method, we do not actually
// have an outlet (instance variable) named soundStateFrom; we just
// want to take a look at the initial value of the button and see
// if sound needs to be turned on...

- (void)setSoundStateFrom:sender {
    if ([sender state]) {
	[SoundEffect setSoundEnabled:YES];
	if (!(soundEnabled = [SoundEffect soundEnabled])) {
	    [sender setState:NO];	// Silently fail
	}
    } else {
	soundEnabled = NO;
    }
}

// If user tries to enable sound once the game is launched, and it
// fails, then we do tell him/her about it.

- (void)setSoundMode:sender {
    BOOL desiredState = [sender state];
    [self setSoundStateFrom:sender];
    if (desiredState && !soundEnabled) {
	NSRunAlertPanel(NSLocalizedString(@"No Sound", "Title of alert indicating sounds aren't available"), NSLocalizedString(@"Can't play sounds.", "Contents of alert panel"), NSLocalizedString(@"Bummer", "Acceptance that sounds can't be played"), nil, nil);
    }
}

- (void)playSound:sound atXLoc:(float)xLoc {
    if (soundEnabled) {
	[sound play:1.0 pan:restrictValue((xLoc / gameSize.width - 0.5) * 2.0, 1.0)];
    }
}

// Alters the given velocity vector so that it is 
// rotated by the indicated amount. We restrict both the resulting x and v
// velocity values to the maximum of their max possible values...

- (void)rotate:(float *)xVel :(float *)yVel by:(float)radians {
    float newAngle = atan2 (*yVel, *xVel) + radians;
    float velocity = hypot (*xVel, *yVel); 

    *yVel = restrictValue(velocity * sin(newAngle), MAX(MAXYV, MAXXV));
    *xVel = restrictValue(velocity * cos(newAngle), MAX(MAXYV, MAXXV));
}

// The step method implements one step through the main game loop.
// The distance traveled by the ball is adjusted by the time between frames.

- (void)step:(NSTimer *)timer
 {
    NSPoint mouseLoc;
    float newX;
    NSTimeInterval timeNow = [NSDate timeIntervalSinceReferenceDate];
    unsigned int timeDelta = MIN(MAX((timeNow - lastFrameTime) * 1000, MINTIMEDIFFERENCE), MAXTIMEDIFFERENCE);
    lastFrameTime = timeNow;
   
    [self lockFocus];
    
    [self eraseBall];
    
    // If the ball is rotating, rotate it by the indicated amount.

    if (revolutionsLeft > 0.0) {
	float revsThisTime = revolutionSpeed * timeDelta;
	[self rotate:&ballXVel :&ballYVel by:revsThisTime];
	revolutionsLeft -= revsThisTime;
	if (revolutionsLeft <= 0.0 && (fabs(ballYVel) < MAXYV * 0.6)) {
	    // Done rotating; make sure we have a good y-velocity
	    ballYVel = MAXYV * 0.8 * (ballYVel < 0.0 ? -1 : 1);
	    ballXVel = restrictValue(ballXVel,MAXXV);
	}
    } else if (ONEIN(1000 + (level < 8 ? (8 - level) * 250 : 0)) && (ballY > gameSize.height * 0.6)) {
	// If we're not rotating, we go into rotating mode one out of 
	// 1500 or more steps, provided that the ball is not too close to
	// the paddle at the time.
        revolutionsLeft = APPROX_PI * (2 + RANDINT(5)); // 1 to 3.5 full turns
	revolutionSpeed = MAXREVOLUTIONSPEED * (RANDINT(8) + 2.0) / 10.0;
    } 

    // Update the ball location

    ballX += ballXVel * timeDelta * gameSize.width / GAMEWIDTH; 
    ballY += ballYVel * timeDelta * gameSize.height / GAMEHEIGHT;


    if (gameRunning) {

	if (ballX < 0.0) { // Hit on the left wall
	    ballX = 0.0;
	    ballXVel = -ballXVel; 
	    [self playSound:wallSound atXLoc:ballX];
	} else if (ballX > gameSize.width - ballSize.width) { // Right wall
	    ballX = gameSize.width - ballSize.width;
	    ballXVel = -ballXVel; 
	    [self playSound:wallSound atXLoc:ballX];
	}

	if (ballY > gameSize.height - ballSize.height) { // Top wall
	    ballY = gameSize.height - ballSize.height;
	    ballYVel = -ballYVel;
	    if (niceBall && !ONEIN(5) && !demoMode) {
		NSPoint mid = {paddleX + paddleSize.width / 2.0, paddleY};
		[self directBallAt:mid];
	    } else if (ONEIN(10)) {
		ballXVel = MAXXV-(RANDINT((int)(MAXXV*20))/10.0);
	    }
	    [self playSound:wallSound atXLoc:ballX];
	}

	// Now checking for collisions with tiles... 

	{
	    int y1 = (int)(floor(ballY /
				    (tileSize.height + INTERTILE)));
	    int x1 = (int)(floor((ballX - leftMargin) /
				    (tileSize.width + INTERTILE)));
	    int y2 = (int)(floor((ballY + ballSize.height) / 
				    (tileSize.height + INTERTILE)));
	    int x2 = (int)(floor((ballX + ballSize.width - leftMargin) / 
				    (tileSize.width + INTERTILE)));
    
	    if ([self hitTileAt:x1 :y1] | [self hitTileAt:x2 :y1] |
		[self hitTileAt:x1 :y2] | [self hitTileAt:x2 :y2]) {
		[self playSound:tileSound atXLoc:ballX];
		if (!killerBall) {
		    ballYVel = -ballYVel;
		}
		[scoreView setIntValue:score];
		[[self window] flushWindow];
	    }
	}
    }

    // Get the mouse location and convert from window to the view coords.
    // If in demo, mode, make the paddle track the ball. Endless fun.

    if (demoMode) {
	mouseLoc.x = ballX + ballSize.width / 2.0;
    } else {
	mouseLoc = [[self window] mouseLocationOutsideOfEventStream];
	mouseLoc = [self convertPoint:mouseLoc fromView:nil];
    }

    newX = MAX(MIN(mouseLoc.x - paddleSize.width/2,
		    gameSize.width - paddleSize.width), 0);

    if (ballY >= paddleY + paddleSize.height) {

	// Ball is above the paddle; redraw it and the paddle and continue
	// We flush twice as the ball and the paddle are not too close 
	// together

	[self showBall];
	[[self window] flushWindow];
	[self erasePaddle];
	paddleX = newX;
	[self showPaddle];
	[[self window] flushWindow];

    } else if (ballY + ballSize.height > 0) {
	
	// Ball is past the paddle but not totally gone...

	[self erasePaddle];
	paddleX = newX;

	// Check to see if the user managed to catch the ball after all

	if ((ballY > paddleY - ballSize.height / 2.0) &&
	    (ballX <= paddleX + paddleSize.width) &&
	    (ballX + ballSize.width > paddleX)) {
	    [self paddleHit];
	}

	// The ball and the paddle are close, so one flushWindow is fine.

	[self showBall];
	[self showPaddle];
	[[self window] flushWindow];

    } else {

	// Too late; the ball is out of sight...

	[self erasePaddle];
	[self stop:self];
	[self playSound:missSound atXLoc:0.0];

	if (--lives == 0) {
	    if (score > highScore) [self setHighScore:score];
	    [statusView setStringValue:NSLocalizedString(@"Game Over", "Message indicating that the game is over...")];
	} else {
	    [self resetBallAndPaddle]; 
	    [self showBall];
	    [self showPaddle]; 
	    [statusView setStringValue:NSLocalizedString(@"Game Ready", "Message indicating the game is ready to run")];
	}
	[[self window] flushWindow];

	[livesView setIntValue:lives];
	
    }

    // numTilesLeft <= 0 indicates that we've blown away every tile. But,
    // to make the game more exciting, we start decrementing numTilesLeft, 
    // by one everytime through this loop, until it reaches the value 
    // STOPGAMEAT. This makes the ball move a bit more after all the tiles 
    // are gone. But, if gameRunning is NO, then it means we probably just
    // missed the ball, in which case we should go ahead and jump to the 
    // next level.
    
    if ((numTilesLeft <= 0) && 
	((lives && !gameRunning) || (--numTilesLeft == STOPGAMEAT))) {
	[self incrementGameScore:LEVELBONUS];
	[self gotoNextLevel:self];
    }

    PSWait ();	// Synchronize postscript for smoother animation

    [self unlockFocus];
}

@end
