/*
 * PuzzleActions.m
 * 5/1/99 Mike Trent
 *
 * A series of classes that make up our Puzzle state engine. YumAction is the
 * base class, which defines some data and accessors. Subclasses define
 * specific implementations.
 *
 * The PuzzleView will call oneStep, expired, and newAction methods.
 * All other methods are private to the classes in this file. oneStep will
 * actually perform drawing for our view, changing state as necessary. expired
 * tells the view to aquire a new action. newAction both autoreleases  the
 * current action and returns a new action in one shot. Each action keeps
 * records a list of objects it may return in newAction; this is what builds
 * the edges in our state graph.
 *
 * This is very similar to the state engine in the Yum module. The major
 * difference between Puzzle's state engine and Yum's state engine is the
 * Puzzle engine actually does the necessary work: drawing, maintaining state,
 * etc. The Yum engine merely fiddles with state values, but relies on the
 * YumView class to actually do the drawing. Yum has its reasons for doing this,
 * but these same reasons do not apply to Puzzle.
 */
#import "PuzzleActions.h"

static float gSpeed = 1.0;
static BOOL gSound = NO;

float randBetween(float a, float b);

@interface PuzzleLeftAction : PuzzleAction
@end

@interface PuzzleRightAction : PuzzleAction
@end

@interface PuzzleUpAction : PuzzleAction
@end

@interface PuzzleDownAction : PuzzleAction
@end

@interface PuzzleAction (PrivateStuff)
- (void)_resetTimer;
- (NSRect)_rectForTile:(NSPoint)tile;
- (int)_indexForTile:(NSPoint)tile;
- (void)setEndInterval:(NSTimeInterval)interval;
- (void)_printDebugMessage:(NSString*)message;
- (NSCell *)_cellForString:(id)str color:(NSColor *)color font:(NSFont *)font wrap:(BOOL)wrap;
@end

// You'll regret it if you turn sound on ... 
void tink(void)
{
    static NSSound *sound = nil;

    if (!gSound) return;
    if (!sound) sound = [NSSound soundNamed:@"Pop"];

    [sound play];
    while ([sound isPlaying]) [[NSRunLoop currentRunLoop] runUntilDate:[NSDate date]];

}

//
// PuzzleAction Base Class
//

@implementation PuzzleAction

+ (float)speed
{
    return gSpeed;
}

+ (void)setSpeed:(float)speed
{
    gSpeed = speed;
}

+ (float)playSounds
{
    return gSound;
}

+ (void)setPlaySounds:(BOOL)sound
{
    gSound = sound;
}

- (id)init
{
    self = [super init];

    if (self) {
        _startDate = [[NSDate date] retain];
        _endInterval = [PuzzleAction speed]; // all actions take the same time
    }

    return self;
}

- (id)initWithState:(State*)state
{
    self = [self init];

    if (self) {
        _state = *state;
        if (_state.screen) [_state.screen retain];
        if (_state.screenRects) [_state.screenRects retain];
    }

    return self;
}

- (void)dealloc
{
    [_startDate release];
    [_actions release];
    if (_state.screen) [_state.screen release];
    if (_state.screenRects) [_state.screenRects release];
    
    [super dealloc];
}

- (BOOL)expired
{
    return _expired;
}

- (void)setEndInterval:(NSTimeInterval)interval
{
    _endInterval = interval;
}

- (void)oneStep
{
    // note if we have expired
    NSTimeInterval now = [[NSDate date] timeIntervalSinceDate:_startDate];
    if (now >= _endInterval) _expired = YES;
}

- (id)newAction
{
    if (_actions) {
        int index, count = [_actions count];

        index = count * ((float) random() / (float) RAND_MAX);
        if (index == count) index = 0;

        if (!NSEqualPoints(_state.liveTile, _state.deadTile)) {
            
            [_state.screenRects swapIndex:[self _indexForTile:_state.liveTile]
                                withIndex:[self _indexForTile:_state.deadTile]];
            
            _state.deadTile = _state.liveTile;
        } 

        [self autorelease];
        return [(PuzzleAction*)[[_actions objectAtIndex:index] alloc] initWithState:&_state];
    }

    return self;
}

//
// PuzzleAction Private Methods
//

- (void)_resetTimer
{
    [_startDate release];
    _startDate = [[NSDate date] retain];
    _expired = NO;
}

- (NSRect)_rectForTile:(NSPoint)tile
{
    NSRect r;
    r.origin = NSMakePoint(tile.x * (_state.tileSize.width + 1),
                           tile.y * (_state.tileSize.height + 1));
    r.size = _state.tileSize;
    return r;
}

- (int)_indexForTile:(NSPoint)tile
{
    return tile.y * _state.tileArea.width + tile.x;
}
   
- (void)_printDebugMessage:(NSString*)message
{
    NSCell *cell;
    NSRect bigRect = [[NSView focusView] bounds];
    NSRect textRect;

    cell = [self _cellForString:message color:[NSColor whiteColor] font:[NSFont systemFontOfSize:12.0] wrap:NO];
    textRect.size = [cell cellSizeForBounds:bigRect];
    textRect.size.width = 300;
    textRect.origin = NSMakePoint(0, bigRect.size.height-textRect.size.height);
    [[NSColor blackColor] set];
    NSRectFill(textRect);
    [cell drawWithFrame:textRect inView:[NSView focusView]];
}

- (NSCell *)_cellForString:(id)str color:(NSColor *)color font:(NSFont *)font wrap:(BOOL)wrap
{
    // str can be NSString or NSAttributedString.
    static NSTextFieldCell *cell = nil;
    if (!cell) {
        cell = [[NSTextFieldCell allocWithZone:[self zone]] initTextCell:@"Hello"];
        [cell setBordered:NO];
        [cell setBezeled:NO];
        [cell setDrawsBackground:NO];
    }
    if ([str isKindOfClass:[NSAttributedString class]]) {
        [cell setAttributedStringValue:str];
    } else {
        [cell setWraps:wrap];
        [cell setStringValue:str];
        [cell setFont:font];
        [cell setTextColor:color];
    }

    return cell;
}

@end

//
// Puzzle Action Subclasses
//

@implementation PuzzleStartAction

// PuzzleStartAction initializes state, elects the block to remove, and
// blackens out that tile.

- (id)initWithImage:(NSImage*)image tileWidth:(float)width
{
    int i,j;

    self = [super init];
    if (self) {
        [self makeTileList:image tileSize:width];
        _state.screen = [image retain];
        
        // pick our dead tile ... this could be cleaner ...
        j = _state.tileArea.width;
        i = j * ((float) random() / (float) RAND_MAX);
        if (i == j) i = 0;
        _state.deadTile.x = i;
        j = _state.tileArea.height;
        i = j * ((float) random() / (float) RAND_MAX);
        if (i == j) i = 0;
        _state.deadTile.y = i;

        // our dead tile is also our live tile for this action
        _state.liveTile = _state.deadTile;

        // set our action list
        _actions = [[NSMutableArray array] retain];
        if (_state.liveTile.x > 0)
            [_actions addObject:[PuzzleLeftAction class]];
        if (_state.liveTile.x < _state.tileArea.width - 1)
            [_actions addObject:[PuzzleRightAction class]];
        if (_state.liveTile.y < _state.tileArea.height - 1)
            [_actions addObject:[PuzzleUpAction class]];
        if (_state.liveTile.y > 0)
            [_actions addObject:[PuzzleDownAction class]];
    }

    return self;
}

- (void)oneStep
{
    NSRect r;
    float xinset, yinset;
    
    [super oneStep];

    if (!_once) {
        // This is the first time through. We should draw the whole grid.
        int i,j;
        
        _once = YES;

        // black out the screen, just in case.
        r = NSMakeRect(0,0,_state.tileArea.width * (_state.tileSize.width+1), _state.tileArea.height * (_state.tileSize.height+1));
        [[NSColor blackColor] set];
        NSRectFill(r);

        // plot our squares.
        for (j = 0; j < _state.tileArea.height; j += 1) {
            for (i = 0; i < _state.tileArea.width; i += 1) {
                r.origin = NSMakePoint(i * (_state.tileSize.width + 1),j * (_state.tileSize.height + 1));
                r.size = _state.tileSize;
                [_state.screen compositeToPoint:r.origin fromRect:r operation:NSCompositeCopy];
            }
        }

        // now, reset our timer....
        if (_endInterval > 0.0) [self _resetTimer];
    }
    r = [self _rectForTile:_state.deadTile];

    // if we have not yet expired, inset our dead rect. This gives us a nice
    // expanding-blackness effect.
    if (!_expired) {
        NSTimeInterval now = [[NSDate date] timeIntervalSinceDate:_startDate];
        xinset = _state.tileSize.width / 2 * (1 - now / _endInterval);
        yinset = _state.tileSize.height / 2 * (1 - now / _endInterval);
        r = NSInsetRect(r, xinset, yinset);
    }

    [[NSColor blackColor] set];
    NSRectFill(r);
}

// cut the screen into pieces.
// this function originally created an NSArray of NSImages; hence the title
// "makeTileList". The function now creates an ANSI C array of NSRects via
// PuzzleRectArray. This is *much* more efficient.
- (void)makeTileList:(NSImage*)screen tileSize:(float)tileSize
{
    int x,y,n = 0;
    NSRect bounds;

    bounds.origin = NSMakePoint(0,0);
    bounds.size = [screen size];

    // when calculating tiles wide, add grid width to tile width.
    _state.tileSize = NSMakeSize(tileSize,tileSize);
    _state.tileArea =
        NSMakeSize(ceil(bounds.size.width / (_state.tileSize.width + 1)),
                   ceil(bounds.size.height / (_state.tileSize.height + 1)));

    // allocate our rect list
    _state.screenRects = [[PuzzleRectArray alloc] initWithCount:_state.tileArea.width * _state.tileArea.height];

    // loop across our screen and set rects. Our "grid" is a side effect
    // of this looping process (it's the "+ 1"s here). 
    for (y = 0; y < bounds.size.height; y += _state.tileSize.height + 1) {
        for (x = 0; x < bounds.size.width; x += _state.tileSize.width + 1) {
            [_state.screenRects setRect:NSMakeRect(x,y,_state.tileSize.width, _state.tileSize.height) atIndex:n++];
        }
    }
}

@end

@implementation PuzzleLeftAction

// PuzzleLeftAction moves the tile to the left of the space into the space.
// Mnemonic: space moves left.

- (id)initWithState:(State*)state
{
    self = [super initWithState:state];

    if (self) {
        _state.liveTile = _state.deadTile;
        _state.liveTile.x--;

        // set our action list
        _actions = [[NSMutableArray array] retain];
        if (_state.liveTile.x > 0)
            [_actions addObject:[PuzzleLeftAction class]];
        if (_state.liveTile.y < _state.tileArea.height - 1)
            [_actions addObject:[PuzzleUpAction class]];
        if (_state.liveTile.y > 0)
            [_actions addObject:[PuzzleDownAction class]];
    }

    return self;
}

- (void)dealloc
{
    if (_expired && gSound) tink();
    [super dealloc];
}

- (void)oneStep
{
    NSRect lr, dr, ur;

    [super oneStep];

    lr = [self _rectForTile:_state.liveTile];
    dr = [self _rectForTile:_state.deadTile];
    ur = NSUnionRect(lr,dr);

    [[NSColor blackColor] set];
    NSRectFill(ur);

    if (_expired) {
        lr = dr;
    } else {
        NSTimeInterval now = [[NSDate date] timeIntervalSinceDate:_startDate];
        lr.origin.x += _state.tileSize.width * (now / _endInterval);
    }

    [_state.screen compositeToPoint:lr.origin fromRect:[_state.screenRects rectAtIndex:[self _indexForTile:_state.liveTile]] operation:NSCompositeCopy];
}

@end

@implementation PuzzleRightAction

// PuzzleRightAction moves the tile to the right of the space into the space.
// Mnemonic: space moves right.

- (id)initWithState:(State*)state
{
    self = [super initWithState:state];

    if (self) {
        _state.liveTile = _state.deadTile;
        _state.liveTile.x++;

        // set our action list
        _actions = [[NSMutableArray array] retain];
        if (_state.liveTile.x < _state.tileArea.width - 1)
            [_actions addObject:[PuzzleRightAction class]];
        if (_state.liveTile.y < _state.tileArea.height - 1)
            [_actions addObject:[PuzzleUpAction class]];
        if (_state.liveTile.y > 0)
            [_actions addObject:[PuzzleDownAction class]];
    }

    return self;
}

- (void)dealloc
{
    if (_expired && gSound) tink();
    [super dealloc];
}

- (void)oneStep
{
    NSRect lr, dr, ur;

    [super oneStep];

    lr = [self _rectForTile:_state.liveTile];
    dr = [self _rectForTile:_state.deadTile];
    ur = NSUnionRect(lr,dr);

    [[NSColor blackColor] set];
    NSRectFill(ur);

    if (_expired) {
        lr = dr;
    } else {
        NSTimeInterval now = [[NSDate date] timeIntervalSinceDate:_startDate];
        lr.origin.x -= _state.tileSize.width * (now / _endInterval);
    }

    [_state.screen compositeToPoint:lr.origin fromRect:[_state.screenRects rectAtIndex:[self _indexForTile:_state.liveTile]] operation:NSCompositeCopy];
}

@end

@implementation PuzzleUpAction

// PuzzleUpAction moves the tile above the space into the space.
// Mnemonic: space moves up.

- (id)initWithState:(State*)state
{
    self = [super initWithState:state];

    if (self) {
        _state.liveTile = _state.deadTile;
        _state.liveTile.y++;

        // set our action list
        _actions = [[NSMutableArray array] retain];
        if (_state.liveTile.x > 0)
            [_actions addObject:[PuzzleLeftAction class]];
        if (_state.liveTile.x < _state.tileArea.width - 1)
            [_actions addObject:[PuzzleRightAction class]];
        if (_state.liveTile.y < _state.tileArea.height - 1)
            [_actions addObject:[PuzzleUpAction class]];
    }

    return self;
}

- (void)dealloc
{
    if (_expired && gSound) tink();
    [super dealloc];
}

- (void)oneStep
{
    NSRect lr, dr, ur;

    [super oneStep];

    lr = [self _rectForTile:_state.liveTile];
    dr = [self _rectForTile:_state.deadTile];
    ur = NSUnionRect(lr,dr);

    [[NSColor blackColor] set];
    NSRectFill(ur);

    if (_expired) {
        lr = dr;
    } else {
        NSTimeInterval now = [[NSDate date] timeIntervalSinceDate:_startDate];
        lr.origin.y -= _state.tileSize.height * (now / _endInterval);
    }

    [_state.screen compositeToPoint:lr.origin fromRect:[_state.screenRects rectAtIndex:[self _indexForTile:_state.liveTile]] operation:NSCompositeCopy];
}

@end

@implementation PuzzleDownAction

// PuzzleDownAction moves the tile below space into the space.
// Mnemonic: space moves down.

- (id)initWithState:(State*)state
{
    self = [super initWithState:state];

    if (self) {
        _state.liveTile = _state.deadTile;
        _state.liveTile.y--;

        // set our action list
        _actions = [[NSMutableArray array] retain];
        if (_state.liveTile.x > 0)
            [_actions addObject:[PuzzleLeftAction class]];
        if (_state.liveTile.x < _state.tileArea.width - 1)
            [_actions addObject:[PuzzleRightAction class]];
        if (_state.liveTile.y > 0)
            [_actions addObject:[PuzzleDownAction class]];
    }

    return self;
}

- (void)dealloc
{
    if (_expired && gSound) tink();
    [super dealloc];
}

- (void)oneStep
{
    NSRect lr, dr, ur;

    [super oneStep];

    lr = [self _rectForTile:_state.liveTile];
    dr = [self _rectForTile:_state.deadTile];
    ur = NSUnionRect(lr,dr);

    [[NSColor blackColor] set];
    NSRectFill(ur);

    if (_expired) {
        lr = dr;
    } else {
        NSTimeInterval now = [[NSDate date] timeIntervalSinceDate:_startDate];
        lr.origin.y += _state.tileSize.height * (now / _endInterval);
    }

    [_state.screen compositeToPoint:lr.origin fromRect:[_state.screenRects rectAtIndex:[self _indexForTile:_state.liveTile]] operation:NSCompositeCopy];
}

@end