/* io.c
 *
 * Copyright (c) 1996 Mike Gleason, NCEMRSoft.
 * All rights reserved.
 *
 * THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF NCEMRSOFT.
 * The copyright notice above does not evidence any actual or intended
 * publication of such source code.
 */

#include "syshdrs.h"

static int gGotBrokenData;
static int gUnused;

#ifdef HAVE_SIGSETJMP
static sigjmp_buf gBrokenDataJmp;
#else
static jmp_buf gBrokenDataJmp;
#endif	/* HAVE_SIGSETJMP */
static int gCanBrokenDataJmp;

static void
BrokenData(int signum)
{
	gGotBrokenData = signum;
	if (gCanBrokenDataJmp != 0) {
		gCanBrokenDataJmp = 0;
#ifdef HAVE_SIGSETJMP
		siglongjmp(gBrokenDataJmp, 1);
#else
		longjmp(gBrokenDataJmp, 1);
#endif	/* HAVE_SIGSETJMP */
	}
}	/* BrokenData */





void
FTPInitIOTimer(const FTPCIPtr cip)
{
	cip->bytesTransferred = (longest_int) 0;
	cip->expectedSize = kSizeUnknown;
	cip->mdtm = kModTimeUnknown;
	cip->rname = NULL;
	cip->lname = NULL;
	cip->kBytesPerSec = -1.0;
	cip->percentCompleted = -1.0;
	cip->sec = -1.0;
	cip->secLeft = -1.0;
	cip->nextProgressUpdate = 0;
	cip->stalled = 0;
	cip->useProgressMeter = 1;
	(void) gettimeofday(&cip->t0, NULL);
}	/* FTPInitIOTimer */




void
FTPStartIOTimer(const FTPCIPtr cip)
{
	(void) gettimeofday(&cip->t0, NULL);
	if (cip->progress != (FTPProgressMeterProc) 0)
		(*cip->progress)(cip, kPrInitMsg);
}	/* FTPStartIOTimer */




void
FTPUpdateIOTimer(const FTPCIPtr cip)
{
	double sec;
	struct timeval *t0, t1;
	time_t now;

	(void) time(&now);
	if (now < cip->nextProgressUpdate)
		return;
	now += 1;
	cip->nextProgressUpdate = now;

	(void) gettimeofday(&t1, NULL);
	t0 = &cip->t0;

	if (t0->tv_usec > t1.tv_usec) {
		t1.tv_usec += 1000000;
		t1.tv_sec--;
	}
	sec = ((double) (t1.tv_usec - t0->tv_usec) * 0.000001)
		+ (t1.tv_sec - t0->tv_sec);
	if (sec > 0.0) {
		cip->kBytesPerSec = ((double) cip->bytesTransferred) / (1024.0 * sec);
	} else {
		cip->kBytesPerSec = -1.0;
	}
	if (cip->expectedSize == kSizeUnknown) {
		cip->percentCompleted = -1.0;
		cip->secLeft = -1.0;
	} else if (cip->expectedSize <= 0) {
		cip->percentCompleted = 100.0;
		cip->secLeft = 0.0;
	} else {
		cip->percentCompleted = ((double) (100.0 * (cip->bytesTransferred + cip->startPoint))) / ((double) cip->expectedSize);
		if (cip->percentCompleted >= 100.0) {
			cip->percentCompleted = 100.0;
			cip->secLeft = 0.0;
		} else if (cip->percentCompleted <= 0.0) {
			cip->secLeft = 999.0;
		}
		if (cip->kBytesPerSec > 0.0) {
			cip->secLeft = ((cip->expectedSize - cip->bytesTransferred - cip->startPoint) / 1024.0) / cip->kBytesPerSec;
			if (cip->secLeft < 0.0)
				cip->secLeft = 0.0;
		}
	}
	cip->sec = sec;
	if ((cip->progress != (FTPProgressMeterProc) 0) && (cip->useProgressMeter != 0))
		(*cip->progress)(cip, kPrUpdateMsg);
}	/* FTPUpdateIOTimer */




void
FTPStopIOTimer(const FTPCIPtr cip)
{
	cip->nextProgressUpdate = 0;	/* force last update */
	FTPUpdateIOTimer(cip);
	if (cip->progress != (FTPProgressMeterProc) 0)
		(*cip->progress)(cip, kPrEndMsg);
}	/* FTPStopIOTimer */




/* This isn't too useful -- it mostly serves as an example so you can write
 * your own function to do what you need to do with the listing.
 */
int
FTPList(const FTPCIPtr cip, int outfd, int longMode, const char *lsflag)
{
	char *cmd;
	int result, nread;
	char secondaryBuf[512];
	char *secBufPtr, *secBufLimit;
	char line[128];

	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);

	cmd = (longMode != 0) ? "LIST" : "NLST";
	if ((lsflag == NULL) || (lsflag[0] == '\0')) {
		result = FTPStartDataCmd(cip, kNetReading, kTypeAscii, 0, "%s", cmd);
	} else {
		result = FTPStartDataCmd(cip, kNetReading, kTypeAscii, 0, "%s %s", cmd, lsflag);
	}
	if (result == 0) {
		/* This line sets the buffer pointer so that the first thing
		 * BufferGets will do is reset and fill the buffer using
		 * real I/O.
		 */
		secBufPtr = secondaryBuf + sizeof(secondaryBuf);
		secBufLimit = (char *) 0;

		while (1) {
			if (cip->xferTimeout > 0)
				(void) alarm(cip->xferTimeout);
			nread = BufferGets(line, sizeof(line), cip->dataSocket, secondaryBuf, &secBufPtr, &secBufLimit, sizeof(secondaryBuf));
			if (nread <= 0) {
				if (nread < 0)
					break;
			} else {
				cip->bytesTransferred += (longest_int) nread;
				(void) STRNCAT(line, "\n");
				(void) write(outfd, line, strlen(line));
			}
		}
		if (cip->xferTimeout > 0)
			(void) alarm(0);
		result = FTPEndDataCmd(cip, 1);
		if (result < 0) {
			result = kErrLISTFailed;
			cip->errNo = kErrLISTFailed;
		}
		result = kNoErr;
	} else if (result == kErrGeneric) {
		result = kErrLISTFailed;
		cip->errNo = kErrLISTFailed;
	}
	return (result);
}	/* FTPList */




int
FTPListToMemory2(const FTPCIPtr cip, const char *pattern, LineListPtr lines, const char *lsflags, int blanklines)
{
	volatile int result;
	int nread;
	char secondaryBuf[512];
	char *secBufPtr, *secBufLimit;
	char line[128];
	char lsflags1[128];
	const char *command = "NLST";
	const char *scp;
	char *dcp, *lim;
	volatile FTPSigProc osigpipe;
	volatile FTPCIPtr vcip;
	int sj;

	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);

	if ((lines == NULL) || (pattern == NULL) || (lsflags == NULL))
		return (kErrBadParameter);

	/* See if we should use LIST instead. */
	if (lsflags[0] == '-') {
		scp = lsflags + 1;
		dcp = lsflags1;
		lim = lsflags1 + sizeof(lsflags1) - 2;
		*dcp++ = '-';
		for (; *scp != '\0'; scp++) {
			if (*scp == 'l') {
				/* do not add the 'l' */
				command = "LIST";
			} else if (dcp < lim) {
				*dcp++ = *scp;
			}
		}
		*dcp = '\0';
		if (dcp == (lsflags1 + 1)) {
			lsflags1[0] = '\0';
		} else {
			*dcp++ = ' ';
			*dcp = '\0';
		}
	} else {
		(void) STRNCPY(lsflags1, lsflags);
	}

	InitLineList(lines);

	if ((lsflags1[0] != '\0') || (pattern[0] != '\0')) {
		result = FTPStartDataCmd(cip, kNetReading, kTypeAscii, 0, "%s %s%s", command, lsflags1, pattern);
	} else {
		result = FTPStartDataCmd(cip, kNetReading, kTypeAscii, 0, "%s", command);
	}

	vcip = cip;
	osigpipe = (volatile FTPSigProc) signal(SIGPIPE, BrokenData);

	gGotBrokenData = 0;
	gCanBrokenDataJmp = 0;

#ifdef HAVE_SIGSETJMP
	sj = sigsetjmp(gBrokenDataJmp, 1);
#else
	sj = setjmp(gBrokenDataJmp);
#endif	/* HAVE_SIGSETJMP */

	if (sj != 0) {
		(void) signal(SIGPIPE, (FTPSigProc) osigpipe);
		FTPShutdownHost(vcip);
		vcip->errNo = kErrRemoteHostClosedConnection;
		return(vcip->errNo);
	}
	gCanBrokenDataJmp = 1;

	if (result == 0) {
		/* This line sets the buffer pointer so that the first thing
		 * BufferGets will do is reset and fill the buffer using
		 * real I/O.
		 */
		secBufPtr = secondaryBuf + sizeof(secondaryBuf);
		secBufLimit = (char *) 0;

		while (1) {
			if (cip->xferTimeout > 0)
				(void) alarm(cip->xferTimeout);
			nread = BufferGets(line, sizeof(line), cip->dataSocket, secondaryBuf, &secBufPtr, &secBufLimit, sizeof(secondaryBuf));
			if (nread <= 0) {
				if (nread < 0)
					break;
				if (blanklines != 0)
					(void) AddLine(lines, line);
			} else {
				cip->bytesTransferred += (longest_int) nread;
				(void) AddLine(lines, line);
			}
		}
		if (cip->xferTimeout > 0)
			(void) alarm(0);
		result = FTPEndDataCmd(cip, 1);
		if (result < 0) {
			result = kErrLISTFailed;
			cip->errNo = kErrLISTFailed;
		}
		result = kNoErr;
	} else if (result == kErrGeneric) {
		result = kErrLISTFailed;
		cip->errNo = kErrLISTFailed;
	}
	(void) signal(SIGPIPE, (FTPSigProc) osigpipe);
	return (result);
}	/* FTPListToMemory2 */




/* The purpose of this is to provide updates for the progress meters
 * during lags.
 */
static void
WaitForRemoteOutput(const FTPCIPtr cip)
{
	fd_set ss;
	struct timeval tv;
	int result;
	int fd;
	int wsecs;
	int xferTimeout;

	xferTimeout = cip->xferTimeout;
	if (xferTimeout == 1)
		return;

	fd = cip->dataSocket;
	wsecs = 0;

	while ((xferTimeout <= 0) || (wsecs < xferTimeout)) {
		FD_ZERO(&ss);
		FD_SET(fd, &ss);
		tv.tv_sec = 1;
		tv.tv_usec = 0;
		result = select(fd + 1, NULL, &ss, NULL, &tv);
		if (result == 1) {
			/* ready */
			cip->stalled = 0;
			return;
		} else if (result < 0) {
			cip->stalled = 0;
			if (errno != EINTR) {
				perror("select");
				return;
			}
		} else {
			cip->stalled++;
			wsecs++;
		}
		FTPUpdateIOTimer(cip);
	}

	/* Shouldn't get here -- alarm() should have
	 * went off by now.
	 */
	(void) kill(getpid(), SIGALRM);
}	/* WaitForRemoteOutput */




static int
FTPPutOneF(const FTPCIPtr cip, const char *file, const char *volatile dstfile, int xtype, int fdtouse, volatile int appendmode, const char *volatile tmppfx, const char *volatile tmpsfx, int resumeflag, int deleteflag)
{
	char *buf, *cp;
	const char *cmd;
	const char *odstfile;
	size_t bufSize;
	size_t l;
	int tmpResult, result;
	int nread, nwrote;
	volatile int fd;
	char inbuf[256];
	char dstfile2[512];
	char *src, *srclim, *dst;
	int ntowrite;
	int fstatrc;
	longest_int startPoint;
	struct stat st;
	volatile FTPSigProc osigpipe;
	volatile FTPCIPtr vcip;
	volatile int vfd, vfdtouse;
	int sj;

	if (fdtouse < 0) {
		fd = open(file, O_RDONLY, 0);
		if (fd < 0) {
			Error(cip, kDoPerror, "Cannot open local file %s for reading.\n", file);
			cip->errNo = kErrOpenFailed;
			return (cip->errNo);
		}
	} else {
		fd = fdtouse;
	}

	fstatrc = fstat(fd, &st);

	if (resumeflag == kResumeYes) {
		resumeflag = kResumeNo;		/* Assume we can't yet. */
		if ((fstatrc == 0)
			&& (S_ISREG(st.st_mode) != 0)
			&& (xtype == kTypeBinary)
			&& (FTPFileSize(cip, dstfile, &startPoint, xtype) == 0)
		) {
			if ((longest_int) st.st_size == startPoint) {
				/* Already sent file, done. */
				if (fdtouse < 0) {
					(void) close(fd);
				}

				if (deleteflag == kDeleteYes) {
					if (unlink(file) < 0) {
						cip->errNo = kErrLocalDeleteFailed;
						return (cip->errNo);
					}
				}
				return (kNoErr);
			} else if ((longest_int) st.st_size > startPoint) {
				if (lseek(fd, (off_t) startPoint, SEEK_SET) != (off_t) -1) {
					/* Use APPE instead of REST+STOR.  */
					resumeflag = kResumeYes;
					appendmode = kAppendYes;
					cip->startPoint = startPoint;
				}
			}

		}
	}

	vcip = cip;
	vfdtouse = fdtouse;
	vfd = fd;
	osigpipe = (volatile FTPSigProc) signal(SIGPIPE, BrokenData);

	gGotBrokenData = 0;
	gCanBrokenDataJmp = 0;

#ifdef HAVE_SIGSETJMP
	sj = sigsetjmp(gBrokenDataJmp, 1);
#else
	sj = setjmp(gBrokenDataJmp);
#endif	/* HAVE_SIGSETJMP */

	if (sj != 0) {
		(void) signal(SIGPIPE, (FTPSigProc) osigpipe);
		if (vfdtouse < 0) {
			(void) close(vfd);
		}
		FTPShutdownHost(vcip);
		vcip->errNo = kErrRemoteHostClosedConnection;
		return(vcip->errNo);
	}
	gCanBrokenDataJmp = 1;

	if (appendmode != 0) {
		cmd = "APPE";
		tmppfx = "";	/* Can't use that here. */
		tmpsfx = "";
	} else {
		cmd = "STOR";
		if (tmppfx == NULL)
			tmppfx = "";
		if (tmpsfx == NULL)
			tmpsfx = "";
	}

	odstfile = dstfile;
	if ((tmppfx[0] != '\0') || (tmpsfx[0] != '\0')) {
		cp = strrchr(dstfile, '/');
		if (cp == NULL) {
			(void) STRNCPY(dstfile2, tmppfx);
			(void) STRNCAT(dstfile2, dstfile);
			(void) STRNCAT(dstfile2, tmpsfx);
		} else {
			cp++;
			l = (size_t) (cp - dstfile);
			(void) STRNCPY(dstfile2, dstfile);
			dstfile2[l] = '\0';	/* Nuke stuff after / */
			(void) STRNCAT(dstfile2, tmppfx);
			(void) STRNCAT(dstfile2, cp);
			(void) STRNCAT(dstfile2, tmpsfx);
		}
		dstfile = dstfile2;
	}

	tmpResult = FTPStartDataCmd(
		cip,
		kNetWriting,
		xtype,
		0,
		"%s %s",
		cmd,
		dstfile
	);

	if (tmpResult < 0) {
		cip->errNo = tmpResult;
		if (fdtouse < 0) {
			(void) close(fd);
		}
		(void) signal(SIGPIPE, (FTPSigProc) osigpipe);
		return (cip->errNo);
	}

	result = kNoErr;
	buf = cip->buf;
	bufSize = cip->bufSize;

	FTPInitIOTimer(cip);
	if ((fstatrc == 0) && (S_ISREG(st.st_mode) != 0)) {
		cip->expectedSize = (longest_int) st.st_size;
		cip->mdtm = st.st_mtime;
	}
	cip->lname = file;	/* could be NULL */
	cip->rname = odstfile;
	if (fdtouse >= 0)
		cip->useProgressMeter = 0;
	FTPStartIOTimer(cip);

	if (xtype == kTypeAscii) {
		/* ascii */
		while (1) {
			gCanBrokenDataJmp = 0;
			nread = read(fd, inbuf, sizeof(inbuf));
			if (nread < 0) {
				if ((gGotBrokenData != 0) || (errno == EPIPE)) {
					result = kErrWriteFailed;
					cip->errNo = kErrWriteFailed;
					errno = EPIPE;
				} else if (errno == EINTR) {
					continue;
				} else {
					Error(cip, kDoPerror, "Local read failed.\n");
					cip->errNo = kErrReadFailed;
				}
				break;
			} else if (nread == 0) {
				break;
			}
			cip->bytesTransferred += (longest_int) nread;

			gCanBrokenDataJmp = 1;
			src = inbuf;
			srclim = src + nread;
			dst = cip->buf;		/* must be 2x sizeof inbuf or more. */
			while (src < srclim) {
				if (*src == '\n')
					*dst++ = '\r';
				*dst++ = *src++;
			}
			ntowrite = (size_t) (dst - cip->buf);
			cp = cip->buf;

			if (cip->xferTimeout > 0)
				(void) alarm(cip->xferTimeout);
			WaitForRemoteOutput(cip);
			do {
				if (cip->cancelXfer > 0) {
					FTPAbortDataTransfer(cip);
					result = cip->errNo = kErrDataTransferAborted;
					goto brk;
				}
				nwrote = write(cip->dataSocket, cp, ntowrite);
				if (nwrote < 0) {
					if (errno == EINTR)
						continue;
					Error(cip, kDoPerror, "Remote write failed.\n");
					cip->errNo = kErrSocketWriteFailed;
					goto brk;
				}
				cp += nwrote;
				ntowrite -= nwrote;
			} while (ntowrite > 0);
			FTPUpdateIOTimer(cip);
		}
	} else {
		/* binary */
		while (1) {
			gCanBrokenDataJmp = 0;
			cp = buf;
			nread = read(fd, cp, bufSize);
			if (nread < 0) {
				if ((gGotBrokenData != 0) || (errno == EPIPE)) {
					result = kErrWriteFailed;
					cip->errNo = kErrWriteFailed;
					errno = EPIPE;
				} else if (errno == EINTR) {
					continue;
				} else {
					Error(cip, kDoPerror, "Local read failed.\n");
					cip->errNo = result = kErrReadFailed;
				}
				break;
			} else if (nread == 0) {
				break;
			}
			cip->bytesTransferred += (longest_int) nread;

			gCanBrokenDataJmp = 1;
			if (cip->xferTimeout > 0)
				(void) alarm(cip->xferTimeout);
			WaitForRemoteOutput(cip);
			do {
				if (cip->cancelXfer > 0) {
					FTPAbortDataTransfer(cip);
					result = cip->errNo = kErrDataTransferAborted;
					goto brk;
				}
				nwrote = write(cip->dataSocket, cp, nread);
				if (nwrote < 0) {
					if (errno == EINTR)
						continue;
					Error(cip, kDoPerror, "Remote write failed.\n");
					cip->errNo = result = kErrSocketWriteFailed;
					goto brk;
				}
				cp += nwrote;
				nread -= nwrote;
			} while (nread > 0);
			FTPUpdateIOTimer(cip);
		}
	}
brk:
	WaitForRemoteOutput(cip);	/* Close could block. */
	gCanBrokenDataJmp = 0;
	if (cip->xferTimeout > 0)
		(void) alarm(0);
	tmpResult = FTPEndDataCmd(cip, 1);
	if ((tmpResult < 0) && (result == kNoErr)) {
		cip->errNo = result = kErrSTORFailed;
	}
	FTPStopIOTimer(cip);

	if (fdtouse < 0) {
		/* If they gave us a descriptor (fdtouse >= 0),
		 * leave it open, otherwise we opened it, so
		 * we need to close it.
		 */
		(void) fstat(fd, &st);
		(void) close(fd);
	}

	if (result == kNoErr) {
		/* The store succeeded;  If we were
		 * uploading to a temporary file,
		 * move the new file to the new name.
		 */
		if ((tmppfx[0] != '\0') || (tmpsfx[0] != '\0')) {
			if ((result = FTPRename(cip, dstfile, odstfile)) < 0) {
				/* May fail if file was already there,
				 * so delete the old one so we can move
				 * over it.
				 */
				if (FTPDelete(cip, odstfile, kRecursiveNo, kGlobNo) == kNoErr) {
					result = FTPRename(cip, dstfile, odstfile);
					if (result < 0) {
						Error(cip, kDontPerror, "Could not rename %s to %s: %s.\n", dstfile, odstfile, FTPStrError(cip->errNo));
					}
				} else {
					Error(cip, kDontPerror, "Could not delete old %s, so could not rename %s to that: %s\n", odstfile, dstfile, FTPStrError(cip->errNo));
				}
			}
		}

		if (FTPUtime(cip, odstfile, st.st_atime, st.st_mtime, st.st_ctime) != kNoErr) {
			if (cip->errNo != kErrUTIMENotAvailable)
				Error(cip, kDontPerror, "Could not preserve times for %s: %s.\n", odstfile, FTPStrError(cip->errNo));
		}

		if (deleteflag == kDeleteYes) {
			if (unlink(file) < 0) {
				result = cip->errNo = kErrLocalDeleteFailed;
			}
		}
	}

	(void) signal(SIGPIPE, (FTPSigProc) osigpipe);
	return (result);
}	/* FTPPutOneF */




int
FTPPutOneFile3(const FTPCIPtr cip, const char *file, const char *dstfile, int xtype, int fdtouse, int appendflag, const char *tmppfx, const char *tmpsfx, int resumeflag, int deleteflag, int reserved)
{
	int result;

	gUnused = reserved;
	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);
	
	if ((dstfile == NULL) || (dstfile[0] == '\0'))
		return (kErrBadParameter);
	if (fdtouse < 0) {
		if ((file == NULL) || (file[0] == '\0'))
			return (kErrBadParameter);
	}
	result = FTPPutOneF(cip, file, dstfile, xtype, fdtouse, appendflag, tmppfx, tmpsfx, resumeflag, deleteflag);
	return (result);
}	/* FTPPutOneFile3 */




int
FTPPutFiles3(const FTPCIPtr cip, const char *pattern, const char *dstdir, int recurse, int doGlob, int xtype, int appendflag, const char *tmppfx, const char *tmpsfx, int resumeflag, int deleteflag, int reserved)
{
	LineList globList;
	FileInfoList files;
	FileInfoPtr filePtr;
	int batchResult;
	int result;

	gUnused = reserved;
	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);

	(void) FTPLocalGlob(cip, &globList, pattern, doGlob);
	if (recurse == kRecursiveYes) {
		(void) FTPLocalRecursiveFileList(cip, &globList, &files);
		(void) ComputeRNames(&files, dstdir, 0, 1);
	} else {
		(void) LineListToFileInfoList(&globList, &files);
		(void) ComputeLNames(&files, NULL, NULL, 1);
		(void) ComputeRNames(&files, dstdir, 0, 0);
	}
	DisposeLineListContents(&globList);

#if 0
	for (filePtr = files.first; filePtr != NULL; filePtr = filePtr->next) {
		PrintF(cip, "  R=%s, L=%s, 2=%s, size=%d, mdtm=%u, type=%c\n",
			filePtr->rname,
			filePtr->lname,
			filePtr->rlinkto ? filePtr->rlinkto : "",
			filePtr->size,
			(unsigned int) filePtr->mdtm,
			filePtr->type
		);
	}
#endif

	batchResult = kNoErr;
	for (filePtr = files.first; filePtr != NULL; filePtr = filePtr->next) {
		if (filePtr->type == 'd') {
			/* mkdir */
			result = FTPMkdir(cip, filePtr->rname, kRecursiveNo);
			if (result != kNoErr)
				batchResult = result;
#ifdef HAVE_SYMLINK
		} else if (filePtr->type == 'l') {
			/* symlink */
			/* no RFC way to create the link, though. */
			if ((filePtr->rlinkto != NULL) && (filePtr->rlinkto[0] != '\0'))
				(void) FTPSymlink(cip, filePtr->rname, filePtr->rlinkto);
#endif
		} else if (recurse != kRecursiveYes) {
			result = FTPPutOneF(cip, filePtr->lname, filePtr->rname, xtype, -1, appendflag, tmppfx, tmpsfx, resumeflag, deleteflag);
			if (result != kNoErr)
				batchResult = result;
		} else {
			result = FTPPutOneF(cip, filePtr->lname, filePtr->rname, xtype, -1, appendflag, tmppfx, tmpsfx, resumeflag, deleteflag);
			if (result != kNoErr)
				batchResult = result;
		}
	}
	DisposeFileInfoListContents(&files);
	if (batchResult < 0)
		cip->errNo = batchResult;
	return (batchResult);
}	/* FTPPutFiles3 */




/* The purpose of this is to provide updates for the progress meters
 * during lags.
 */
static void
WaitForRemoteInput(const FTPCIPtr cip)
{
	fd_set ss;
	struct timeval tv;
	int result;
	int fd;
	int wsecs;
	int xferTimeout;

	xferTimeout = cip->xferTimeout;
	if (xferTimeout == 1)
		return;

	fd = cip->dataSocket;
	wsecs = 0;

	while ((xferTimeout <= 0) || (wsecs < xferTimeout)) {
		FD_ZERO(&ss);
		FD_SET(fd, &ss);
		tv.tv_sec = 1;
		tv.tv_usec = 0;
		result = select(fd + 1, NULL, &ss, NULL, &tv);
		if (result == 1) {
			/* ready */
			return;
		} else if (result < 0) {
			if (result != EINTR) {
				perror("select");
				return;
			}
		} else {
			wsecs++;
		}
		FTPUpdateIOTimer(cip);
	}

	/* Shouldn't get here -- alarm() should have
	 * went off by now.
	 */
	(void) kill(getpid(), SIGALRM);
}	/* WaitForRemoteInput */




static int
FTPGetOneF(const FTPCIPtr cip, const char *file, const char *dstfile, int xtype, int fdtouse, longest_int expectedSize, time_t mdtm, int resumeflag, int appendflag, int deleteflag)
{
	char *buf;
	size_t bufSize;
	int tmpResult;
	volatile int result;
	int nread, nwrote;
	volatile int fd;
	char outbuf[512];
	char *src, *srclim;
	char *dst, *dstlim;
	volatile longest_int startPoint = 0;
	struct utimbuf ut;
	struct stat st;
	volatile FTPSigProc osigpipe;
	volatile FTPCIPtr vcip;
	volatile int vfd, vfdtouse;
	int sj;

	result = kNoErr;
	if (fdtouse < 0) {
		if (appendflag == kAppendYes) {
			fd = open(dstfile, O_WRONLY|O_CREAT|O_APPEND, 00666);
		} else if (resumeflag == kResumeYes) {
			if (stat(dstfile, &st) == 0) {
				/* File existed.
				 *
				 * Start transfer where
				 * the current file ends.
				 */
				startPoint = (longest_int) st.st_size;
				fd = open(dstfile, O_WRONLY|O_APPEND, 00666);
			} else {
				fd = open(dstfile, O_WRONLY|O_CREAT|O_TRUNC, 00666);
			}
		} else {
			fd = open(dstfile, O_WRONLY|O_CREAT|O_TRUNC, 00666);
		}

		if (fd < 0) {
			Error(cip, kDoPerror, "Cannot open local file %s for writing.\n", dstfile);
			result = kErrOpenFailed;
			cip->errNo = kErrOpenFailed;
			return (result);
		}
		if (cip->progress != (FTPProgressMeterProc) 0) {
			/* Only ask for extended information
			 * if we have the name of the file
			 * and we didn't already have the
			 * info.  Note that if there is no
			 * progress meter, we don't even
			 * bother requesting this info.
			 */
			if (expectedSize == kSizeUnknown)
				(void) FTPFileSize(cip, file, &expectedSize, xtype);
		}
		if (expectedSize != kSizeUnknown) {
			if (startPoint == expectedSize) {
				/* Don't go to all the trouble of downloading nothing. */
				(void) close(fd);
				return (kNoErr);
			} else if (startPoint > expectedSize) {
				/* Panic;  odds are the file we have
				 * was a different file altogether,
				 * since it is larger than the
				 * remote copy.  Re-do it all.
				 */
				startPoint = 0;
			}
		}
		if (mdtm == kModTimeUnknown)
			(void) FTPFileModificationTime(cip, file, &mdtm);
	} else {
		fd = fdtouse;
	}

	vcip = cip;
	vfdtouse = fdtouse;
	vfd = fd;
	osigpipe = (volatile FTPSigProc) signal(SIGPIPE, BrokenData);

	gGotBrokenData = 0;
	gCanBrokenDataJmp = 0;

#ifdef HAVE_SIGSETJMP
	sj = sigsetjmp(gBrokenDataJmp, 1);
#else
	sj = setjmp(gBrokenDataJmp);
#endif	/* HAVE_SIGSETJMP */

	if (sj != 0) {
		(void) signal(SIGPIPE, (FTPSigProc) osigpipe);
		if (vfdtouse < 0) {
			(void) close(vfd);
		}
		FTPShutdownHost(vcip);
		vcip->errNo = kErrRemoteHostClosedConnection;
		return(vcip->errNo);
	}
	gCanBrokenDataJmp = 1;

	tmpResult = FTPStartDataCmd(cip, kNetReading, xtype, startPoint, "RETR %s", file);

	if (tmpResult < 0) {
		result = tmpResult;
		if (result == kErrGeneric)
			result = kErrRETRFailed;
		cip->errNo = result;
		if (fdtouse < 0) {
			(void) close(fd);
			if ((appendflag == kAppendNo) && (startPoint == 0))
				(void) unlink(dstfile);
		}
		(void) signal(SIGPIPE, (FTPSigProc) osigpipe);
		return (result);
	}

	buf = cip->buf;
	bufSize = cip->bufSize;

	FTPInitIOTimer(cip);
	cip->mdtm = mdtm;
	cip->expectedSize = expectedSize;
	cip->lname = dstfile;	/* could be NULL */
	cip->rname = file;
	if (fdtouse >= 0)
		cip->useProgressMeter = 0;
	FTPStartIOTimer(cip);

	if (xtype == kTypeAscii) {
		/* Ascii */
		while (1) {
			if (cip->cancelXfer > 0) {
				FTPAbortDataTransfer(cip);
				result = cip->errNo = kErrDataTransferAborted;
				break;
			}
			gCanBrokenDataJmp = 1;
			if (cip->xferTimeout > 0)
				(void) alarm(cip->xferTimeout);
			WaitForRemoteInput(cip);
			nread = read(cip->dataSocket, buf, bufSize);
			if (nread < 0) {
				if (errno == EINTR)
					continue;
				Error(cip, kDoPerror, "Remote read failed.\n");
				result = kErrSocketReadFailed;
				cip->errNo = kErrSocketReadFailed;
				break;
			} else if (nread == 0) {
				break;
			}

			gCanBrokenDataJmp = 0;
			src = buf;
			srclim = src + nread;
			dst = outbuf;
			dstlim = dst + sizeof(outbuf);
			while (src < srclim) {
				if (*src == '\r') {
					src++;
					continue;
				}
				if (dst >= dstlim) {
					nwrote = write(fd, outbuf, (size_t) (dst - outbuf));
					if (nwrote == (int) (dst - outbuf)) {
						/* Success. */
						dst = outbuf;
					} else if ((gGotBrokenData != 0) || (errno == EPIPE)) {
						result = kErrWriteFailed;
						cip->errNo = kErrWriteFailed;
						errno = EPIPE;
						goto brk;
					} else {
						Error(cip, kDoPerror, "Local write failed.\n");
						result = kErrWriteFailed;
						cip->errNo = kErrWriteFailed;
						goto brk;
					}
				}
				*dst++ = *src++;
			}
			if (dst > outbuf) {
				nwrote = write(fd, outbuf, (size_t) (dst - outbuf));
				if (nwrote != (int) (dst - outbuf)) {
					if ((gGotBrokenData != 0) || (errno == EPIPE)) {
						result = kErrWriteFailed;
						cip->errNo = kErrWriteFailed;
						errno = EPIPE;
						goto brk;
					} else {
						Error(cip, kDoPerror, "Local write failed.\n");
						result = kErrWriteFailed;
						cip->errNo = kErrWriteFailed;
						goto brk;
					}
				}
			}

			cip->bytesTransferred += (longest_int) nread;
			FTPUpdateIOTimer(cip);
		}
	} else {
		/* Binary */
		while (1) {
			if (cip->cancelXfer > 0) {
				FTPAbortDataTransfer(cip);
				result = cip->errNo = kErrDataTransferAborted;
				break;
			}
			gCanBrokenDataJmp = 1;
			if (cip->xferTimeout > 0)
				(void) alarm(cip->xferTimeout);
			WaitForRemoteInput(cip);
			nread = read(cip->dataSocket, buf, bufSize);
			if (nread < 0) {
				if (errno == EINTR)
					continue;
				Error(cip, kDoPerror, "Remote read failed.\n");
				result = kErrSocketReadFailed;
				cip->errNo = kErrSocketReadFailed;
				break;
			} else if (nread == 0) {
				break;
			}
			gCanBrokenDataJmp = 0;
			nwrote = write(fd, buf, nread);
			if (nwrote != nread) {
				if ((gGotBrokenData != 0) || (errno == EPIPE)) {
					result = kErrWriteFailed;
					cip->errNo = kErrWriteFailed;
					errno = EPIPE;
				} else {
					Error(cip, kDoPerror, "Local write failed.\n");
					result = kErrWriteFailed;
					cip->errNo = kErrWriteFailed;
				}
				break;
			}
			cip->bytesTransferred += (longest_int) nread;
			FTPUpdateIOTimer(cip);
		}
	}
brk:
	if (cip->xferTimeout > 0)
		(void) alarm(0);
	gCanBrokenDataJmp = 0;

	if (fdtouse < 0) {
		/* If they gave us a descriptor (fdtouse >= 0),
		 * leave it open, otherwise we opened it, so
		 * we need to close it.
		 */
		(void) close(fd);
	}
	tmpResult = FTPEndDataCmd(cip, 1);
	if ((tmpResult < 0) && (result == 0)) {
		result = kErrRETRFailed;
		cip->errNo = kErrRETRFailed;
	}
	FTPStopIOTimer(cip);
	(void) signal(SIGPIPE, (FTPSigProc) osigpipe);

	if ((mdtm != kModTimeUnknown) && (cip->bytesTransferred > 0)) {
		(void) time(&ut.actime);
		ut.modtime = mdtm;
		(void) utime(dstfile, &ut);
	}

	
	if ((deleteflag == kDeleteYes) && (result == kNoErr)) {
		result = FTPDelete(cip, file, kRecursiveNo, kGlobNo);
	}

	return (result);
}	/* FTPGetOneF */




int
FTPGetOneFile3(const FTPCIPtr cip, const char *file, const char *dstfile, int xtype, int fdtouse, int resumeflag, int appendflag, int deleteflag, int reserved)
{
	int result;

	gUnused = reserved;
	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);
	
	if ((file == NULL) || (file[0] == '\0'))
		return (kErrBadParameter);
	if (fdtouse < 0) {
		if ((dstfile == NULL) || (dstfile[0] == '\0'))
			return (kErrBadParameter);
	}

	result = FTPGetOneF(cip, file, dstfile, xtype, fdtouse, kSizeUnknown, kModTimeUnknown, resumeflag, appendflag, deleteflag);
	return (result);
}	/* FTPGetOneFile3 */




int
FTPGetFiles3(const FTPCIPtr cip, const char *pattern, const char *dstdir, int recurse, int doGlob, int xtype, int resumeflag, int appendflag, int deleteflag, int reserved)
{
	LineList globList;
	LinePtr itemPtr;
	FileInfoList files;
	FileInfoPtr filePtr;
	int batchResult;
	int result;
	char *ldir;
	char *cp;

	gUnused = reserved;
	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);

	batchResult = kNoErr;
	(void) FTPRemoteGlob(cip, &globList, pattern, doGlob);
	for (itemPtr = globList.first; itemPtr != NULL; itemPtr = itemPtr->next) {
		if (recurse == kRecursiveYes) {
			(void) FTPRemoteRecursiveFileList1(cip, itemPtr->line, &files);
			(void) ComputeLNames(&files, itemPtr->line, dstdir, 1);
		} else {
			(void) LineListToFileInfoList(&globList, &files);
			(void) ComputeRNames(&files, ".", 0, 1);
			(void) ComputeLNames(&files, NULL, dstdir, 0);
		}

#if 0
		for (filePtr = files.first; filePtr != NULL; filePtr = filePtr->next) {
			PrintF(cip, "  R=%s, L=%s, 2=%s, size=%d, mdtm=%u, type=%c\n",
				filePtr->rname,
				filePtr->lname,
				filePtr->rlinkto ? filePtr->rlinkto : "",
				filePtr->size,
				(unsigned int) filePtr->mdtm,
				filePtr->type
			);
		}
#endif

		for (filePtr = files.first; filePtr != NULL; filePtr = filePtr->next) {
			if (filePtr->type == 'd') {
				(void) mkdir(filePtr->lname, 00777);
#ifdef HAVE_SYMLINK
			} else if (filePtr->type == 'l') {
				(void) unlink(filePtr->lname);
				if (symlink(filePtr->rlinkto, filePtr->lname) < 0) {
					batchResult = kErrGeneric;
				}
#endif
			} else if (recurse != kRecursiveYes) {
				result = FTPGetOneF(cip, filePtr->rname, filePtr->lname, xtype, -1, filePtr->size, filePtr->mdtm, resumeflag, appendflag, deleteflag);
				if (result != kNoErr)
					batchResult = result;
			} else {
				ldir = filePtr->lname;
				cp = strrchr(ldir, '/');
				if (cp != NULL) {
					*cp = '\0';
					if (access(ldir, F_OK) < 0)
						(void) mkdir(ldir, 00777);
					*cp = '/';
				}
				result = FTPGetOneF(cip, filePtr->rname, filePtr->lname, xtype, -1, filePtr->size, filePtr->mdtm, resumeflag, appendflag, deleteflag);
				if (result != kNoErr)
					batchResult = result;
			}
		}
		DisposeFileInfoListContents(&files);
	}

	DisposeLineListContents(&globList);
	if (batchResult < 0)
		cip->errNo = batchResult;
	return (batchResult);
}	/* FTPGetFiles3 */




/*------------------------- wrappers for old routines ----------------------*/

int
FTPGetOneFile(const FTPCIPtr cip, const char *file, const char *dstfile)
{
	return (FTPGetOneFile3(cip, file, dstfile, kTypeBinary, -1, kResumeNo, kAppendNo, kDeleteNo, 0));
}	/* FTPGetOneFile */




int
FTPGetOneFile2(const FTPCIPtr cip, const char *file, const char *dstfile, int xtype, int fdtouse, int resumeflag, int appendflag)
{
	return (FTPGetOneFile3(cip, file, dstfile, xtype, fdtouse, resumeflag, appendflag, kDeleteNo, 0));
}	/* FTPGetOneFile2 */




int
FTPGetFiles(const FTPCIPtr cip, const char *pattern, const char *dstdir, int recurse, int doGlob)
{
	return (FTPGetFiles3(cip, pattern, dstdir, recurse, doGlob, kTypeBinary, kResumeNo, kAppendNo, kDeleteNo, 0));
}	/* FTPGetFiles */




int
FTPGetFiles2(const FTPCIPtr cip, const char *pattern, const char *dstdir, int recurse, int doGlob, int xtype, int resumeflag, int appendflag)
{
	return (FTPGetFiles3(cip, pattern, dstdir, recurse, doGlob, xtype, resumeflag, appendflag, kDeleteNo, 0));
}	/* FTPGetFiles2 */




int
FTPGetOneFileAscii(const FTPCIPtr cip, const char *file, const char *dstfile)
{
	return (FTPGetOneFile3(cip, file, dstfile, kTypeAscii, -1, kResumeNo, kAppendNo, kDeleteNo, 0));
}	/* FTPGetOneFileAscii */




int
FTPGetFilesAscii(const FTPCIPtr cip, const char *pattern, const char *dstdir, int recurse, int doGlob)
{
	return (FTPGetFiles3(cip, pattern, dstdir, recurse, doGlob, kTypeAscii, kResumeNo, kAppendNo, kDeleteNo, 0));
}	/* FTPGetFilesAscii */




int
FTPPutOneFile(const FTPCIPtr cip, const char *file, const char *dstfile)
{
	return (FTPPutOneFile3(cip, file, dstfile, kTypeBinary, -1, 0, NULL, NULL, kResumeNo, kDeleteNo, 0));
}	/* FTPPutOneFile */




int
FTPPutOneFile2(const FTPCIPtr cip, const char *file, const char *dstfile, int xtype, int fdtouse, int appendflag, const char *tmppfx, const char *tmpsfx)
{
	return (FTPPutOneFile3(cip, file, dstfile, xtype, fdtouse, appendflag, tmppfx, tmpsfx, kResumeNo, kDeleteNo, 0));
}	/* FTPPutOneFile2 */




int
FTPPutFiles(const FTPCIPtr cip, const char *pattern, const char *dstdir, int recurse, int doGlob)
{
	return (FTPPutFiles3(cip, pattern, dstdir, recurse, doGlob, kTypeBinary, 0, NULL, NULL, kResumeNo, kDeleteNo, 0));
}	/* FTPPutFiles */




int
FTPPutFiles2(const FTPCIPtr cip, const char *pattern, const char *dstdir, int recurse, int doGlob, int xtype, int appendflag, const char *tmppfx, const char *tmpsfx)
{
	return (FTPPutFiles3(cip, pattern, dstdir, recurse, doGlob, xtype, appendflag, tmppfx, tmpsfx, kResumeNo, kDeleteNo, 0));
}	/* FTPPutFiles2 */




int
FTPPutOneFileAscii(const FTPCIPtr cip, const char *file, const char *dstfile)
{
	return (FTPPutOneFile3(cip, file, dstfile, kTypeAscii, -1, 0, NULL, NULL, kResumeNo, kDeleteNo, 0));
}	/* FTPPutOneFileAscii */




int
FTPPutFilesAscii(const FTPCIPtr cip, const char *pattern, const char *dstdir, int recurse, int doGlob)
{
	return (FTPPutFiles3(cip, pattern, dstdir, recurse, doGlob, kTypeAscii, 0, NULL, NULL, kResumeNo, kDeleteNo, 0));
}	/* FTPPutFilesAscii */



int
FTPListToMemory(const FTPCIPtr cip, const char *pattern, LineListPtr lines, const char *lsflags)
{
	return (FTPListToMemory2(cip, pattern, lines, lsflags, 1));
}	/* FTPListToMemory */

/* eof IO.c */
