/* cpdfAxis.c  -- axis at any orientation (horizontal, vertical, and oblique)
 * Copyright (C) 1998 FastIO Systems, All Rights Reserved.
 * For conditions of use, license, and distribution, see LICENSE.txt or LICENSE.pdf.

1998-08-24 [IO]
*/

#include "version.h"

#include <string.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "cpdflib.h"		/* This must be included before all other local include files */
#include "cglobals.h"

typedef struct {
	float Tcut;
	int   MinorVar;
	int   MajorVar;
	int   MinorBump;
	int   MajorBump;
} CPDFtimeLU;

/* This is for transfering edge of numbers to figure out how far move the axis label off */
static float numEdgeY;
/* ohter temprary data area */
static double xa2points, xaLlog, xaHlog;
static CPDFaxis *anAx2;


/* ------------------------------------------------------------------------------------------ */

/* A plot domain must exist before calling this function. */
CPDFaxis *cpdf_createAxis(float angle, float axislength, int typeflag, float valL, float valH)
{
CPDFaxis *anAx;
float sL, sH;
	anAx = (CPDFaxis *) malloc((size_t)sizeof(CPDFaxis));
	_cpdf_malloc_check((void *)anAx);
	/* It will be attached later by cpdf_attachAxisToDomain() to a real plot domain.
	   But, for now just attach it to the default domain.
	*/
	anAx->magic = (unsigned long)AXIS_MAGIC_NUMBER;
	anAx->plotDomain = defaultDomain;
	anAx->angle = angle;
	anAx->type = typeflag;
	anAx->xloc = 0.0;
	anAx->yloc = 0.0;
	anAx->length = axislength;
	if(typeflag == LINEAR ) {
	    cpdf_suggestLinearDomainParams(valL, valH, &sL, &sH,
		&anAx->valFirstTicLinMajor, &anAx->ticIntervalLinMajor,
		&anAx->valFirstTicLinMinor, &anAx->ticIntervalLinMinor);
	}
	anAx->valL = valL;
	anAx->valH = valH;

	anAx->ticEnableMajor = 1;
	anAx->ticEnableMinor = 1;
	anAx->ticLenMajor = 10.0;
	anAx->ticLenMinor = 5.0;
	anAx->axisLineWidth = 1.0;
	anAx->tickWidthMajor = 2.0;
	anAx->tickWidthMinor = 1.0;
	if(angle > 60.0 && angle < 120.0) {
	    anAx->ticPosition = 0;	/* almost vertical axis */
	    anAx->numPosition = 2;
	}
	else {
	    anAx->ticPosition = 2;
	    anAx->numPosition = 0;
	}

	anAx->numEnable = 1;		/* put numbers */
	anAx->ticNumGap = 7.0;
	anAx->numFontSize = 14.0;
	anAx->horizNumber = 1;
	anAx->numStyle = 0;	 /* 0=Regular, 1=exponential, 2=free style (e.g. Jan98  Feb98 ..) */
	anAx->numFormat = NULL;
	anAx->numFontName = NULL;

	anAx->numLabelGap = 7.0; /* gap (in points) between number and axis label */
	anAx->labelFontSize = 18.0;
	anAx->horizLabel = 0;	 /* if true, label will be plotted at horizontal orientation */
	anAx->labelEncoding = NULL;
	anAx->labelFontName = NULL;
	anAx->axisLabel = NULL;  /* Axis label string, if NULL, no label is shown */
	/* ------------------------------------------------------------ */
	/* num   min   1    2    3    4    5    6    7    8    9   max  */
	/* bit    0    1    2    3    4    5    6    7    8    9    10  */
	/* ------------------------------------------------------------ */
	anAx->numSelectorLog = LOGAXSEL_1;
	anAx->ticSelectorLog = LOGAXSEL_123456789;

	cpdf_setAxisNumberFormat(anAx, "%g", "Helvetica", 12.0);

	return anAx;
}

/* Create Time Axis */
CPDFaxis *cpdf_createTimeAxis(float angle, float axislength, int typeflag, struct tm *vTL, struct tm *vTH)
{
CPDFaxis *anAx;
	anAx = (CPDFaxis *) malloc((size_t)sizeof(CPDFaxis));
	_cpdf_malloc_check((void *)anAx);
	/* It will be attached later by cpdf_attachAxisToDomain() to a real plot domain.
	   But, for now just attach it to the default domain.
	*/
	anAx->magic = (unsigned long)AXIS_MAGIC_NUMBER;
	anAx->plotDomain = defaultDomain;
	anAx->angle = angle;
	anAx->type = typeflag;
	anAx->xloc = 0.0;
	anAx->yloc = 0.0;
	anAx->length = axislength;
	mktime(vTL); mktime(vTH);	/* This sets all fields of struct tm correctly */
	memcpy(&anAx->vTL, vTL, sizeof(struct tm));
	memcpy(&anAx->vTH, vTH, sizeof(struct tm));
	anAx->valL = 0.0;			/* keep also float representation for date */
	anAx->valH = tm_to_NumDays(vTL, vTH);	 /* in float number of days */
	anAx->ticEnableMajor = 1;
	anAx->ticEnableMinor = 1;
	anAx->ticLenMajor = 10.0;
	anAx->ticLenMinor = 5.0;
	anAx->axisLineWidth = 1.0;
	anAx->tickWidthMajor = 2.0;
	anAx->tickWidthMinor = 1.0;
	if(angle > 60.0 && angle < 120.0) {
	    anAx->ticPosition = 0;	/* almost vertical axis */
	    anAx->numPosition = 2;
	}
	else {
	    anAx->ticPosition = 2;
	    anAx->numPosition = 0;
	}

	anAx->numEnable = 1;		/* put numbers */
	anAx->ticNumGap = 7.0;
	anAx->numFontSize = 14.0;
	anAx->useMonthName = 1;		/* use month names */
	anAx->use2DigitYear = 0;	/* use full year number as in 1998 */
	anAx->horizNumber = 1;
	anAx->numStyle = 0;	 /* 0=Regular, 1=exponential, 2=free style (e.g. Jan98  Feb98 ..) */
	anAx->numFormat = NULL;
	anAx->numFontName = NULL;

	anAx->numLabelGap = 7.0; /* gap (in points) between number and axis label */
	anAx->labelFontSize = 18.0;
	anAx->horizLabel = 0;	 /* if true, label will be plotted at horizontal orientation */
	anAx->labelEncoding = NULL;
	anAx->labelFontName = NULL;
	anAx->axisLabel = NULL;  /* Axis label string, if NULL, no label is shown */
	/* ------------------------------------------------------------ */
	/* num   min   1    2    3    4    5    6    7    8    9   max  */
	/* bit    0    1    2    3    4    5    6    7    8    9    10  */
	/* ------------------------------------------------------------ */
	anAx->numSelectorLog = LOGAXSEL_1;
	anAx->ticSelectorLog = LOGAXSEL_123456789;

	cpdf_setTimeAxisNumberFormat(anAx, MONTH_NAME, YEAR_2DIGIT, "Helvetica", 12.0);

	return anAx;
}



void cpdf_freeAxis(CPDFaxis *anAx)
{
	if(anAx->axisLabel) free(anAx->axisLabel);
	if(anAx->labelFontName) free(anAx->labelFontName);
	if(anAx->labelEncoding) free(anAx->labelEncoding);
	if(anAx->numFormat) free(anAx->numFormat);
	if(anAx->numFontName) free(anAx->numFontName);
	if(anAx) free(anAx);
}


/* This function draws the axis */
void cpdf_drawAxis(CPDFaxis *anAx)
{
float a, b, c, d, e, f, angle, vcos, vsin;
    anAx2 = anAx;
    angle = PI*(anAx->angle)/180.0;
    vcos = cos(angle);
    vsin = sin(angle);
    a =  vcos;
    b =  vsin;
    c = -vsin;
    d =  vcos;
    /* axis origin is relative to the plotDomain to which it is attached */
    e = (anAx->xloc) + (anAx->plotDomain)->xloc;
    f = (anAx->yloc) + (anAx->plotDomain)->yloc;
    cpdf_comments("\n% Axis starts here \n");
    cpdf_gsave();
    cpdf_rawConcat(a, b, c, d, e, f);	/* This moves the origin to axis origin, and rotates. */

    /* precompute these scaling factors */
    xa2points = (anAx->length)/((anAx->valH) - (anAx->valL));
    if(anAx->type == 1) {
	xaLlog = log10((double)(anAx->valL));
	xaHlog = log10((double)(anAx->valH));
    }

    /* OK, now draw the main axis line. */
    cpdf_setlinewidth(anAx->axisLineWidth);
    /* Project out line ends by 1/2 of Major tick linewidth
       This eliminates a notch at the axis ends when ticks are one-sided. */
    cpdf_rawMoveto(-0.5*anAx->tickWidthMajor, 0.0);
    cpdf_rawLineto(anAx->length + 0.5*anAx->tickWidthMajor, 0.0);
    cpdf_stroke();

    if(anAx->type == 0) {
	_do_linearTics(anAx);
	_do_linearNumbers(anAx);
    }
    else if(anAx->type == 1) {
	_do_logTics(anAx);
	_do_logNumbers(anAx);
    }
    else if(anAx->type == 2) {
	_do_timeTics(anAx);
	_do_timeNumbers(anAx);
    }

    _do_axisLabel(anAx);

    cpdf_grestore();
}

void cpdf_attachAxisToDomain(CPDFaxis *anAx, CPDFplotDomain *domain, float x, float y)
{
	anAx->plotDomain = domain;
	anAx->xloc = x;
	anAx->yloc = y;
}

void cpdf_setTicNumEnable(CPDFaxis *anAx, int ticEnableMaj, int ticEnableMin, int numEnable)
{
	anAx->ticEnableMajor = ticEnableMaj;
	anAx->ticEnableMinor = ticEnableMin;
	anAx->numEnable = numEnable;		/* put numbers */
}

void cpdf_setAxisTicNumLabelPosition(CPDFaxis *anAx, int ticPos, int numPos, int horizNum, int horizLab)
{
	anAx->horizNumber = horizNum;
	anAx->horizLabel = horizLab;	/* if true, label will be plotted at horizontal orientation */
	anAx->ticPosition = ticPos;
	anAx->numPosition = numPos;
}


/* ----------------------------------------------------------------------- */
/*  Axis Numbering fuctions */
/* ----------------------------------------------------------------------- */

/* This shouldn't be necessary, but fixes problems with %g formatting with
   some compilers/libraries. Specifically to fix stupid Metrowerks
   ( which prints trailing zeros with %g.
   E.g., fix "0.300000" to correct "0.3" */
char *fix_trailingZeros(char *sstr)
{
char *p, *pe;
int ch;
	pe = sstr + strlen(sstr);	/* on NULL */
	p = strchr(sstr, '.');
	if(p != NULL) {
	    /* we have a decimal point */
	    pe--;		/* on the last char */
	    while(((ch = *pe) == '0') && pe > p)
		pe--;
	    pe++;
	    *pe = '\0';
	}
	return(sstr);
}

void _do_oneNumber(CPDFaxis *anAx, float v, float ticlen)
{
float startX, startY, ticstartN, ticstartP, vt, ang;
float slen, hslen;
char str[32];
	sprintf(str, anAx->numFormat, v);
	if(anAx->type == 1)
	    fix_trailingZeros(str);	/* If LOG axis. (Only MacOS Metrowerks CW needs it, IDE 1.7.4) */
	slen = cpdf_stringWidth((unsigned char *)str);	/* number string width for centring */
	hslen = slen*0.5;
	ang = PI*anAx->angle/180.0;
	vt = vAxis2Points(v);		/* convert domain value for number to points */
	if(anAx->ticPosition == 0) { ticstartN = -ticlen; ticstartP = 0.0;}
	else if(anAx->ticPosition == 1) { ticstartN = -0.5*ticlen; ticstartP = 0.5*ticlen;}
	else { ticstartN = 0.0; ticstartP = ticlen;}
	if(anAx->horizNumber) {
	    /* number stays always horizontal */
	    if(anAx->numPosition == 0) {
		startY = ticstartN - (anAx->ticNumGap) - (anAx->numFontSize)*0.6;
		numEdgeY = startY - 2.5;	/* store edge of numbers for positioning label */
	    }
	    else {
		startY = ticstartP + (anAx->ticNumGap) + slen;
		numEdgeY = startY + 4.0;
		/* FIXME: note a potential problem -- this function is called multiple times for
			  for a given axis, numEdgeY will store the last value.  If this is not
			  the longest string, there is a problem.
		*/
	    }
	    startX = vt - ((anAx->numFontSize)*0.3*sin(ang) + hslen*cos(ang));
	    cpdf_rawText(startX, startY, -(anAx->angle), str);
	}
	else {
	    /* number is parallel to axis and rotate with axis */
	    if(anAx->numPosition == 0) {
		startY = ticstartN - (anAx->ticNumGap) - (anAx->numFontSize)*0.6;
		numEdgeY = startY;	/* store edge of numbers for positioning label */
	    }
	    else {
		startY = ticstartP + (anAx->ticNumGap);
		numEdgeY = startY + (anAx->numFontSize);
	    }
	    startX = vt - hslen;
	    cpdf_rawText(startX, startY, 0.0, str);
	}
}

char *_yearFormat(int year, int flag)
{
static char stbuf[8];
	if(flag)
	    sprintf(stbuf, "%02d", year%100);
	else
	    sprintf(stbuf, "%d", year);
	return(stbuf);
}

int cpdf_setMonthNames(char *mnArray[])
{
int i;
    _cpdf_freeMonthNames();
    for(i=0; i<12; i++) {
	monthName[i] = (char *)malloc((size_t)(strlen(mnArray[i]) + 1));
	_cpdf_malloc_check((void *)monthName[i]);
	strcpy(monthName[i], mnArray[i]);
    }
    return(0);
}

int _cpdf_freeMonthNames(void)
{
int i;
    for(i=0; i<12; i++) {
     	if(monthName[i])
	    free(monthName[i]);
    }
    return(0);
}


static int lastMin=99, lastHour=99, lastDay=99, lastMonth=99, lastYear=0;

void _do_oneTimeNumber(CPDFaxis *anAx, float v, struct tm *vtm, int majorBumpVar, float ticlen)
{
float startX, startY, ticstartN, ticstartP, vt, ang;
float slen, hslen;
char str[32];
	/* _printfTime(vtm); */		/* debugging */
	switch(majorBumpVar) {
	    case SECOND:
			if(lastMin != vtm->tm_min)
			    sprintf(str, "%d:%d", vtm->tm_min, vtm->tm_sec);
			else
			    sprintf(str, "%d", vtm->tm_sec);
			break;
	    case MINUTE:
			if(lastHour != vtm->tm_hour)
			    sprintf(str, "%d:%d", vtm->tm_hour, vtm->tm_min);
			else
			    sprintf(str, "%d", vtm->tm_min);
			break;
	    case HOUR:
			if(lastDay != vtm->tm_mday)
			    sprintf(str, "%d [%d]", vtm->tm_hour, vtm->tm_mday);
			else
			    sprintf(str, "%d", vtm->tm_hour);
			break;
	    case DAY:
			if(lastMonth != vtm->tm_mon) {
			    if(anAx->useMonthName)
				sprintf(str, "%s %d", monthName[vtm->tm_mon], vtm->tm_mday);
			    else
				sprintf(str, "%d/%d", vtm->tm_mon, vtm->tm_mday);
			}
			else if(vtm->tm_mday < 30)
			    sprintf(str, "%d", vtm->tm_mday);
			break;
	    case MONTH:
			if(lastYear != vtm->tm_year) {
			    if(anAx->useMonthName)
				sprintf(str, "%s %s", monthName[vtm->tm_mon], 
					_yearFormat(vtm->tm_year + 1900, anAx->use2DigitYear) );
			    else
				sprintf(str, "%d/%s", vtm->tm_mon+1,
					_yearFormat(vtm->tm_year + 1900, anAx->use2DigitYear) );
			}
			else {
			    if(anAx->useMonthName)
				sprintf(str, "%s", monthName[vtm->tm_mon]);
			    else
				sprintf(str, "%d", vtm->tm_mon+1);
			}
			break;
	    case YEAR:
			sprintf(str, "%s", _yearFormat(vtm->tm_year + 1900, anAx->use2DigitYear) );
			break;
	}
	/* sprintf(str, anAx->numFormat, v); */
	if(anAx->type == 1)
	    fix_trailingZeros(str);	/* If LOG axis. (Only MacOS Metrowerks CW needs it, IDE 1.7.4) */
	slen = cpdf_stringWidth((unsigned char *)str);	/* number string width for centring */
	hslen = slen*0.5;
	ang = PI*anAx->angle/180.0;
	vt = vAxis2Points(v);		/* convert domain value for number to points */
	if(anAx->ticPosition == 0) { ticstartN = -ticlen; ticstartP = 0.0;}
	else if(anAx->ticPosition == 1) { ticstartN = -0.5*ticlen; ticstartP = 0.5*ticlen;}
	else { ticstartN = 0.0; ticstartP = ticlen;}
	if(anAx->horizNumber) {
	    /* number stays always horizontal */
	    if(anAx->numPosition == 0) {
		startY = ticstartN - (anAx->ticNumGap) - (anAx->numFontSize)*0.6;
		numEdgeY = startY - 2.5;	/* store edge of numbers for positioning label */
	    }
	    else {
		startY = ticstartP + (anAx->ticNumGap) + slen;
		numEdgeY = startY + 4.0;
		/* FIXME: note a potential problem -- this function is called multiple times for
			  for a given axis, numEdgeY will store the last value.  If this is not
			  the longest string, there is a problem.
		*/
	    }
	    /* startX = vt - ((anAx->numFontSize)*0.3*sin(ang) + hslen*cos(ang)); */
	    /* do not center date/time values on tick */
	    startX = vt - ((anAx->numFontSize)*0.3*sin(ang) + (anAx->numFontSize)*0.25);
	    cpdf_rawText(startX, startY, -(anAx->angle), str);
	}
	else {
	    /* number is parallel to axis and rotate with axis */
	    if(anAx->numPosition == 0) {
		startY = ticstartN - (anAx->ticNumGap) - (anAx->numFontSize)*0.6;
		numEdgeY = startY;	/* store edge of numbers for positioning label */
	    }
	    else {
		startY = ticstartP + (anAx->ticNumGap);
		numEdgeY = startY + (anAx->numFontSize);
	    }
	    /* startX = vt - hslen; */
	    startX = vt - (anAx->numFontSize)*0.25;  /* do not center date/time values on tick */
	    cpdf_rawText(startX, startY, 0.0, str);
	}
	/* Save time fields for showing next field up when it rolls up */
	lastMin = vtm->tm_min;
	lastHour = vtm->tm_hour;
	lastDay = vtm->tm_mday;
	lastMonth = vtm->tm_mon;
	lastYear = vtm->tm_year;
}



void _do_linearNumbers(CPDFaxis *anAx)
{
float vL, vH, v;
	cpdf_beginText(0);
	cpdf_setFont(anAx->numFontName, "MacRomanEncoding", anAx->numFontSize);
	vL = anAx->valL;
	vH = anAx->valH;
	/* do numbers at major ticks */
	for(v = anAx->valFirstTicLinMajor; v <= vH*1.0001; v += anAx->ticIntervalLinMajor) {
	    _do_oneNumber(anAx, v, anAx->ticLenMajor);
	}
	cpdf_endText();
}

void _do_logNumbers(CPDFaxis *anAx)
{
float vL, vH, v;
int m, ep;
int mL, expL, mH, expH;	/* mantissa MSD and exponents */
	cpdf_beginText(0);
	/* Encoding shouldn't matter for numbers on the axis. */
	cpdf_setFont(anAx->numFontName, "MacRomanEncoding", anAx->numFontSize);
	vL = anAx->valL;
	vH = anAx->valH;
	mL = (int)getMantissaExp(vL*1.0001, &expL);
	mH = (int)getMantissaExp(vH*1.0001, &expH);
	/* fprintf(stderr, "%dE%d -- %dE%d\n", mL, expL, mH, expH); */
	/* do numbers at major ticks */
	for(m=mL, ep=expL; (v = (float)m * pow(10.0, (double)ep)) <= vH*1.0001; ) {
	    if( _bittest(anAx->numSelectorLog, m) )	/* Test if we want this number */
	        _do_oneNumber(anAx, v, anAx->ticLenMajor);
	    m++;
	    if(m >= 10) {	/* cycle m through 1 .. 9 */
		m = 1;
		ep++;
	    }
	}
	cpdf_endText();
}


void _do_timeNumbers(CPDFaxis *anAx)
{
float fndays, v;
int minorBump = 1, majorBump = 1;
int minorBumpVar = MINUTE, majorBumpVar = HOUR;
struct tm vtm;
	lastMin=99; lastHour=99; lastDay=99; lastMonth=99; lastYear=0;
	cpdf_beginText(0);
	/* Encoding shouldn't matter for numbers on the axis. */
	cpdf_setFont(anAx->numFontName, "MacRomanEncoding", anAx->numFontSize);

	/* float number of days spanned by the axis */
	fndays = tm_to_NumDays(&anAx->vTL, &anAx->vTH);
	_setDefaultTimeBumpVar(fndays, &minorBumpVar, &majorBumpVar, &minorBump, &majorBump);
	/* do numbers at major ticks */
	memcpy(&vtm, &anAx->vTL, sizeof(struct tm));
	for(v = tm_to_NumDays(&anAx->vTL, &vtm) ; v <= fndays*1.0001 ;
		v = _bump_tm_Time(&anAx->vTL, &vtm, majorBumpVar, majorBump)) {
	    _do_oneTimeNumber(anAx, v, &vtm, majorBumpVar, anAx->ticLenMajor);
	}
	cpdf_endText();
}

/*                              1   2   3   4   5   6   7   8   9  10  11  12 */
static int daysInMon[][12] = {{31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31},
			      {31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31}};

float _bump_tm_Time(struct tm *rT, struct tm *vT, int bumpVar, int bump)
{
float fndays;
int leapY = 0;
	mktime(vT);		/* correct all fields of struct tm */
        leapY = isLeapYear(vT->tm_year + 1900);
	switch(bumpVar) {
	    case SECOND:
			vT->tm_sec += bump;
			break;
	    case MINUTE:
			vT->tm_min += bump;
			break;
	    case HOUR:
			vT->tm_hour += bump;
			break;
	    case DAY:
			vT->tm_mday += bump;
			if(vT->tm_mday > daysInMon[leapY][vT->tm_mon]) {
			    vT->tm_mday = 1;
			    vT->tm_mon++;
			}
			else {
			    /* do: [1, 5, 10, 15, 20, 25, 30] or [1, 10, 20, 30] */
			    if(( (bump == 5 && vT->tm_mday % 5 == 1) ||
				 (bump == 10 && vT->tm_mday % 10 == 1) ) && vT->tm_mday > 5 )
				vT->tm_mday--;
			}
			break;
	    case MONTH:
			vT->tm_mon += bump;
			break;
	    case YEAR:
			vT->tm_year += bump;
			break;
	    default:
			break;
	}
	fndays = tm_to_NumDays(rT, vT);		/* This calls mktime() for both internally */
	return(fndays);
}


/*  You may edit/add entries in this table to customize Time domain mesh and
	axis tick spacings.
	Tcut 	   : max # of days (float value) that the entry applies.
	MinorVar   : variable that defines minor tick intervals
	MajorVar   : variable that defines major tick intervals
	MinorBump  : Minor tick interval in the unit of MinorVar (interger)
	MajorBump  : Major tick interval in the unit of MajorVar (interger)
*/

static CPDFtimeLU timeLU[] = {
    /*	  Tcut	 MinorVar, 	MajorVar	MinorBump	MajorBump */
	{ 0.042,   MINUTE,	MINUTE,		 1,		10 },	/* 1 hour or less */
	{ 0.084,   MINUTE,	HOUR,		10,		1 },	/* 2 hours or less */
	{  0.51,   MINUTE,	HOUR,		30,		1 },	/* 12 hours or less */
	{   1.1,   HOUR,	HOUR,		1,		6 },	/* 1 day or less */
	{   8.0,   HOUR,	DAY,		6,		1 },	/* 8 days or less */
	{  12.5,   HOUR,	DAY,		12,		1 },	/* 12 days or less */
	{  21.5,   HOUR,	DAY,		12,		5 },	/* 21 days or less */
	{  42.0,   DAY,		DAY,		1,		10 },	/* 42 days or less */
	{ 190.0,   DAY,		MONTH,		10,		1 },	/* 190 days or less */
	{ 367.0,   MONTH,	MONTH,		1,		3 },	/* 1 year of less */
	{ 735.0,   MONTH,	MONTH,		1,		6 },	/* 2 years or less */
	{ 2193.0,  MONTH,	YEAR,		6,		1 },	/* 6 years or less */
	{ 7306.0,  YEAR,	YEAR,		1,		5 },	/* 20 years or less */
	{ 21916.0, YEAR,	YEAR,		1,		10 },	/* 60 years or less */
	{ 54788.0, YEAR,	YEAR,		5,		10 },	/* 150 years or less */
	{ 99999.0, YEAR,	YEAR,		10,		100 }	/* long time */
};

#define	NtimeLU		(sizeof(timeLU)/sizeof(CPDFtimeLU))

void _setDefaultTimeBumpVar(float fndays, int *minorBumpVar, int *majorBumpVar, int *minorBump, int *majorBump)
{
int i, idx =-1;

	*minorBumpVar = MINUTE;
	*majorBumpVar = HOUR;
	*minorBump = 10;
	*majorBump = 1;

	for(i=0; i < NtimeLU; i++) {
	    if(  fndays <= timeLU[i].Tcut ) {
		*minorBumpVar = timeLU[i].MinorVar;
		*majorBumpVar = timeLU[i].MajorVar;
		*minorBump = timeLU[i].MinorBump;
		*majorBump = timeLU[i].MajorBump;
		idx = i;
		break;
	    }
	}

	if(idx < 0) {
	    *minorBumpVar = YEAR;
	    *majorBumpVar = YEAR;
	    *minorBump = 10;
	    *majorBump = 100;
	}
}


/* return non-zero if aNumber (in binary representation) has bitpos ON */
/* bitpos must be 0 .. MSB of int */
int _bittest(int aNumber, int bitpos)
{
int tmask = 1;
	tmask <<= bitpos;
	if(aNumber & tmask) return(1);
	else		    return(0);
}

void cpdf_setAxisNumberFormat(CPDFaxis *anAx, char *format, char *fontName, float fontSize)
{
	if(anAx->numFormat) free(anAx->numFormat);
	if(anAx->numFontName) free(anAx->numFontName);
	anAx->numFormat = (char *)malloc((size_t)(strlen(format) + 1));
	_cpdf_malloc_check((void *)anAx->numFormat);
	anAx->numFontName = (char *)malloc((size_t)(strlen(fontName) + 1));
	_cpdf_malloc_check((void *)anAx->numFontName);
	strcpy(anAx->numFormat, format);
	strcpy(anAx->numFontName, fontName);
	anAx->numFontSize = fontSize;
}

void cpdf_setTimeAxisNumberFormat(CPDFaxis *anAx, int useMonName, int use2DigYear, char *fontName, float fontSize)
{
	if(anAx->numFontName) free(anAx->numFontName);
	anAx->numFontName = (char *)malloc((size_t)(strlen(fontName) + 1));
	_cpdf_malloc_check((void *)anAx->numFontName);
	strcpy(anAx->numFontName, fontName);
	anAx->numFontSize = fontSize;
	anAx->use2DigitYear = use2DigYear;
	anAx->useMonthName = useMonName;
}


/* ----------------------------------------------------------------------- */
/*  Axis Tic fuctions */
/* ----------------------------------------------------------------------- */

void cpdf_setAxisLineParams(CPDFaxis *anAx, float axLineWidth, float ticLenMaj, float ticLenMin,
				float tickWidMaj, float tickWidMin)
{
	anAx->ticLenMajor = ticLenMaj;
	anAx->ticLenMinor = ticLenMin;
	anAx->axisLineWidth = axLineWidth;
	anAx->tickWidthMajor = tickWidMaj;
	anAx->tickWidthMinor = tickWidMin;
}


void _do_oneTick(CPDFaxis *anAx, float vt, float ticlen)
{
float tstart, tend;
	if(anAx->ticPosition == 0) {
	     tstart = -ticlen;
	     tend = 0.0;
	     /* tend = 0.4 * anAx->axisLineWidth; */	/* Adjust 1/2 linewidth to avoid a notch at axis ends */
	}
	else if(anAx->ticPosition == 1) {
	    tstart = -0.5*ticlen;
	    tend = 0.5*ticlen;
	}
	else {
	    /* tstart = -0.4 * anAx->axisLineWidth; */ /* Adjust 1/2 linewidth to avoid a notch at axis ends */
	    tstart = 0.0;
	    tend = ticlen;
	}
	cpdf_rawMoveto(vt, tstart);
	cpdf_rawLineto(vt, tend);
}


void _do_linearTics(CPDFaxis *anAx)
{
float vL, vH, v, vt;
    vL = anAx->valL;
    vH = anAx->valH;

    /* do minor ticks */
    if(anAx->ticEnableMinor) {
        cpdf_setlinewidth(anAx->tickWidthMinor);
	for(v = anAx->valFirstTicLinMinor; v <= vH*1.0001; v += anAx->ticIntervalLinMinor) {
	    vt = vAxis2Points(v);
	    _do_oneTick(anAx, vt, anAx->ticLenMinor);
	}
	cpdf_stroke();
    }

    /* do major ticks */
    if(anAx->ticEnableMajor) {
        cpdf_setlinewidth(anAx->tickWidthMajor);
	for(v = anAx->valFirstTicLinMajor; v <= vH*1.0001; v += anAx->ticIntervalLinMajor) {
	    vt = vAxis2Points(v);
	    _do_oneTick(anAx, vt, anAx->ticLenMajor);
	}
	cpdf_stroke();
    }
}


void _do_logTics(CPDFaxis *anAx)
{
float vL, vH, v, vt;
int m, ep;
int mL, expL, mH, expH;	/* mantissa MSD and exponents */
	vL = anAx->valL;
	vH = anAx->valH;
	mL = (int)getMantissaExp(vL*1.0001, &expL);
	mH = (int)getMantissaExp(vH*1.0001, &expH);
	/* fprintf(stderr, "%dE%d -- %dE%d\n", mL, expL, mH, expH); */

	/* do ticks at 1 sig digits */
	for(m=mL, ep=expL; (v = (float)m * pow(10.0, (double)ep)) <= vH*1.0001; ) {
	    /* fprintf(stderr, "tick %g\n", v); */
	    vt = vAxis2Points(v);
	    if( _bittest(anAx->ticSelectorLog, m) ) {	/* Test if we want this tick */
		if(m==1) {
        	    cpdf_setlinewidth(anAx->tickWidthMajor);
		    _do_oneTick(anAx, vt, anAx->ticLenMajor);
		    cpdf_stroke();
		}
		else {
        	    cpdf_setlinewidth(anAx->tickWidthMinor);
		    _do_oneTick(anAx, vt, anAx->ticLenMinor);
		    cpdf_stroke();
		}
	    }
	    m++;
	    if(m >= 10) {
		m = 1;
		ep++;
	    }
	}
	/* cpdf_stroke(); */
}


void _do_timeTics(CPDFaxis *anAx)
{
float v, vt;
float fndays;
int minorBump = 1, majorBump = 1;
int minorBumpVar = MINUTE, majorBumpVar = HOUR;
struct tm vtm;

    fndays = tm_to_NumDays(&anAx->vTL, &anAx->vTH);
    _setDefaultTimeBumpVar(fndays, &minorBumpVar, &majorBumpVar, &minorBump, &majorBump);

    /* do minor ticks */
    if(anAx->ticEnableMinor) {
	cpdf_setlinewidth(anAx->tickWidthMinor);
	memcpy(&vtm, &anAx->vTL, sizeof(struct tm));
	for(v = tm_to_NumDays(&anAx->vTL, &vtm) ; v <= fndays*1.0001 ;
			v = _bump_tm_Time(&anAx->vTL, &vtm, minorBumpVar, minorBump)) {
	    /* _printfTime(&vtm); */
	    vt = vAxis2Points(v);
	    _do_oneTick(anAx, vt, anAx->ticLenMinor);
	}
	cpdf_stroke();
    }

    /* do major ticks */
    if(anAx->ticEnableMajor) {
	cpdf_setlinewidth(anAx->tickWidthMajor);
	memcpy(&vtm, &anAx->vTL, sizeof(struct tm));
	for(v = tm_to_NumDays(&anAx->vTL, &vtm) ; v <= fndays*1.0001 ;
			v = _bump_tm_Time(&anAx->vTL, &vtm, majorBumpVar, majorBump)) {
	    /* _printfTime(&vtm); */
	    vt = vAxis2Points(v);
	    _do_oneTick(anAx, vt, anAx->ticLenMajor);
	}
	cpdf_stroke();
    }
}

void cpdf_setLinearAxisParams(CPDFaxis *anAx, float tic1ValMajor, float intervalMajor,
			float tic1ValMinor, float intervalMinor)
{
	anAx->valFirstTicLinMajor = tic1ValMajor;
	anAx->valFirstTicLinMinor = tic1ValMinor;
	anAx->ticIntervalLinMajor = intervalMajor;
	anAx->ticIntervalLinMinor = intervalMinor;
}


void cpdf_setLogAxisTickSelector(CPDFaxis *anAx, int ticselect)
{
	anAx->ticSelectorLog = ticselect;
}

void cpdf_setLogAxisNumberSelector(CPDFaxis *anAx, int numselect)
{
	anAx->numSelectorLog = numselect;
}


/* ----------------------------------------------------------------------- */
/*  Axis label fuctions */
/* ----------------------------------------------------------------------- */

void cpdf_setAxisLabel(CPDFaxis *anAx, char *labelstring, char *fontName, char *encoding, float fontSize)
{
	if(anAx->axisLabel) free(anAx->axisLabel);
	anAx->axisLabel = (char *)malloc((size_t)(strlen(labelstring) + 1));
	_cpdf_malloc_check((void *)anAx->axisLabel);
	strcpy(anAx->axisLabel, labelstring);

	if(anAx->labelFontName) free(anAx->labelFontName);
	anAx->labelFontName = (char *)malloc((size_t)(strlen(fontName) + 1));
	_cpdf_malloc_check((void *)anAx->labelFontName);
	strcpy(anAx->labelFontName, fontName);

	if(anAx->labelEncoding) free(anAx->labelEncoding);
	anAx->labelEncoding = (char *)malloc((size_t)(strlen(encoding) + 1));
	_cpdf_malloc_check((void *)anAx->labelEncoding);
	strcpy(anAx->labelEncoding, encoding);

	anAx->labelFontSize = fontSize;
}


void _do_axisLabel(CPDFaxis *anAx)
{
float startX, startY, v, vt, ang;
float slen, hslen;
	if(anAx->axisLabel == NULL) return;		/* no axis label */

	cpdf_beginText(0);
	cpdf_setFont(anAx->labelFontName, anAx->labelEncoding, anAx->labelFontSize);
	slen = cpdf_stringWidth((unsigned char *)(anAx->axisLabel));	/* number string width for centring */
	hslen = slen*0.5;
	ang = PI*anAx->angle/180.0;
	/* find the value for half point of axis to center label */
	if(anAx->type == 1)
	    v = sqrt(anAx->valH / anAx->valL) * anAx->valL;	/* log axis mid value */
	else
	    v = 0.5* (anAx->valH - anAx->valL) + anAx->valL;	/* linear axis mid value */
	vt = vAxis2Points(v);		/* X pos of the center of axis in points */
	if(anAx->horizLabel) {
	    /* number stays always horizontal */
	    if(anAx->numPosition == 0)		/* numPosition also controls label position */
		startY = numEdgeY - (anAx->numLabelGap) - (anAx->labelFontSize)*0.6;
	    else
		startY = numEdgeY + (anAx->numLabelGap) + slen;
	    startX = vt - ((anAx->labelFontSize)*0.3*sin(ang) + hslen*cos(ang));
	    cpdf_rawText(startX, startY, -(anAx->angle), anAx->axisLabel);
	}
	else {
	    /* number is parallel to axis and rotate with axis */
	    if(anAx->numPosition == 0)		/* numPosition also controls label position */
		startY = numEdgeY - (anAx->numLabelGap) - (anAx->labelFontSize)*0.6;
	    else
		startY = numEdgeY + (anAx->numLabelGap);
	    startX = vt - hslen;
	    cpdf_rawText(startX, startY, 0.0, anAx->axisLabel);
	}
	cpdf_endText();
}


/* ----------------------------------------------------------------------- */
/*  Utility fuctions */
/* ----------------------------------------------------------------------- */


float vAxis2Points(float x)
{
float xrval = 0.0;
double xvlog = 0.0, fraction=0.0;

    /* xrval = anAx2->xloc; */	/* min point value for X */
    if(anAx2->type == 0) {
	/* linear */
	xrval += (x - (anAx2->valL)) * xa2points;
    }
    else if(anAx2->type == 1) {
	/* logarithmic */
	if(x > 0.0) {
	    xvlog = log10((double)x);
	    fraction = (xvlog - xaLlog)/(xaHlog - xaLlog); /* xHlog, xLlog precomputed */
	    xrval += (anAx2->length) * fraction;
	}
    }
    else if(anAx2->type == 2) {
	/* time */
	xrval += (x - (anAx2->valL)) * xa2points;
    }
    return xrval;
}



