/* cpdfImage.c
 * Copyright (C) 1998 FastIO Systems, All Rights Reserved.
 * For conditions of use, license, and distribution, see LICENSE.txt or LICENSE.pdf.

1998-09-03 [IO]
	JPEG (baseline) image import.
-------------------------------------------------------------------------------------------
Usage Notes on cpdf_importImage(), and cpdf_rawImportImage()

   These two functions are identical except for the domain in which the coordinate
   system for (x, y) is defined.  With  cpdf_importImage(), (x, y) are defined
   in the current domain coordinate system.  Width and height should still be
   defined in points.  With cpdf_rawImportImage(), all of (x, y, width, height)
   are defined in points in the default coordinate system.

   When you import an image file, you may not know image dimensions (in pixels).
   For each X or Y dimension, specify either size (in points) or scaling factor
   and set the unspecified one to zero.
   The unspecified variable will be set upon return (that's why these parameters
   are passed by address).  If both are set, width and height will take
   precedence over scaling factors.  For example if the image file "/tmp/myimage.jpg" has
   the dimension of 150x100 pixels, a call:
   width = 0.0;
   height = 200.0;
   xscale = 0.0;
   yscale = 0.0;
   cpdf_rawImportImage("/tmp/myimage.jpg", JPEG_IMG, 100.0, 100.0, &width, &height, &xscale, &yscale, 1);

   will return with width = 300, and xscale = yscale = 2.0, while the aspect ratio is
   preserved.  If the ratio of width/height does not match the aspect ratio of the original
   image, the aspect ratio will not be preserved.

   ** Important Note:
	When the last argument is non-zero, the call brackets image import with
	a cpdf_gsave() and cpdf_grestore() pair.  If the last argument is zero,
	the function does not perform gsave/grestore internally to allow flexibility
	of having an arbitrary path clip the image or to draw image inside text
	(See Example 14.4 in the  PDF Reference Manual version 1.2).

*/


/* #define DEBUG 1 */

#include "version.h"

#include <string.h>
#include <math.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>

#include "cpdflib.h"		/* This must be included before all other local include files */
#include "cglobals.h"

#define M_SOF0  0xC0		/* Baseline JPEG */
#define M_SOF1  0xC1		/* Extended sequential, Huffman */
#define M_SOF2  0xC2		/* Progressive JPEG */



/* Same as cpdf_rawImportImage(), but (x, y) are defined in the current domain
   coordinate system.  Width and height should still be defined in points.
   If 'gsave' is non-zero, it performs gsave/grestore internally.
*/

int cpdf_importImage(char *imagefile, int type, float x, float y, float angle,
	float *width, float *height, float *xscale, float *yscale, int gsave)
{
    return( cpdf_rawImportImage(imagefile, type, x_Domain2Points(x), y_Domain2Points(y), angle,
		width, height, xscale, yscale, gsave) );
}


/* same as cpdf_importImage(), but (x,y) are in points.
*/

int cpdf_rawImportImage(char *imagefile, int type, float x, float y, float angle,
	float *width, float *height, float *xscale, float *yscale, int gsave)
{
int i;
int foundInPageList = 0;	/* in pageInfos[].imageIdx[] */
int retcode=0;
int imageIndexFound = 0;	/* in imageInfos[] */
int currentImageIndex = 0;
float a, b=0.0, c=0.0, d, e, f;
CPDFimageInfo *newImage;
char imagename[32], *image;

    /* check if this image has already been included or new image */
    if( isNewImage(imagefile, &imageIndexFound) ) {

	if(numImages >= NMAXIMAGES) {
	    fprintf(stderr, "ClibPDF: Too many images. Increase NMAXIMAGES in cpdflib.h and recompile library.\n");
	    return(1);
	}
	newImage = &imageInfos[numImages];
	currentImageIndex = numImages;
	/* check if image file actually exists */
	if((retcode = read_JPEG_header(imagefile, newImage))) {
	    if(retcode == -1) {
		return(retcode);	/* fprintf(stderr ...) done already in function */
	    }
	    if(retcode == -2) {
		fprintf(stderr, "ClibPDF: Not a JPEG file?: %s\n", imagefile);
		return(retcode);
	    }
	    if( newImage->process != M_SOF0 ) {
		fprintf(stderr, "ClibPDF: JPEG file does not use the baseline compression: %s\n", imagefile);
		if( newImage->process == M_SOF2 )
		    fprintf(stderr, "  It is a Progressive JPEG file.\n");
		fprintf(stderr, "  Please save it as a baseline JPEG.\n");
		return(-3);
	    }
	}
	if(newImage->ncomponents == 1)
	    imageFlagBCI |= 1;		/* /ImageB procset */
	else if(newImage->ncomponents > 2 )
	    imageFlagBCI |= 2;		/* /ImageC procset */
        sprintf(imagename, "IMcpdf%d", numImages);
	image = imagename;
	/* memcpy(newImage, &tmpImage, sizeof(CPDFimageInfo)); */
	newImage->name = (char *)malloc((size_t)(strlen(imagename) + 1));
	_cpdf_malloc_check((void *)newImage->name);
	newImage->filepath = (char *)malloc((size_t)(strlen(imagefile) + 1));
	_cpdf_malloc_check((void *)newImage->filepath);
	strcpy(newImage->name, imagename);
	strcpy(newImage->filepath, imagefile);
	newImage->filesize = getFileSize(imagefile);
	newImage->type = type;
	numImages++;
    }
    else {
	/* This image has been used already, and is in imageInfos[imageIndexFound] */
	image = imageInfos[imageIndexFound].name;
	currentImageIndex = imageIndexFound;
    }

    /*  Store the index of this image in the used image list in pageInfos[].
	Multiple pages can share images.
    */
    foundInPageList = 0;
    for(i=0; i < pageInfos[currentPage].npImage; i++) {
	if(pageInfos[currentPage].imageIdx[i] == currentImageIndex)
	    foundInPageList = 1;
    }
    if(!foundInPageList)
        pageInfos[currentPage].imageIdx[pageInfos[currentPage].npImage++] = currentImageIndex;

/*
    fprintf(stderr, "Image file: %s\n", imageInfos[imageIndexFound].filepath);
    fprintf(stderr, "File size: %ld\n", imageInfos[imageIndexFound].filesize);
    fprintf(stderr, "Resolution: (%d x %d)\n", imageInfos[imageIndexFound].width,
				imageInfos[imageIndexFound].height);
*/

    /* work out positioning and scaling for use with concat operator */
    e = x;
    f = y;
    if(fabs(*width) > 0.00001) {
	/* use the width value passed */
	a = *width;			/* scale unit image to this */
	*xscale = a / imageInfos[imageIndexFound].width;
    }
    else if( fabs(*xscale) > 0.00001) {
	/* width not specified, but we have xscale */
	a = *xscale * imageInfos[imageIndexFound].width;
	*width = a;
    }
    /* neither width or xscale is set, see if Y specs are present */
    else if( fabs(*height) > 0.00001) {
 	/* Nothing set for X, but height is set, so use the Y scaling factor for xscale */
	*yscale = *height / imageInfos[imageIndexFound].height;
	*xscale = *yscale;
	a = *xscale * imageInfos[imageIndexFound].width;
    }
    else if( fabs(*yscale) > 0.00001) {
	/* Nothing set for X, but y scale is set, so use that for xscale too */
	*xscale = *yscale;
	a = *xscale * imageInfos[imageIndexFound].width;
	*width = a;
    }
    else {
	/* Nothign is specified. Use image xpixel count as number of points */
	a = (float)imageInfos[imageIndexFound].width;
	*width = a;
	*xscale = 1.0;
    }

    /* NOW do the same for y scaling and height */
    if(fabs(*height) > 0.00001) {
	/* use the width value passed */
	d = *height;			/* scale unit image to this */
	*yscale = d / imageInfos[imageIndexFound].height;
    }
    else if( fabs(*yscale) > 0.00001) {
	/* height not specified, but we have yscale */
	d = *yscale * imageInfos[imageIndexFound].height;
	*height = d;
    }
    /* Nothing is set for Y, see if there is X scaling info we can use */
    else if( fabs(*width) > 0.00001) {
 	/* Nothing set for Y, but width is set, so use the X scaling factor for yscale */
	*xscale = *width / imageInfos[imageIndexFound].width;
	*yscale = *xscale;
	d = *yscale * imageInfos[imageIndexFound].height;
    }
    else if( fabs(*xscale) > 0.00001) {
	/* Nothing set for Y, but x scale is set, so use that for yscale too */
	*yscale = *xscale;
	d = *yscale * imageInfos[imageIndexFound].height;
	*height = d;
    }
    else {
	/* Really nothing is set, so use image xpixel count as number of points */
	d = (float)imageInfos[imageIndexFound].height;
	*height = d;
	*yscale = 1.0;
    }

    /* Write image use line to current Contents stream */
    if(gsave) cpdf_gsave();
    cpdf_rawTranslate(x, y);
    if(fabs(angle) > 0.001)
	cpdf_rotate(angle);

    /* fprintf(stderr, "CTM = [%g %g %g %g %g %g]\n", a, b, c, d, e, f); */
    cpdf_rawConcat(a, b, c, d, 0.0, 0.0);	/* concatename CTM */

    if(useContentMemStream) {
	sprintf(spbuf, "/%s Do\n", image);
	cpdf_writeMemoryStream(currentMemStream, spbuf, strlen(spbuf));
    }
    else
        fprintf(fpcontent, "/%s Do\n", image);
    if(gsave) cpdf_grestore();
    return(0);
}


int isNewImage(char *filepath, int *imageFound)
{
int i, isNew = 1;
    /* look in the list */
    for(i=0; i < numImages; i++) {
	if( strcmp(filepath, imageInfos[i].filepath) == 0 ) {
	    isNew = 0;		/* already defined */
	    *imageFound = i;	/* return found image index */
	    break;
	}
    }
    if(isNew)
	*imageFound = numImages;		/* index of the new image is this */
    return(isNew);
}


int  _cpdf_freeAllImageInfos(void)
{
CPDFimageInfo *timage;
int i;
    for(i=0; i< numImages; i++) {
	timage = &imageInfos[i];
	if(timage->name) {
	    free(timage->name);
	    timage->name = NULL;
	}
	if(timage->filepath) {
	    free(timage->filepath);
	    timage->filepath = NULL;
	}
    }
    return(0);
}


