/* cpdfFont.c
 * Copyright (C) 1998 FastIO Systems, All Rights Reserved.
 * For conditions of use, license, and distribution, see LICENSE.txt or LICENSE.pdf.

*/

/* #define DEBUG 1 */

#include "version.h"

#include <string.h>
#include <math.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>

#include "cpdflib.h"		/* This must be included before all other local include files */
#include "cglobals.h"

/* Current Text Matrix.  give Text Matrix concatenation behavior as Tm operator
   resets it fresh every time */

/* 2 x 3 matrix for CTM : mat mult is done by adding 0 0 1 as a column (see cpdflib.h)
typedef struct {
    float a; float b;
    float c; float d;
    float x; float y;
} CPDFctm;
*/

static CPDFctm textCTM = { 1.0, 0.0, 0.0, 1.0, 0.0, 0.0 };
static int textClipMode = 0;
static int count, bufsize;
static char *mbuff;

/* ======================================================================= */
void _cpdf_resetTextCTM(void)
{
     textCTM.a = 1.0;
     textCTM.b = 0.0;
     textCTM.c = 0.0;
     textCTM.d = 1.0;
     textCTM.x = 0.0;
     textCTM.y = 0.0;
}


void cpdf_beginText(int clipmode)
{
    cpdf_clearMemoryStream(scratchMem);
    cpdf_memPutc('\n', scratchMem);
    if(textClipMode) {
	cpdf_memPuts("q\n", scratchMem);
    }
    cpdf_memPuts("BT\n", scratchMem);
    cpdf_getMemoryBuffer(scratchMem, &mbuff, &count, &bufsize);

    if(useContentMemStream)
	cpdf_writeMemoryStream(currentMemStream, mbuff, count);
    else
	fputs(mbuff, fpcontent);

    inTextObj = 1;
    _cpdf_resetTextCTM();
}

void cpdf_endText(void)
{
    cpdf_clearMemoryStream(scratchMem);
    cpdf_memPuts("ET\n", scratchMem);
    if(textClipMode)
	cpdf_memPuts("Q\n", scratchMem);
    cpdf_getMemoryBuffer(scratchMem, &mbuff, &count, &bufsize);
    if(useContentMemStream)
	cpdf_writeMemoryStream(currentMemStream, mbuff, count);
    else
	fputs(mbuff, fpcontent);

    inTextObj = 0;
    textClipMode = 0;
}

/* Position and draw text in current domain using its coordinate system.
   This function will set a new Text CTM.
   Lower left corner of text string is place at (x, y).
*/
void cpdf_text(float x, float y, float orientation, char *textstr)
{
float xc, yc;
    xc = x_Domain2Points(x);
    yc = y_Domain2Points(y);
    cpdf_rawText(xc, yc, orientation, textstr);
}

/* Position  and draw text in raw domain in points with rotation.
   This function will set a new Text CTM.
   Lower left corner of text string is place at (x, y).
 */
void cpdf_rawText(float x, float y, float orientation, char *textstr)
{
float a, b, c, d, angle, vcos, vsin;

    angle = PI*orientation/180.0;
    vcos = cos(angle);
    vsin = sin(angle);
    a =  vcos;
    b =  vsin;
    c = -vsin;
    d =  vcos;
    cpdf_setTextMatrix(a, b, c, d, x, y);
    cpdf_textShow(textstr);
}


/* Position  and draw text in raw domain in points with rotation.
   This function will set a new Text CTM.
   Text string is placed at (x, y) with various justification mode.

#define	TEXTPOS_LL	0
#define	TEXTPOS_LM	1
#define	TEXTPOS_LR	2
#define	TEXTPOS_ML	3
#define	TEXTPOS_MM	4
#define	TEXTPOS_MR	5
#define	TEXTPOS_UL	6
#define	TEXTPOS_UM	7
#define	TEXTPOS_UR	8
 */

void cpdf_textAligned(float x, float y, float orientation, int centmode, char *textstr)
{
    cpdf_rawTextAligned(x_Domain2Points(x), y_Domain2Points(y), orientation, centmode, textstr);
}

void cpdf_rawTextAligned(float x, float y, float orientation, int centmode, char *textstr)
{
float a, b, c, d, angle, vcos, vsin;
float wd=0.0, hd=0.0, swidth, sheight;
float xc, yc;
int xcent, ycent;
    angle = PI*orientation/180.0;
    vcos = cos(angle);
    vsin = sin(angle);

    xcent = centmode % 3;	/* 0=left, 1=middle, 2=right */
    ycent = centmode / 3;	/* 0=lower,  1=middle, 2=upper */
    swidth = cpdf_stringWidth((unsigned char *)textstr);	/* string width according to current font */
    sheight = FONTSIZE2HEIGHT * font_size;		/* current font size */
    /* necessary offset along the string's width dimension */
    if(xcent == 2)
	wd = swidth;
    else if(xcent == 1)
	wd = 0.5 * swidth;
    else
	wd = 0.0;
    /* necessary offset along the string's height dimension */
    if(ycent == 2)
	hd = sheight;
    else if(ycent == 1)
	hd = 0.5 * sheight;
    else
	hd = 0.0;

    xc = x - (wd*vcos - hd*vsin);	/* correction to the X position in unroted coord */
    yc = y - (wd*vsin + hd*vcos);
    a =  vcos;
    b =  vsin;
    c = -vsin;
    d =  vcos;
    cpdf_setTextMatrix(a, b, c, d, xc, yc);
    cpdf_textShow(textstr);
}

void cpdf_setWordSpacing(float spacing)
{
    word_spacing = spacing;
    if(useContentMemStream) {
	sprintf(spbuf, "%.3f Tw\n", spacing);
	cpdf_writeMemoryStream(currentMemStream, spbuf, strlen(spbuf));
    }
    else
	fprintf(fpcontent, "%.3f Tw\n", spacing);
}


void cpdf_setCharacterSpacing(float spacing)
{
    char_spacing = spacing;
    if(useContentMemStream) {
	sprintf(spbuf, "%.3f Tc\n", spacing);
	cpdf_writeMemoryStream(currentMemStream, spbuf, strlen(spbuf));
    }
    else
	fprintf(fpcontent, "%.3f Tc\n", spacing);
}

void cpdf_setHorizontalScaling(float scale)
{
    horiz_scaling = scale;
    if(useContentMemStream) {
	sprintf(spbuf, "%.2f Tz\n", scale);
	cpdf_writeMemoryStream(currentMemStream, spbuf, strlen(spbuf));
    }
    else
	fprintf(fpcontent, "%.2f Tz\n", scale);
}

void cpdf_setTextLeading(float leading)
{
    text_leading = leading;
    if(useContentMemStream) {
	sprintf(spbuf, "%.2f TL\n", leading);
	cpdf_writeMemoryStream(currentMemStream, spbuf, strlen(spbuf));
    }
    else
	fprintf(fpcontent, "%.2f TL\n", leading);
}

/* Only in Text mode */
void cpdf_setTextMatrix(float a, float b, float c, float d, float x, float y)
{
    if(useContentMemStream) {
	sprintf(spbuf, "%.4f %.4f %.4f %.4f %.4f %.4f Tm\n", a, b, c, d, x, y);
	cpdf_writeMemoryStream(currentMemStream, spbuf, strlen(spbuf));
    }
    else
	fprintf(fpcontent, "%.4f %.4f %.4f %.4f %.4f %.4f Tm\n", a, b, c, d, x, y);

     textCTM.a = a;
     textCTM.b = b;
     textCTM.c = c;
     textCTM.d = d;
     textCTM.x = x;
     textCTM.y = y;
}

/* We keep a concatenated Text CTM in C, update it and set it using "Tm" operator.
*/
void cpdf_concatTextMatrix(float a, float b, float c, float d, float x, float y)
{
CPDFctm  sCTM;
    sCTM.a = a;
    sCTM.b = b;
    sCTM.c = c;
    sCTM.d = d;
    sCTM.x = x;
    sCTM.y = y;
    multiplyCTM(&textCTM, &sCTM);
    cpdf_setTextMatrix(textCTM.a, textCTM.b, textCTM.c, textCTM.d, textCTM.x, textCTM.y);
}

/* Only in Text mode */
void cpdf_rotateText(float degrees)
{
float a, b, c, d, e, f, angle, vcos, vsin;

    angle = PI*degrees/180.0;
    vcos = cos(angle);
    vsin = sin(angle);
    a =  vcos;
    b =  vsin;
    c = -vsin;
    d =  vcos;
    e = 0.0; f = 0.0;
    cpdf_concatTextMatrix(a, b, c, d, e, f);
}

/* Only in Text mode */
void cpdf_skewText(float alpha, float beta)
{
float a, b, c, d, e, f;
    a =  1.0;
    b =  tan(PI*alpha/180.0);
    c =  tan(PI*beta/180.0);
    d =  1.0;
    e = 0.0; f = 0.0;
    cpdf_concatTextMatrix(a, b, c, d, e, f);
}

/* Only in Text mode */
void cpdf_setTextPosition(float x, float y)
{
    cpdf_rawSetTextPosition(x_Domain2Points(x), y_Domain2Points(y));
}

void cpdf_rawSetTextPosition(float x, float y)
{
float a, b, c, d, e, f;

    a =  1.0;		/* Hmm, should I use textCTM.a here???? */
    b =  0.0;
    c =  0.0;
    d =  1.0;
    e = x; f = y;
    cpdf_setTextMatrix(a, b, c, d, e, f);
}

/* In order to change Text Rendering Mode, you must first end text, and
   begin new text.
#define	TEXT_FILL		0
#define	TEXT_STROKE		1
#define	TEXT_FILL_STROKE	2
#define	TEXT_INVISIBLE		3
#define	TEXT_FILL_CLIP		4
#define	TEXT_STROKE_CLIP	5
#define	TEXT_FILL_STROKE_CLIP	6
#define	TEXT_CLIP		7

*/
void cpdf_setTextRenderingMode(int mode)
{
    if(useContentMemStream) {
	sprintf(spbuf, "%d Tr\n", mode);
	cpdf_writeMemoryStream(currentMemStream, spbuf, strlen(spbuf));
    }
    else
	fprintf(fpcontent, "%d Tr\n", mode);
}

void cpdf_setTextRise(float rise)
{
    text_rise = rise;
    if(useContentMemStream) {
	sprintf(spbuf, "%.3f Ts\n", rise);
	cpdf_writeMemoryStream(currentMemStream, spbuf, strlen(spbuf));
    }
    else
	fprintf(fpcontent, "%.3f Ts\n", rise);
}

/* This sends Td operator: only in Text mode */
void cpdf_rawSetNextTextLineOffset(float x, float y)
{
    if(useContentMemStream) {
	sprintf(spbuf, "%.3f %.3f Td\n", x, y);
	cpdf_writeMemoryStream(currentMemStream, spbuf, strlen(spbuf));
    }
    else
	fprintf(fpcontent, "%.3f %.3f Td\n", x, y);
}

/* Only in Text mode */
void cpdf_setNextTextLineOffset(float x, float y)
{
    cpdf_rawSetNextTextLineOffset(x_Domain2Points(x), y_Domain2Points(y));
}


/* This sends a T* operator: only in Text mode */
void cpdf_textCRLF()
{
    if(useContentMemStream)
	cpdf_memPuts("T*\n", currentMemStream);
    else
	fprintf(fpcontent, "T*\n");
}

/* Tj operator: only in Text mode */
void cpdf_textShow(char *txtstr)
{
char *fixedstr;
    fixedstr = cpdf_escapeSpecialChars(txtstr);
    if(useContentMemStream) {
	sprintf(spbuf, "(%s) Tj\n", fixedstr);
	cpdf_writeMemoryStream(currentMemStream, spbuf, strlen(spbuf));
    }
    else
	fprintf(fpcontent, "(%s) Tj\n", fixedstr);
    free(fixedstr);		/* MUST */
}


/* ' operator: only in Text mode */
void cpdf_textCRLFshow(char *txtstr)
{
char *fixedstr;
    fixedstr = cpdf_escapeSpecialChars(txtstr);
    if(useContentMemStream) {
	sprintf(spbuf, "(%s) '\n", fixedstr);
	cpdf_writeMemoryStream(currentMemStream, spbuf, strlen(spbuf));
    }
    else
	fprintf(fpcontent, "(%s) '\n", fixedstr);
    free(fixedstr);		/* MUST */
}

/* This function will return a new string (with malloced memory)
   that contains appropriately escaped version of the input string.
   It will escape '\', '(', and ')'.
   You must FREE the returned string after use.
   Otherwise, memory leak will result.
*/

char *cpdf_escapeSpecialChars(char *instr)
{
char *ptr, *ptr2, *buf, ch;
int escapecount = 0;
	ptr = instr;
	while( (ch = *ptr++) != '\0') {
	    if( (ch == '(') || (ch == ')') || (ch == '\\') ) escapecount++;
	}

	ptr = instr;
	buf = (char *)malloc((size_t)(strlen(instr) + escapecount + 16));
	_cpdf_malloc_check((void *)buf);
	ptr2 = buf;
	while( (ch = *ptr++) != '\0') {
	    if( (ch == '\\') || (ch == '(') || (ch == ')') ) {
		*ptr2++ = '\\';
		*ptr2++ = ch;
	    }
	    else
		*ptr2++ = ch;
	}
	*ptr2 = '\0';
	return(buf);
}

