/*
 File:       MagnifyingView.m

 Contains:   Source code for the MagnifyingView class

 Written by: Eric Simenel

 Created:    July 1997

 Copyright:  (c)1997 by Apple Computer, Inc., all rights reserved.

 Change History (most recent first):

 You may incorporate this sample code into your applications without
 restriction, though the sample code has been provided "AS IS" and the
 responsibility for its operation is 100% yours.  However, what you are
 not permitted to do is to redistribute the source as "DSC Sample Code"
 after having made changes. If you're going to re-distribute the source,
 we require that you make it clear in the source that the code was
 descended from Apple Sample Code, but that you've made changes.
*/

#import "MagnifyingView.h"
#import "MagnifyingGlassController.h"
#import <AppKit/psopsNeXT.h>

@implementation MagnifyingView

- (id)initWithFrame:(NSRect)rect
{
    if (self = [super initWithFrame:rect])
      {
        // grabwindow is a ghost window which we move around just to be able to steal the screen bits it covers
        if (!grabWindow)
          {
            // use Nonretained window so we can steal the screen bits
            grabWindow = [[NSWindow alloc] initWithContentRect:NSZeroRect
                                                     styleMask:NSBorderlessWindowMask
                                                       backing:NSBackingStoreNonretained
                                                         defer:NO
                                                        screen:nil];
            if (!grabWindow) return nil;
            [grabWindow setReleasedWhenClosed:NO];
            [grabWindow disableCursorRects];
            // set the level to high value so that we stay on top of everything else
            PSsetwindowlevel(NSPopUpMenuWindowLevel+100, [grabWindow windowNumber]);
          }
      }
    return self;
}

- (void)dealloc
{
    if (grabWindow) [grabWindow release];
    if (screenBitmap) [screenBitmap release];
}

- (void)drawRect:(NSRect)rect
{
    NSRect frameRect = [self frame];
    frameRect.origin.x = frameRect.origin.y = 0.0;
    // first draw the fatbits...
    [screenBitmap drawInRect:frameRect];
    // then a first reticule frame in black
    frameRect.origin.x = frameRect.origin.y = 63.0;
    frameRect.size.width = frameRect.size.height = 10.0;
    NSFrameRect(frameRect);
    // and another reticule frame in white so that we can see the
    // reticule no matter what the magnified view contains.
    frameRect.origin.x = frameRect.origin.y = 64.0;
    frameRect.size.width = frameRect.size.height = 8.0;
    [[NSColor whiteColor] set];
    NSFrameRect(frameRect);
}

// we answer YES to the 3 messages below because that's how we'll get
// the mouseMoved event
- (BOOL)acceptsFirstResponder
{
    return YES;
}

- (BOOL)becomeFirstResponder
{
    [[self window] setAcceptsMouseMovedEvents: YES];
    return YES;
}

- (BOOL)acceptsFirstMouse:(NSEvent *)theEvent
{
    return YES;
}

// sent to the view which got mouseDown:
- (void)mouseUp:(NSEvent *)theEvent
{
    // need to reprime; it seems to get shut off on mouseDown:
    [[self window] setAcceptsMouseMovedEvents: YES];
}

// sent to the first responder
// not in the window's event mask by default
- (void)mouseMoved:(NSEvent *)theEvent
{
    int i, found = 0, screenCount;
    NSPoint currentMouseLocation;
    NSRect srcRect, screenRect;
    NSArray *screens;
    NSView *grabbedContentView;

    if (![self window]) return;
    if (![[self window] isVisible]) return;
    if ([magnifyingGlassController imageIsLocked]) return;

    currentMouseLocation = [[self window] mouseLocationOutsideOfEventStream];
    currentMouseLocation = [[self window] convertBaseToScreen:currentMouseLocation];

    // weird bug ????? The y coordinate is 1.0 too high...
    currentMouseLocation.y -= 1.0;

    [magnifyingGlassController setGlobalLocation:currentMouseLocation];

    // in which screen is the cursor?
    screens = [NSScreen screens];
    screenCount = [screens count];
    for (i = 0; (i < screenCount) && !found; i++)
      {
        NSScreen *thisScreen = [screens objectAtIndex:i];
        screenRect = [thisScreen frame];
        found = NSPointInRect(currentMouseLocation, screenRect);
      }
    if (!found)
      {
        NSLog(@"no screen found !!!!");
        return; // something weird has happened...
      }

    srcRect.origin = currentMouseLocation;
    srcRect.size = [self frame].size;
    srcRect.size.width /= 8;
    srcRect.size.height /= 8;
    srcRect = NSOffsetRect(srcRect, -8.0, -8.0);

    // the rectangle we magnify has to belong entirely to the screen where the cursor is
    if (!NSContainsRect(screenRect, srcRect))
      {
        if (srcRect.origin.x < screenRect.origin.x) srcRect.origin.x = screenRect.origin.x;
        if (srcRect.origin.y < screenRect.origin.y) srcRect.origin.y = screenRect.origin.y;
        if ((srcRect.origin.x + srcRect.size.width) >= (screenRect.origin.x + screenRect.size.width))
            srcRect.origin.x = screenRect.origin.x + screenRect.size.width - srcRect.size.width - 1.0;
        if ((srcRect.origin.y + srcRect.size.height) >= (screenRect.origin.y + screenRect.size.height))
            srcRect.origin.y = screenRect.origin.y + screenRect.size.height - srcRect.size.height - 1.0;
      }

    // is this rectangle the same we displayed previously?
    if (NSEqualRects(srcRect, lastScreenBitmapRect)) return;

    lastScreenBitmapRect = srcRect;
    grabbedContentView = [grabWindow contentView];
    // but the user never sees it, of course
    [grabWindow setFrame:srcRect display:NO];
    // we don't want the window to be filled so it stays transparent
    PSsetautofill(NO, [grabWindow windowNumber]);
    // we move it above all other windows
    [grabWindow orderFrontRegardless];
    // we prepare the graphic context
    [grabbedContentView lockFocus];
    srcRect.origin.x = srcRect.origin.y = 0.0;
    if (screenBitmap) [screenBitmap release];
    // get the bits in our offscreen
    screenBitmap = [[NSBitmapImageRep alloc] initWithFocusedViewRect:srcRect];
    [grabbedContentView unlockFocus];
    // and we remove the window from the screen to prevent undesirable effects
    [grabWindow orderOut:nil];

    // and update the view
    [self setNeedsDisplay:YES];
}

@end
