/********************************************************************************

File:       CustomGridView.m

This is a simple view that just draws a grid of rectangles, with text inside them.
It demonstrates how to provide pagination hints within a printable view, by only
breaking pages on a grid boundary.

The coordinate system of the custom view is flipped, primarily because the NSString
drawing APIs only work correctly in a flipped view.  Without the text drawing, there
is no reason why the coordinate system couldn't be used normally.

This sample is based on a NeXTStep sample written by Samuel Streeper.


Written by: Timothy Carroll

Created:    Wed 15-Oct-1997

Copyright:  (c) 1991, 1997 by Apple Computer, Inc., all rights reserved.

Change History (most recent first):

You may incorporate this sample code into your applications without
restriction, though the sample code has been provided "AS IS" and the
responsibility for its operation is 100% yours.  However, what you are
not permitted to do is to redistribute the source as "DSC Sample Code"
after having made changes. If you're going to re-distribute the source,
we require that you make it clear in the source that the code was
descended from Apple Sample Code, but that you've made changes.
********************************************************************************/


#import "CustomGridView.h"
#import "CustomPrintInfo.h"

// These constants define the layout for the page
enum
{
	kPixelsPerInch = 72,
	kPageWidth = (8 * kPixelsPerInch),
	kPageHeight = (10 * kPixelsPerInch),
	kGridWidth = (kPixelsPerInch),
	kGridHeight = (kPixelsPerInch),
	kMaxColumns = 3,
	kMaxRows = 4,
	kTotalPages = (kMaxColumns * kMaxRows)
};

#define min(X, Y)  ((X) < (Y) ? (X) : (Y))

@implementation CustomGridView
// initializer
- (id) initWithFrame: (NSRect) frameRect
{
// Normally, you'll always want to initialize using whatever frame
// was passed into the routine.  Because we're using a fixed size
// grid, we override passed in rect and use one for the entire grid.
// You probably shouldn't do this in real code.
	NSRect r;

	r.origin.x = 0;
	r.origin.y = 0;
        r.size.width = (kMaxColumns*kPageWidth);
	r.size.height = (kMaxRows*kPageHeight);

	[super initWithFrame:r];

	return self;

}

// We override to provide a flipped coordinate system
- (BOOL) isFlipped
{
	return YES;
}

- (void) awakeFromNib
{
	NSPoint	tempPt;
// Scroll the view so that the top,left page is showing, and
// then bring it to the front.	

	tempPt.x = 0;
	tempPt.y = 0;
	
	[self scrollPoint: tempPt];	
	[[self window] makeKeyAndOrderFront: self];
}


// drawing functions
- (void) drawRect: (NSRect) aRect
{
	int	minColumn, maxColumn,minRow,maxRow, col, row;
	int	flag;
	int	number;
	NSPoint	point;

	// calculate the minimum number of pages to draw to fill the rect. 
	minColumn =  aRect.origin.x / kPageWidth;
	minRow =  aRect.origin.y / kPageHeight;

	// Subtract 1 point from each size, so that a full size rectangle
	// results in one page, not four.
	maxColumn =  (aRect.origin.x + aRect.size.width - 1.0) / kPageWidth;
	maxRow =  (aRect.origin.y + aRect.size.height - 1.0) / kPageWidth;

	NSRectClip(aRect);

	for (col = minColumn; col <= maxColumn; col++)
	{
               for (row = minRow; row <= maxRow; row++)
		{
			number = row*kMaxColumns + col + 1;
			flag = ((row + col) & 1);
			point.x = col*kPageWidth;
			point.y = row*kPageHeight;

			[self drawGridPageAt: point 
			          withNumber: number 
				    darkFlag: flag];
		}
	}
}


- (void) drawGridPageAt: (NSPoint) point
	     withNumber: (int) number
	       darkFlag: (int) flag
{
	int			loop;
	NSRect			rect;
	NSPoint			linePoint;
	NSBezierPath		*path;
	NSMutableDictionary	*stringAttributes;
	NSString		*numString;
	NSDPSContext		*context;
	id			printInfo;

	context = [NSDPSContext currentContext];
	printInfo = [NSPrintInfo sharedPrintInfo];

// Save the current postscript state, since we're going to change
// a few settings.
	PSgsave();

// Set the origin of the postscript world to the top left corner of the page.

	PStranslate (point.x, point.y);

// If we're printing with application controlled printing, we want to scale
// the 8x10 page to fit inside the margins.
	if (context && ![context isDrawingToScreen])
	{
		if ([printInfo paginationMode] == kPaginationAppControlled)
		{
			float pageScale = [self scale];
			PSscale (pageScale, pageScale);
		}
	}

// Fill the outside rectangle. This rectangle's color varies based on the flag.
	rect.origin.x = rect.origin.y = 0;
	rect.size.height = kPageHeight;
	rect.size.width = kPageWidth;

	if (flag)
	{
		[[NSColor colorWithDeviceRed: 0.4 green: 0.0 blue: 0.0 alpha: 1.0] set];
	}
	else
	{
		[[NSColor colorWithDeviceRed: 0.0 green: 0.7 blue: 0.0 alpha: 1.0] set];
	}
	[NSBezierPath fillRect: rect];

// Fill the inside rectangle, inset a half inch on all borders
	rect.origin.x = rect.origin.y =  kPixelsPerInch/2; 
	rect.size.height = kPageHeight-kPixelsPerInch;
	rect.size.width = kPageWidth-kPixelsPerInch;

	[[NSColor colorWithDeviceRed: 1 green: 1 blue: 0.6 alpha: 1.0] set];
	[NSBezierPath fillRect: rect];

// Define the path for the Grid's lines.  This path is always the same,
// so this could probably be cached somewhere is speed were a problem.

	path = [NSBezierPath bezierPath];
	// the vertical lines
	for (loop = kGridWidth; loop < kPageWidth; loop += kGridWidth)
	{
		linePoint.x = loop;
		linePoint.y = 0;
		[path moveToPoint:linePoint];
		linePoint.y += kPageHeight;
		[path lineToPoint:linePoint];
	}
	// the horizontal lines
	for (loop = kGridHeight; loop < kPageHeight; loop += kGridHeight)
	{
		linePoint.x = 0;
		linePoint.y = loop;
		[path moveToPoint:linePoint];
		linePoint.x += kPageWidth;
		[path lineToPoint:linePoint];
	}

// Draw the path
	[[NSColor colorWithDeviceRed: 0.4 green: 0.4 blue: 1 alpha: 1.0] set];
	[NSBezierPath setLineWidth : 2.0];
	[path stroke];

// Draw the page number.  We calculate the size of the rectangle
// required for the text, and center this inside the page.
	numString = [NSString stringWithFormat: @"%d",number];

	stringAttributes = [NSMutableDictionary dictionaryWithCapacity: 2];
	[stringAttributes setObject: [NSFont fontWithName: @"Times-Bold" size: 400]
			     forKey: NSFontAttributeName];
	[stringAttributes setObject: [NSColor colorWithDeviceRed: 0 green: 0.0 
							    blue: 0.2 alpha: 1.0]
			     forKey: NSForegroundColorAttributeName];
	
	rect.size = [numString sizeWithAttributes: stringAttributes];

	rect.origin.x = (kPageWidth - rect.size.width) / 2;
	rect.origin.y = (kPageHeight- rect.size.height) /2;

	[numString drawInRect: rect withAttributes: stringAttributes];

	PSgrestore();

}

// Printing Functions


// knowsPagesFirst:last should ruturn yes if it wants to control pagination, and
// it should return the page values.

// For our method, we only override AppKit if the user has selected "app controlled"
// from our PageLayoutPanel.  In our case, we already know exactly how many pages
// are going to be printed.  In a real application, you'd calculate the number of
// pages and return them.
- (BOOL) knowsPagesFirst: (int *) firstPageNum last: (int *) lastPageNum
{
	id printInfo = [NSPrintInfo sharedPrintInfo];

	if ([printInfo paginationMode] == kPaginationAppControlled)
	{
		*firstPageNum = 1;
		*lastPageNum = kTotalPages;
		return YES;
	}
	else
	{
		return NO;
	}
}


// if knowsPagesFirst:last returned YES, then rectForPage will be called once
// for each printed page.  You need to calculate the Rect inside the view that
// encloses that page, and return it.
- (NSRect) rectForPage: (int) page
{

	if (page < 1 || page > kTotalPages)
	{
		return NSZeroRect;
	}
	else
	{
		int row, col;
        	NSRect pageRect;

	// determine the row and column of the page, and use this to calculate the pageRect;
		page--;
		row = page/kMaxColumns;
		col = page % kMaxColumns;

		pageRect.origin.x = col * kPageWidth;
		pageRect.origin.y = row * kPageHeight;
		pageRect.size.width = kPageWidth * [self scale];
		pageRect.size.height = kPageHeight * [self scale];

		return pageRect;
	}
}

/*
- (void) addToPageSetup
{

}

- (void) endPrologue
{

}
*/

// If knowsPagesFirst:last returned NO, then AppKit will paginate your document
// automatically.  The next two methods are called when AppKit is paginating
// your document, to give you the opportunity to adjust the printing rectangles.
// This allows the app to adjust the printing rectangles slightly  -- for example,
// to prevent a line of text from being split across two pages.

// In our app, we adjust the pages if the user has asked PageLab to help adjust
// the pages.

- (void) adjustPageHeightNew:(float *)newBottom
       top:(float)oldTop
       bottom:(float)proposedBottom
       limit:(float)bottomLimit
{
	id printInfo = [NSPrintInfo sharedPrintInfo];
	if ([printInfo paginationMode] == kPaginationAutoWithHelp)
        {
                *newBottom = (floor(proposedBottom/kGridHeight)) * kGridHeight;
       		if (*newBottom > proposedBottom || *newBottom < bottomLimit)
                	*newBottom = proposedBottom;
        }
        else
        {
        	[super adjustPageHeightNew:newBottom top:oldTop bottom:proposedBottom
                        limit:bottomLimit];
        }
}

- (void) adjustPageWidthNew:(float *)newRight
       left:(float)oldLeft
       right:(float)proposedRight
       limit:(float)rightLimit
{
	id printInfo = [NSPrintInfo sharedPrintInfo];
	if ([printInfo paginationMode] == kPaginationAutoWithHelp)
        {
        	*newRight = (floor(proposedRight/kGridWidth)) * kGridWidth;
        	if (*newRight > proposedRight || *newRight < rightLimit)
        		*newRight = proposedRight;
        }
        else
        {
        	[super adjustPageWidthNew:newRight left:oldLeft right:proposedRight
                        limit:rightLimit];
        }
}

// Calculates a scaling factor required to draw one full grid page inside
// the current margins.

// This only should be called when the application controls printing.
- (float) scale
{
	id printInfo = [NSPrintInfo sharedPrintInfo]; 
	float l,r,t,b;
	float horScale, vertScale, minScale;
	NSSize size;

	l = [printInfo leftMargin];
	r = [printInfo rightMargin];
	t = [printInfo topMargin];
	b = [printInfo bottomMargin];
	size = [printInfo paperSize];

// calculate scaling factor so that rect will fit within margins
	horScale = (size.width - r - l)/kPageWidth;
	vertScale = (size.height - t - b)/kPageHeight;
	minScale = min(horScale, vertScale);

// Too small or too large, just use the default page size.
	if ((minScale <= 0) || (minScale > 1))
		minScale = 1;

	return minScale;
}

@end
