/*
    File:       FinderListModeController.m

    Contains:   Implementation for FinderListModeController, a datasource for an
                NSOutlineView object that interfaces with FileInfo objects to
                provide information about the directory structure of the user's 
                file system.  Information is displayed by a NSOutlineView in a
                manner similar to the List View in the Macintosh Finder.
 
    Written by: Andy Wildenberg

    Created:    9 July 1997

    Copyright:  (c)1997 by Apple Computer, Inc., all rights reserved.

    Change History:
       version 1.0: first public version

    You may incorporate this sample code into your applications without
    restriction, though the sample code has been provided "AS IS" and the
    responsibility for its operation is 100% yours.  However, what you are
    not permitted to do is to redistribute the source as "DTS Sample Code"
    after having made changes. If you're going to re-distribute the source,
    we require that you make it clear in the source that the code was
    descended from Apple Sample Code, but that you've made changes.
*/
#import "FinderListModeController.h"

@implementation FinderListModeController

// routines to implement the informal protocol of the data source of the outline view.

// The NSOutlineView pretty much assumes that you have a seperate item for each row in
// the outlineView, each of which may or may not have children.  This program implements
// its model exactly to that specification: there is one FileInfo for each of the rows in
// the outlineView, each of which may or may not have children



// This routine asks for the object corresponding to the indexth child of item.  If
// the object is at the top level (i.e. the files contained in your home directory), the
// outlineView passes an item==nil, so that has to be special cased.  In this case, there
// is a single FileInfo corresponding to the home directory that contains all of the
// top level FileInfos.  

- (id)outlineView:(NSOutlineView *)outlineView child:(int)index ofItem:(id)item
{
    if (item == nil)  //it's the topmost level
        return [[topFileInfo containedFiles] objectAtIndex: index];
    else
        return [[item containedFiles] objectAtIndex: index];
}

// This routine checks to see whether a given item is expandable (i.e. needs a little 
// spinning triangle thingy.  In this case, it does if and only if it is a directory.  
// Of course that doesn't mean that there are necessarily any subitems, since it should
// be possible to open an empty directory even though there's nothing inside.
- (BOOL)outlineView:(NSOutlineView *)outlineView isItemExpandable: (id)item
{
    return ([item filetype] == NSFileTypeDirectory);
}


// outlineView:numberOfChildrenOfItem: performs a query to see how many children a
// particular item has.  Presumably the item has already been tested by 
// outlineView:isItemExpandable to make sure it can be opened in the first place.
// Again, an item==nil means that it's the top-level (see outlineView:child:ofItem:)
- (int)outlineView:(NSOutlineView *)outlineView numberOfChildrenOfItem:(id)item
{
    if (item == nil)
        return [[topFileInfo containedFiles] count];
    else
        return [[item containedFiles] count];
}


// now that it's determined what the right object for a given row is, the NSOutlineView
// will ask what the appropriate data it should stick into each of the columns for that
// item.  In this case, the NSTableColumns have been setup in Interface Builder to have
// identifiers that correspond exactly to the correct accessor methods of the FileInfo
// objects.  Using the power of the Objective C runtime, these NSStrings are converted into
// selectors (similar to pointers to virtual functions in C++), which are then called on
// the items to return the correct information.  
- (id)outlineView:(NSOutlineView *)outlineView objectValueForTableColumn:(NSTableColumn *)tableColumn byItem:(id)item
{
    return [item performSelector: NSSelectorFromString([tableColumn identifier])];
}


// this routine creates the topmost FileInfo, i.e. the one that contains information about
// the topmost displayed directory.  Any time item==nil in a call to 
// outlineView:child:ofItem: or outlineView:numberOfChildrenOfItem:, the controller
// substitues makes the topFileInfo object the item and continues from there.
- (void)createTopFileInfo:(NSString*)topPath
{
    topFileInfo = [[FileInfo alloc] initWithPath: topPath];
}


// awakeFromNib does some basic setup, such as creating the top FileInfo object and setting 
// the title of the window to the current directory.  It is called immediately after all the objects
// have been unarchived from the nib file.
- (void)awakeFromNib
{
    NSString* currentPath = NSHomeDirectory(); // a C function to get the home directory of the current user

    // update the title of the window to be the right one
    [finderListWindow setTitle: [@"Directory " stringByAppendingString: currentPath]];

    // read the file info in the first time
    // it's important that we do this before we show the window
    [self createTopFileInfo: currentPath]; 

    // bring the main window to the front
    [finderListWindow makeKeyAndOrderFront:self];
}


// windowShouldClose is a delegate method of the NSWindow.  Since there is only one real
// window in the application, and there's no provision to open another one, this
// makes sure the application quits when the window is closed.  Note that it is only
// the delegate of the main window, not the About Box, so the application won't quit 
// when you close the about box.
- (BOOL)windowShouldClose:(id)sender
{
    // if we close the window, then we should quit the application.  This is also possible through the use of
    // - (BOOL)applicationShouldTerminateAfterLastWindowClosed:(NSApplication *)theApplication
    // but this controller is for the window not the application, so it makes more sense if it is the
    // delegate of the window instead of the application.

    [NSApp terminate: self];
    return YES;
}

// dealloc is called by the Objective-C runtime when self is no longer retained by any object.
// At this point it is necessary to free all the instance variables self has retained or copied.
- (void)dealloc
{
    // don't need to release finderListWindow since we never retained it.
    // in general, we probably should have, but since it was only used in awakeFromNib, it wasn't necessary.
    [topFileInfo release];
}
@end
