/**
 * OpenAmp Console Controller
 *
 * Wilfredo Sanchez | wsanchez@apple.com
 * September 26, 1998
 **/

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <SoundKit/Sound.h>

#import "Extentions.h"
#import "AudioFile.h"
#import "PlayList.h"
#import "SoundDecoder.h"
#import "ConsoleController.h"

#ifndef DEBUG
#define DEBUG 0
#endif

@interface ConsoleController (Private)

// UI Controls
- (void) _updatePlayControls;

@end

#if DEBUG
#define MYLOCK_LOCK   [myLock lock  ]; NSLog(@"%s Locked console"  , __PRETTY_FUNCTION__);
#define MYLOCK_UNLOCK [myLock unlock]; NSLog(@"%s Unlocked console", __PRETTY_FUNCTION__);
#else
#define MYLOCK_LOCK   [myLock lock  ];
#define MYLOCK_UNLOCK [myLock unlock];
#endif

@implementation ConsoleController

/**
 * Inits
 **/

- (id) init
{
    if ((self = [super init]))
      {
        if (! [NSBundle loadNibNamed:@"Console" owner:self] ||
            ! window              ||
            ! volumeSlider        ||
            ! balanceSlider       ||
            ! repeatButton        ||
            ! shuffleButton       ||
            ! previousTrackButton ||
            ! nextTrackButton     ||
            ! playButton          ||
            ! pauseButton         ||
            ! stopButton          ||
            ! progressBar         ||
            ! timeDisplay         ||
            ! totalTimeDisplay    ||
            ! titleDisplay        )
          {
            NSLog(@"Failed to open console nib.");
            [self release];
            return nil;
          }

        // Init iVars
        myDecoder      = [[SoundDecoder alloc] initWithOwner: self];
        myPlayList     = nil;
        myState        = STOPPED;
        myLock         = [[NSLock alloc] init];

        // Init audio
        [self volumeChanged: self];

        // Init window
        [window makeKeyAndOrderFront: self];
        [self _updatePlayControls];
      }
    return self;
}

- (void) dealloc
{
    [ myDecoder  release ];
    [ myPlayList release ];
    [ myLock     release ];
    [ window     release ]; // I own the window

    [super dealloc];
}

/**
 * Accessors
 **/

- (PlayList*) playList { return myPlayList; }
- (PlayState) state    { return myState   ; }

- (void) setPlayList: (PlayList*) aValue
{
    MYLOCK_LOCK;

    setAndRetain(myPlayList, aValue);

    [self _updatePlayControls];

    MYLOCK_UNLOCK;
}

- (AudioFile*) currentTrack
{
    if ([shuffleButton state] == NSOffState)
        return [myPlayList selectedAudioFile];
    else
      {
        NSLog(@"Current track unimplemented for shuffle play.");
        return nil;
      }
}

- (void) setProgressIsIndeterminate: (BOOL) anIndeterminateFlag
{
    [progressBar setIndeterminate: anIndeterminateFlag];
}

/**
 * Actions
 **/

- (AudioFile*) nextTrack
{
    if ([shuffleButton state] == NSOffState)
      {
        AudioFile* aTrack = [myPlayList selectNextAudioFile];

        if (!aTrack && [repeatButton state] == NSOnState)
            aTrack = [myPlayList selectFirstAudioFile];

        return aTrack;
      }
    else
      {
        NSLog(@"Next track unimplemented for shuffle play.");
        return nil;
      }
}

- (AudioFile*) previousTrack
{
    if ([shuffleButton state] == NSOffState)
        return [myPlayList selectPreviousAudioFile];
    else
        return nil;
}

- (void) updateProgressWithValue: (double) aValue
                             max: (double) aTotal
{
    if ([progressBar isIndeterminate])
      {
        [progressBar animate        : self  ];
        [progressBar setDoubleValue : aValue];
      }
    else
      {
        [progressBar setMaxValue    : aTotal];
        [progressBar setDoubleValue : aValue];
      }
}

- (NSString*) _timeStringWithTime: (time_t) aTime
{
    return [NSString stringWithFormat: @"%2d:%2d", aTime % 60, aTime / 60];
}

- (void) updateTimeWithValue: (time_t) aValue
{
    [timeDisplay setStringValue: [self _timeStringWithTime: aValue]];
}

- (void) updateTotalTimeWithValue: (time_t) aValue
{
    [totalTimeDisplay setStringValue: [self _timeStringWithTime: aValue]];
}

/**
 * IB Actions
 **/

- (IBAction) play: (id) aSender
{
    MYLOCK_LOCK;

    NSAssert((myState != PLAYING), @"Player is already playing.");

    if (myState == STOPPED)
      {
        AudioFile* aTrack = [self currentTrack];

        [NSApplication detachDrawingThread: @selector(play:)
                                  toTarget: myDecoder
                                withObject: aTrack];
      }
    else
        [myDecoder resume];

    myState = PLAYING;

    [self _updatePlayControls];

    MYLOCK_UNLOCK;
}

- (IBAction) stop: (id) aSender
{
    MYLOCK_LOCK;

    NSAssert((myState != STOPPED), @"Player is already stopped.");

    [myDecoder stop];

    myState = STOPPED;

    [self _updatePlayControls];

    MYLOCK_UNLOCK;
}

- (IBAction) pause: (id) aSender
{
    MYLOCK_LOCK;

    NSAssert((myState == PLAYING), @"Player is not playing; can't pause.");

    [myDecoder pause];

    myState = PAUSED;

    [self _updatePlayControls];

    MYLOCK_UNLOCK;
}

- (IBAction) previousTrack: (id) aSender
{
    if (myState != STOPPED) [self stop: self];

    [self previousTrack];

    [myDecoder wait];

    [self play: self];
}

- (IBAction) nextTrack: (id) aSender
{
    if (myState != STOPPED) [self stop: self];

    [self nextTrack];

    [myDecoder wait];

    [self play: self];
}

- (IBAction) volumeChanged: (id) aSender
{
    int aVolume  = [volumeSlider  intValue];
    int aBalance = [balanceSlider intValue];

    float aLeft  = aVolume/100.0;
    float aRight = aVolume/100.0;

         if (aBalance < 50) aRight = ( (     aBalance *2) / 100.0)*aRight;
    else if (aBalance > 50) aLeft  = ( ((100-aBalance)*2) / 100.0)*aLeft ;

    [Sound setVolume:aLeft:aRight];
}

- (IBAction) selectShuffle: (id) aSender { NSLog(@"Unimplemented: Shuffle"); }

/**
 * Decoder notices
 **/

- (void) donePlaying
{
    MYLOCK_LOCK;

    myState = STOPPED;

    [self _updatePlayControls];

    MYLOCK_UNLOCK;
}

/**
 * UI Controls
 **/

- (void) _updatePlayControls
{
    if (myPlayList)
      {
        [ repeatButton  setEnabled: YES ];
        [ shuffleButton setEnabled: NO  ]; // FIX ME

        [ playButton  setEnabled: (myState == PLAYING) ? NO  : YES ];
        [ pauseButton setEnabled: (myState == PLAYING) ? YES : NO  ];
        [ stopButton  setEnabled: (myState != STOPPED) ? YES : NO  ];

        if (myState == STOPPED)
          {
            if ([[myPlayList audioFiles] count] <= 0)
                [playButton setEnabled: NO];

            [ previousTrackButton setEnabled: NO ];
            [ nextTrackButton     setEnabled: NO ];
          }
        else
          {
            if ([shuffleButton state] == NSOffState)
              {
                [ previousTrackButton setEnabled: [myPlayList hasPrevious] ];
                [ nextTrackButton     setEnabled: [myPlayList hasNext    ] ];
              }
            else
              {
                // FIX ME
                [ previousTrackButton setEnabled: NO ];
                [ nextTrackButton     setEnabled: NO ];
              }
          }
      }
    else
      {
        [ repeatButton        setEnabled: NO ];
        [ shuffleButton       setEnabled: NO ];
        [ previousTrackButton setEnabled: NO ];
        [ nextTrackButton     setEnabled: NO ];
        [ playButton          setEnabled: NO ];
        [ pauseButton         setEnabled: NO ];
        [ stopButton          setEnabled: NO ];
      }
}

@end
