/**
 * OpenAmp Play List
 *
 * Wilfredo Sanchez | wsanchez@apple.com
 * September 22, 1998
 **/

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>

#import "Extentions.h"
#import "ApplicationController.h"
#import "AudioFile.h"
#import "PlayList.h"

@interface PlayList (Private)

// Actions
- (void) _resetPlayList;

// UI Controls
- (void) _initPlayListView;

@end

@implementation PlayList

/**
 * Inits
 **/

- (id) init
{
    if ((self = [super init]))
      {
        // Init iVars
        myPlayList = [[NSMutableArray alloc] init];

        // Init IB outlets
        playListWindow   = nil;
        listMatrix       = nil;
        listUpButton     = nil;
        listDownButton   = nil;
        listDeleteButton = nil;
        listAddButton    = nil;
      }
    return self;
}

- (void) dealloc
{
    [ myPlayList release ];

    [super dealloc];
}

/**
 * Accessors
 **/

- (NSArray*) audioFiles { return myPlayList; }

- (void) addAudioFile: (AudioFile*) aFile
{
    if (aFile)
      {
        [myPlayList addObject: aFile];
        [self _initPlayListView];
      }
}

- (AudioFile*) selectedAudioFile
{
    NSButtonCell* aCell = [listMatrix selectedCell];

    return aCell ? [aCell representedObject] : nil;
}

- (BOOL) hasPrevious
{
    int aRow = [listMatrix selectedRow];

    return (aRow > 0) ? YES : NO;
}

- (BOOL) hasNext
{
    int aRow   = [listMatrix selectedRow ];
    int aCount = [listMatrix numberOfRows];

    return (aRow < aCount-1) ? YES : NO;
}

- (AudioFile*) selectFirstAudioFile
{
    int aCount = [listMatrix numberOfRows];

    if (aCount > 0)
      {
        [listMatrix selectCellAtRow:0 column:0];

        [self playListSelectItem: self];

        return [self selectedAudioFile];
      }
    else
        return nil;
}

- (AudioFile*) selectPreviousAudioFile
{
    int aRow = [listMatrix selectedRow];

    if (aRow > 0)
      {
        [listMatrix selectCellAtRow:aRow-1 column:0];

        [self playListSelectItem: self];

        return [self selectedAudioFile];
      }
    else
        return nil;
}

- (AudioFile*) selectNextAudioFile
{
    int aRow   = [listMatrix selectedRow ];
    int aCount = [listMatrix numberOfRows];

    if (aRow < aCount-1)
      {
        [listMatrix selectCellAtRow:aRow+1 column:0];

        [self playListSelectItem: self];

        return [self selectedAudioFile];
      }
    else
        return nil;
}

/**
 * Actions
 **/

- (void) _resetPlayList
{
    [myPlayList release]; myPlayList = [[NSMutableArray alloc] init];

    // Update display
    [self _initPlayListView];
}

- (BOOL) readFromFile: (NSString*) aFileName
               ofType: (NSString*) aType
{
    NSAssert([aType isEqualToString: @"OpenAmpPlayList"], @"Unknown data type: %@");

    {
        NSArray* aPlayList = [[NSString stringWithContentsOfFile: aFileName] propertyList];

        if (! ( aPlayList                                 &&
               [aPlayList isKindOfClass: [NSArray class]] ) )
          {
            NSString* aMessage =
                [NSString stringWithFormat:
                    @"The file %@ is not a valid OpenAmp play list file.",
                    aFileName];

            [NSPanel runAlertPanelWithName: @"Invalid Play List"
                                   message: aMessage];

            return NO;
          }

        [self _resetPlayList];

        {
            NSEnumerator* aTrackEnumerator = [aPlayList objectEnumerator];
            NSString*     aTrack;

            while ((aTrack = [aTrackEnumerator nextObject]))
              {
                if (! [aTrack isKindOfClass: [NSString class]])
                  {
                    NSString* aMessage =
                    [NSString stringWithFormat:
                        @"The file %@ is not a valid OpenAmp play list file.",
                        aFileName];

                    [NSPanel runAlertPanelWithName: @"Damaged Play List"
                                           message: aMessage];

                    return NO;
                  }
                else
                  {
                    AudioFile* anAudioFile = [AudioFile file];

                    [anAudioFile setFileName: aTrack];

                    [self addAudioFile: anAudioFile];
                  }
              }
        }
    }

    return YES;
}

- (BOOL) writeToFile: (NSString*) aFileName
              ofType: (NSString*) aType
{
    NSAssert([aType isEqualToString: @"OpenAmpPlayList"], @"Unknown data type: %@");

    return [[myPlayList description] writeToFile: aFileName
                                      atomically: NO];
}

- (void) close
{
    [super close];

    [TheApplicationController playListDidClose: self];
}

/**
 * IB Actions
 **/

- (IBAction) playListSelectItem: (id) aSender
{
    int aRow   = [listMatrix selectedRow ];
    int aCount = [listMatrix numberOfRows];

    [ listUpButton     setEnabled: (aRow == 0       ) ? NO : YES ];
    [ listDownButton   setEnabled: (aRow == aCount-1) ? NO : YES ];
    [ listDeleteButton setEnabled:                           YES ];
}

- (IBAction) playListPromoteItem: (id) aSender
{
    int aRow = [listMatrix selectedRow];

    NSAssert((aRow > 0), @"Error: Attempt to promote first item.");

    {
        NSString* anItem = [[myPlayList objectAtIndex: aRow] retain];

        [myPlayList removeObjectAtIndex:aRow];
        [myPlayList insertObject:anItem atIndex:aRow-1];

        [anItem release];
    }

    [self _initPlayListView];

    [listMatrix selectCellAtRow:aRow-1 column:0];

    [self playListSelectItem: self];

    [self updateChangeCount: NSChangeDone];
}

- (IBAction) playListDemoteItem: (id) aSender
{
    int aRow   = [listMatrix selectedRow ];
    int aCount = [listMatrix numberOfRows];

    NSAssert((aRow < aCount-1), @"Error: Attempt to demote last item.");

    {
        NSString* anItem = [[myPlayList objectAtIndex: aRow] retain];

        [myPlayList removeObjectAtIndex: aRow];
        [myPlayList insertObject:anItem atIndex:aRow+1];

        [anItem release];
    }

    [self _initPlayListView];

    [listMatrix selectCellAtRow:aRow+1 column:0];

    [self playListSelectItem: self];

    [self updateChangeCount: NSChangeDone];
}

- (IBAction) playListAddItem: (id) aSender
{
    [[NSDocumentController sharedDocumentController] openDocument: self];

    [self updateChangeCount: NSChangeDone];
}

- (IBAction) playListDeleteItem: (id) aSender
{
    int aRow   = [listMatrix selectedRow ];
    int aCount = [listMatrix numberOfRows];

    NSAssert((aRow >= 0), @"Error: No selected track to delete.");

    [myPlayList removeObjectAtIndex: aRow];

    [self _initPlayListView];

    if (aCount > 1)
      {
        [listMatrix selectCellAtRow:((aRow == aCount-1) ? aRow-1 : aRow) column:0];

        [self playListSelectItem: self];
      }

    [self updateChangeCount: NSChangeDone];
}

/**
 * Window Delegate
 **/

- (NSString*) windowNibName
{
    return @"PlayList";
}

- (void) awakeFromNib
{
    NSAssert(playListWindow   &&
             listMatrix       &&
             listUpButton     &&
             listDownButton   &&
             listDeleteButton &&
             listAddButton    ,
             @"PlayList: nib failed to connect properly.");

    // Init buttons
    [ listUpButton     setEnabled: NO  ];
    [ listDownButton   setEnabled: NO  ];
    [ listDeleteButton setEnabled: NO  ];
    [ listAddButton    setEnabled: YES ];

    [self _initPlayListView];
}

- (void) windowDidBecomeMain: (NSNotification*) aNotification
{
    [TheApplicationController setActivePlayList: self];
}

/**
 * UI Controls
 **/

- (void) _initPlayListView
{
    if (! listMatrix) return; // Haven't awakened yet

    // Reset play list view
    {
        int aRowCount = [listMatrix numberOfRows];

        while (aRowCount--) [listMatrix removeRow: 0];
    }

    // Re-populate play list view
    if (myPlayList)
      {
        NSEnumerator* aTrackEnumerator = [myPlayList objectEnumerator];
        AudioFile*    aTrack;
        int           aRow;

        for (aRow = 0; (aTrack = [aTrackEnumerator nextObject]); aRow++)
          {
            NSButtonCell* aCell;

            [listMatrix addRow];

            aCell = [listMatrix cellAtRow:aRow column:0];

            [aCell setTitle: [aTrack title]];
            [aCell setRepresentedObject: aTrack];
          }
      }

    // Request display update
    [listMatrix sizeToCells];
    [listMatrix setNeedsDisplay: YES];
}

@end
