/**
 * OpenAmp Audio Stream
 *
 * Wilfredo Sanchez | wsanchez@apple.com
 * September 20, 1998
 **/

#import  <Foundation/NSLock.h>
#import  <SoundKit/SoundKit.h>
#include <assert.h>

#include "common.h"
#include "SoundDecoder.h"
#import  "SoundStream.h"

#ifndef DEBUG
#define DEBUG 0
#endif

@implementation SoundStream

/**
 * Inits
 **/

- (id) initWithLayer: (layer*) aLayer
        channelCount: (int   ) aChannelCount
{
    if ((self = [super init]))
      {
        int aSampleRate = (s_freq[aLayer->version][aLayer->sampling_frequency] * 1000);
        int aBitRate    = bitrate[aLayer->version][aLayer->lay-1][aLayer->bitrate_index];

        NXSoundDeviceError anError = NX_SoundDeviceErrorNone;

#if DEBUG
        NSLog(@"Sampling Rate = %d", aSampleRate  );
        NSLog(@"Bit Rate      = %d", aBitRate     );
        NSLog(@"Channel Count = %d", aChannelCount);
#endif

        if (aSampleRate != 22050 &&
            aSampleRate != 44100 )
          {
            NSLog(@"Sample frequency %d is not supported.", aSampleRate);
            [self release];
            return nil;
          }

        if (aBitRate != 128)
          {
            NSLog(@"Bit rate %d is not supported.", aBitRate);
            [self release];
            return nil;
          }

        if (! [NXSoundOut setUseSeparateThread: YES])
          {
            NSLog(@"Failed to init sound output.\n");
            [self release];
            errno = ENXIO;
            [self release];
            return nil;
          }

        mySoundDevice      = [[NXSoundOut alloc] init];
        myStreamParameters = nil;

        myStreamParameters = [[NXSoundParameters alloc] init];

        [ myStreamParameters setParameter: NX_SoundStreamSamplingRate toInt: aSampleRate                         ];
        [ myStreamParameters setParameter: NX_SoundStreamChannelCount toInt: aChannelCount                       ];
        [ myStreamParameters setParameter: NX_SoundStreamDataEncoding toInt: NX_SoundStreamDataEncoding_Linear16 ];

        if (! (myPlayStream = [[NXPlayStream alloc] initOnDevice: mySoundDevice
                                                  withParameters: myStreamParameters]))
          {
            NSLog(@"Failed to init play stream: %@",
                  [NXSoundDevice textForError: anError]);
            [self release];
            errno = ENXIO;
            [self release];
            return nil;
          }

        [myPlayStream setDelegate: self];

        if ((anError = [myPlayStream activate]) != NX_SoundDeviceErrorNone)
          {
            NSLog(@"Failed to activate play stream %@: %@",
                  myPlayStream, [NXSoundDevice textForError: anError]);
            [self release];
            errno = ENXIO;
            [self release];
            return nil;
          }

        [myPlayStream pause: self];

        myPlayingLock = [[NSLock alloc] init];
        myCounterLock = [[NSLock alloc] init];

        myQueuedBuffersCount    = 0;
        myRemainingBuffersCount = 0;
      }
    return self;
}

- (void) dealloc
{
    [ myPlayStream       release ];
    [ mySoundDevice      release ];
    [ myStreamParameters release ];
    [ myPlayingLock      release ];
    [ myCounterLock      release ];

    [super dealloc];
}

/**
 * Actions
 **/

- (NXSoundDeviceError) playBuffer: (const void* ) aBuffer
                             size: (unsigned int) aSize
{
    int                aCount;
    NXSoundDeviceError anError;

    [myCounterLock lock];
    {
        aCount = myQueuedBuffersCount++;
        myRemainingBuffersCount++;
    }
    [myCounterLock unlock];

    if (myRemainingBuffersCount == 1) [myPlayingLock lock];

#if DEBUG
    NSLog(@"Playing %d bytes via %@. Count = %d.", aSize, myPlayStream, aCount);
#endif

    anError = [myPlayStream playBuffer: (void*)aBuffer
                                  size: aSize
                                   tag: aCount];

    if (anError) NSLog(@"Error while playing %ld bytes via stream %@: %@",
                       aSize, myPlayStream, [NXSoundDevice textForError: anError]);

    return anError;
}

- (void) wait
{
    [myPlayingLock lock  ];
    [myPlayingLock unlock];
}

- (void) abort  { [myPlayStream abort  : self]; [myPlayingLock unlock]; }
- (void) pause  { [myPlayStream pause  : self];                         }
- (void) resume { [myPlayStream resume : self];                         }

/**
 * myPlayStream Delegate
 **/

- (void) soundStreamDidUnderrun: (NXSoundStream*) aSender
{
    NSLog(@"Buffer underrun in sound stream %@.", aSender);
}

- (void) soundStream: (NXSoundStream*) aSender
   didCompleteBuffer: (int           ) aTag
{
    int aCount;

    [myCounterLock lock];
    {
        aCount = --myRemainingBuffersCount;
    }
    [myCounterLock unlock];

    // If this is critical code, it should be in the locked
    // section above. But's it's just the display update code
    // and we don't want to hold up the decoder thread (which
    // calls playBuffer) from processing while a display update
    // is going on.
    // It is therefore possible that myQueuedBuffersCount here
    // could be stale, but that just means the progress bar is a
    // little off. Because of the above lock, however, this code
    // won't execute twice simultaneously, which could cause bad
    // drawing.
    [decoder soundStream: self
          finishedBuffer: aTag
                 ofTotal: myQueuedBuffersCount];

#if DEBUG
    NSLog(@"Sound stream %@ finished playing buffer %d. Queue = %d.",
          aSender, aTag, aCount);
#endif

    if (aCount == 0) [myPlayingLock unlock];

    assert(aCount >= 0);
}

@end
