#import "TADocument.h"
#import <Foundation/NSBundle.h>		/* LocalizedString */
#import <Foundation/NSFileManager.h>
#import <AppKit/NSPanel.h>		/* NSRunAlertPanel() */
#import <stdio.h>
#import <stdlib.h>
#import <string.h>
#import <libc.h>
#import "AlbumCtr.h"
#import "TACommon.h"
#import "TAController.h"
#import "ImageInfoNode.h"

#define  MAXLINE	512

@implementation TADocument

+ (int)isAlbumPath:(NSString *)path
{
	NSString *d = [path pathExtension];
	if ([d isEqualToString: Suffix])
		return A_Album;
	if ([d isEqualToString: HTMLDirSuffix])
		return A_HTML;
	return A_None;
}

- (id)init
{
	static int theSerialNumber = 0;

	[super init];
	serialNumber = ++theSerialNumber;	/* Serial Number > 0 */
	info = nil;
	return self;
}

- (void)dealloc
{
	if (docname) [docname release];
	if (_docpath) [_docpath release];
	if (info) [info release];
	[super dealloc];
}

- (void)setController: sender { controller = sender; }
- window { return window; }
- (int)documentMode { return relative; }
- (int)serialNumber { return serialNumber; }
- (NSSize)thumbSize { return thsize; }
- (NSString *)filename { return docname; }
- (void)setFilename:(NSString *)name
{
	int	len;

	docname = [name retain];
	_docpath = [docname stringByDeletingLastPathComponent];
	len = [_docpath length];
	if ([_docpath characterAtIndex:(len - 1)] != '/')
		_docpath = [docname substringToIndex:(len + 1)];
	[_docpath retain];
}

- (int)columns { return columns; }

- (int)rows
{
	if (columns <= 0 || info == nil)
		return 0;
	return ([info count] + columns - 1) / columns;
}


- (ImageInfoNode *)imageInfoAtRow:(int)row column:(int)col
{
	int num;

	if (columns <= 0 || info == nil)
		return nil;
	num = row * columns + col;
	if ([info count] <= num)
		return nil;
	return [info objectAtIndex: num];
}

- (ImageInfoNode *)imageInfoAtSelectedCell
{
	return [albumctr selectedCellInfo];
}

- (BOOL)removeImageInfoOf:(id)rmcel
{
	int	tag = [rmcel tag];

	if ([info count] <= tag)
		return NO;
	[info removeObjectAtIndex:tag];
	return YES;
}

- (BOOL)pasteImageInfo:(ImageInfoNode *)node at:(id)cel mode:(int)mode
{
	int	pos, lev;
	NSString *fname, *title;

	if (cel == nil || [info count] <= 0)
		pos = 0;
	else {
		int tag = [cel tag];
		if (node == nil || [info count] <= tag)
			return NO;
		pos = tag;
		if (mode == P_Add) ++pos;
	}
	if ([node origDocSN] != serialNumber) {
		[node setOrigDocSN:serialNumber];
		fname = [node filename];
		title = nil;
		if (relative == m_absolute ||
			(lev = [self levelOfSubDirectory:fname]) <= 0)
			[node setAbsolute:YES];
		else {
			[node setAbsolute:NO];
			 if (lev == 2 && relative == m_relative)
				title = [self pathRelativeToAlbum:fname];
		}
		[node setTitlename:title];
	}
	[info insertObject:node atIndex:pos];
	return YES;
}

- (BOOL)changeWidthBy:(int)inc
{
	int	w;

	if (columns <= 0 || info == nil)
		return NO;
	w = columns + inc;
	if (w < 1 || w > MAXCOLUMN)
		return NO;
	columns = w;
	[albumctr changeColumns];
	return YES;
}

- (void)setThumbnailSize:(NSSize)size
{
	int	i, n;

	if (thsize.width == size.width && thsize.height == size.height)
		return;
	thsize = size;
	for (i = 0, n = [info count]; i < n; i++)
		[[info objectAtIndex: i] setThumbSize:thsize];
	[albumctr setThumbnailSize:thsize];
}

static unsigned char *get_str(unsigned char *buf)
{
	int	i;
	unsigned char	*r = NULL;

	for (i = 0; buf[i] >= ' '; i++)
		if (buf[i] == '\"') {
			r = &buf[++i];
			break;
		}
	if (r == NULL)
		return NULL;
	for ( ; buf[i] >= ' ' && buf[i] != '\"'; i++)
		;
	buf[i] = 0;
	return r;
}

static unsigned char *get_desc(unsigned char *buf)
{
	int	i;
	unsigned char	*r = NULL;

	for (i = 0; buf[i] == ' ' || buf[i] == '\t'; i++)
		;
	r = &buf[i];
	for ( ; buf[i] >= ' ' || buf[i] == '\t'; i++)
		;
	buf[i] = 0;
	return r;
}

/* Local Method */
- (void)checkVersion
{
	NSString *warn, *mesg;

	if (version == 0 || version <= VERSION_NO)
		return;
	warn = NSLocalizedString(@"WARNING", WARNING);
	mesg = NSLocalizedString(@"Version %d may be misread.", UnknownVersion);
	NSRunAlertPanel(warn, mesg, nil, nil, nil, version);
}

/* Local Method */
- (id)openIndexFile:(NSString *)filename dir:(NSString *)dirname
{
	FILE *fp;
	unsigned char *fnam;
	unsigned char buf[MAXLINE];
	ImageInfoNode *node = nil;
	NSString *str, *relpath;
	int	wid, hei, sz;

	str = [dirname stringByAppendingPathComponent:filename];
	if ((fp = fopen([str cString], "r")) == NULL)
		return nil;
	relpath = [dirname stringByDeletingLastPathComponent];
	columns = 0;
	version = 0;
	relative = m_absolute;
	thsize.width = thsize.height = P_Width;
	while (fgets(buf, MAXLINE, fp) && buf[0] == '#') {
		if (strncasecmp(&buf[1], "version", 7) == 0)
			version = atoi(&buf[8]);
		else if (strncasecmp(&buf[1], "columns", 7) == 0)
			columns = atoi(&buf[8]);
		else if (strncasecmp(&buf[1], "relative", 8) == 0)
			relative = (atoi(&buf[9]) == 2)
					? m_relative_p : m_relative;
		else if (strncasecmp(&buf[1], "size", 4) == 0) {
			int w, h, n;
			n = sscanf(&buf[5], "%d %d", &w, &h);
			if (n >= 1) {
			    thsize.width = (w < P_MIN_Size) ? P_MIN_Size
				: ((w > P_MAX_Size) ? P_MAX_Size : w);
			    if (n == 2) {
				thsize.height = (h < P_MIN_Size) ? P_MIN_Size
					: ((h > P_MAX_Size) ? P_MAX_Size : h);
			    }else
				thsize.height = thsize.width;
				/* Doc Version 3 */
			}
		}
	}
	[self checkVersion];
	if (feof(fp) || columns > MAXCOLUMN || columns <= 0) {
		fclose(fp);
		return nil;
	}
	do {
		if (buf[0] == PathSym || buf[0] == DummySym)
			break;
	}while (fgets(buf, MAXLINE, fp));

	info = [[NSMutableArray alloc] initWithCapacity: 1];
	do {
		switch (buf[0]) {
		case PathSym:
		case DummySym:
			node = [[ImageInfoNode alloc] init];
			[node autorelease];
			[node setThumbSize:thsize];
			[info addObject:node];
			if (buf[0] == DummySym)
				break;
			if ((fnam = get_str(buf+1)) == NULL)
				break;
			str = [NSString stringWithCString: fnam];
			if ([str isAbsolutePath])
				[node setAbsolute:YES];
			else {
				if (relative == m_relative
					&& [[str pathComponents] count] == 2)
					[node setTitlename:str];
				str = [relpath stringByAppendingPathComponent: str];

			}
			[node setFilename: str];
			break;
		case IconSym:
			if ((fnam = get_str(buf+1)) != NULL) {
				str = [NSString stringWithCString: fnam];
				if ([str isAbsolutePath]) {
					[node setIconname:str];
					[node setIconImage];
					[node setIconname:nil];
				}else /* ordinary icon */
					[node setIconname: [dirname
					 stringByAppendingPathComponent: str]];
			}
			break;
		case CommSym:
			if ((fnam = get_desc(buf+1)) != NULL) {
				if (version == 1)
				    str = [NSString stringWithCString: fnam];
				else /* version >= 2 */
				    str = [NSString stringWithCString:
				    			readComment(fnam)];
				[node setComment: str];
			}
			break;
		case SizeSym:
			if (sscanf(buf+1, "%d %d %d", &wid, &hei, &sz) == 3)
				[node setWidth:wid height:hei size:(long)sz];
			break;
		default:
			break;
		}
	}while (fgets(buf, MAXLINE, fp));
	fclose(fp);
	return self;
}

- (id)openDocument:(NSString *)filename
{
	NSFileManager	*manager;
	BOOL	isdir;

	manager = [NSFileManager defaultManager];
	isdir = NO;
	if (![manager fileExistsAtPath:filename isDirectory:&isdir] || !isdir)
		return nil;
	if (![manager isReadableFileAtPath:filename])
		return nil;

	[self setFilename: filename];
	if ([self openIndexFile:IndexFile dir:filename] == nil
	&& [self openIndexFile:IndexFileDos dir:filename] == nil)
		return nil;
	albumctr = [[AlbumCtr alloc] init:self withName:docname];
	[albumctr setIsNew:NO];
	[albumctr locateWindow];
	window = [albumctr window];
	return self;
}

- (void)setMode:(int)rmode columns:(int)cols size:(NSSize)size
{
	version = 0;
	columns = cols;
	thsize = size;
	relative = rmode;
}

- (void)addImageInfoNodes:(NSArray *)fnarray in:(NSString *)dirname
	asMode:(int)mode
{
	if ([[fnarray objectAtIndex:0] isKindOfClass:[NSString class]]) {
		int	i, n;
		NSString *s;
		n = [fnarray count];
		if (info == nil)
			info = [[NSMutableArray alloc] initWithCapacity: n];
		for (i = 0; i < n; i++) {
			ImageInfoNode *node;
			node = [[ImageInfoNode alloc] init];
			[node autorelease];
			s = [fnarray objectAtIndex: i];
			[node setFilename:
				[dirname stringByAppendingPathComponent:s]];
			if ([[self class] isAlbumPath:s] && mode == m_relative)
				[node setTitlename:s];
			[node setAbsolute:(mode == m_absolute)];
			[node setThumbSize:thsize];
			[info addObject:node];
		}
	}else if (info == nil)
		info = [[fnarray mutableCopy] retain];
	else
		[info addObjectsFromArray:fnarray];
}

- (id)openDirectory:(NSString *)dirname files:(NSArray *)fnarray
{
	int	n;

	[self addImageInfoNodes:fnarray in:dirname asMode:relative];
	if ((n = [info count]) < columns)
		columns = n;
	albumctr = [[AlbumCtr alloc] init:self withName:docname];
	[albumctr setIsNew:YES];
	[albumctr locateWindow];
	window = [albumctr window];
	[albumctr setDocumentEdited:YES];
	return self;
}

- (BOOL)isNew
{
	return [albumctr isNew];
}

- (BOOL)isDocumentEdited
{
	return [albumctr isDocumentEdited];
}

- (void)closeAnyway:(id)sender
{
	[albumctr setDocumentEdited:NO];
	[window performClose:sender];
}

- (void)confirmAndClose:(id)sender
{
	[albumctr confirmAndClose:sender];
}

- (void)releaseSelf
{
	(void)[controller checkWindow: self delete:YES];
	[self release];
}

- (int)levelOfSubDirectory:(NSString *)path
{
	NSString *rp;

	if ((rp = [self pathRelativeToAlbum:path]) == nil)
		return -1;
	return [[rp pathComponents] count];
}

- (NSString *)pathRelativeToAlbum:(NSString *)name
{
	if ([name hasPrefix:_docpath])
		return [name substringFromIndex:[_docpath length]];
	return nil;
}

/* notice from AlbumCtr */
- (void)didBecomeMain:(id)sender
{
	[controller didBecomeMain:self];
}

/*** Print ***/
- (void)print: sender
{
	[albumctr print: sender];
}

@end
