/*
 File:       ApplicationDelegate.m
 Written By: Scott Anguish
 Created:    Dec 9, 1997
 Copyright:  Copyright 1997 by Scott Anguish, all rights reserved.
*/


#import "ApplicationDelegate.h"
#import "NSFileManager+unique.h"
#import "NSString+utils.h"
#import "NSArray+utils.h"
#import "NSColor+utils.h"
#import "FileDeletePanelController.h"
#import "DFProgressPanel.h"
#import "PrefsController.h"



@implementation ApplicationDelegate

- (void)showPreferencesPanel:sender
{
    [[PrefsController sharedInstance] showWindow];
}


- init
{
    [super init];
    initializationComplete=NO;
    createdDirectories = [[NSMutableArray array] retain];
    return self;
}


- (void)deleteFiles:(NSArray *)anArray
{
    NSEnumerator *overFiles;
    NSString *eachFile;
    int i=0;
    DFProgressPanel *progressPanel=[DFProgressPanel sharedInstance];

    [[PrefsController sharedInstance] hideWindow];

    [progressPanel setProgressIndeterminate:NO];
    [progressPanel setProgressMaxValue:[anArray count]];
    [progressPanel setProgressValue:i];
    [progressPanel startAnimation];
    [progressPanel setPanelTitle:@"Deleting temporary files"];
    [progressPanel showWindow:self];

    overFiles=[anArray objectEnumerator];
    while (( eachFile = [overFiles nextObject] ))
      {
        [progressPanel setProgressValue:i];
        i++;
        [progressPanel setProgressLabel:[eachFile lastPathComponent]];
        [[NSFileManager defaultManager] removeFileAtPath:eachFile handler:nil];
      }
    [progressPanel hideWindow:self];

    [createdDirectories removeAllObjects];
    [[NSWorkspace sharedWorkspace] noteFileSystemChanged];

};


// - (BOOL)applicationShouldTerminate:(NSApplication *)app;
//
// The app is about to terminate, so we'll cleanup after ourselves,
// removing temporary files as required.

- (BOOL)applicationShouldTerminate:(NSApplication *)app;
{
    if (initializationComplete)
      {
        return [self removeCreatedDirectoriesAllowCancel:YES];
      }

    return YES;
}

// -(void)applicationWillPowerOff:(NSNotification *)aNotification;
//
// Handles "power off" Workspace notifications (which should have been
// called "logout" notifications...) to which we subscribed in
// -applicationDidFinishLaunching:.
- (void)applicationWillPowerOff:(NSNotification *)aNotification
{
    [self removeCreatedDirectoriesAllowCancel:NO];
}



- (BOOL)removeCreatedDirectoriesAllowCancel:(BOOL)allowCancel
{
    BOOL tempFilesExist;
    BOOL deleteTempFilesOnQuit;
    
    // If there are no files in the /tmp directory, we will just go
    // ahead and delete the temporary directory
    tempFilesExist=([createdDirectories count] > 0);

    // We read this default to determine if we should bother to ask,
    // or just go a head and delete the files
    deleteTempFilesOnQuit=[[NSUserDefaults standardUserDefaults] boolForKey:@"DeleteTempFilesOnQuit"];


    // If there are temp files and we are supposed to ask instead
    // of just deleting the directory
    if (tempFilesExist && !deleteTempFilesOnQuit)
      {
        FileDeletePanelController *theController;
        BOOL clickedYes;

        theController=[[FileDeletePanelController alloc] initWithPromptedFiles:createdDirectories];
        if (!allowCancel)
            [theController disableCancelButton];
        clickedYes=[theController run];
        if (clickedYes)
            [self deleteFiles:[theController checkedFiles]];
        [theController release];
        return clickedYes;
      }
    else
        [self deleteFiles:createdDirectories];
       

    return YES;
}



// - (BOOL)application:(NSApplication *)sender openFile:(NSString *)filename;
// - (BOOL)application:(NSApplication *)sender openTempFile:(NSString *)filename;
//
// These methods are called when a file is double clicked on in the
// Workspace to open OpenUp.

- (BOOL)application:(NSApplication *)sender openFile:(NSString *)filename;
{
    // simply forward to the decompressFile: method the filename
    // that we were passed to open
    [self decompressFile:filename];
    return YES;
}

- (BOOL)application:(NSApplication *)sender openTempFile:(NSString *)filename;
{
    // simply forward to the decompressFile: method the filename
    // that we were passed to open

    return [self application:sender openFile:filename];
}




// - (void)initializationFailure:(NSString *)value;
//
// Its necessary to seperate this from the awakeFromNib method,
// since you can't call NSRunAlertPanel in the awakeFromNib because
// the runloop isn't going yet.  The application hasn't finished
// starting yet.  If an error condition is found during the startup,
// then this method is performed using
// performDelayed:withObject:afterDelay:

- (void)initializationFailure:(NSString *)value;
{
    int result;
    result = NSRunAlertPanel(@"OpenUp",
                             value,
                             NSLocalizedString(@"InitializationFailureButton",@"OK"),
                             NULL,NULL);

    [[NSApplication sharedApplication] terminate:self];
}



// -(void)awakeFromNib;
//
// Another workhorse, this method attempts to setup the working
// directory information and ensure that the information is valid
// before the application finishes launching, or attempts to open
// a file that was double clicked on in the Workspace.

- (void)awakeFromNib;
{
    NSString *thePath;
    NSString *errorString=nil;

    
    // read the user defaults
    [self setupUserDefaults];
    [self setupForInfoPanel];
    [self showInfoPanel:self];

    // read the configuration plist which gives us the launchPath
    // and arguments to use for the decompression of various filetypes,
    // as well as the file types to match
    // 
    // the configuration is stored in the application's wrapper and
    // then retain the dictionary and array for further use
    thePath=[[NSBundle mainBundle] pathForResource:@"filesConfig" ofType:@"plist"];
    fileTypeConfigArray = [[NSArray arrayWithContentsOfFile:thePath] retain];
    if (!fileTypeConfigArray)
        errorString=[NSString stringWithFormat:NSLocalizedString(@"ErrorLoadingFilesConfig",@"Error Loading Files Config")];

    thePath=[[NSBundle mainBundle] pathForResource:@"servicesConfig" ofType:@"plist"];
    servicesDictionary = [[NSDictionary dictionaryWithContentsOfFile:thePath] retain];
    if (!servicesDictionary)
        errorString=[NSString stringWithFormat:NSLocalizedString(@"ErrorLoadingServicesConfig",@"Error Loading Services Config")];

    /*
    // the baseFilename will be used by our Category on NSFileManager
    // to create a unique directory with a more friendly name than
    // we could get using the conventional [NSProcessInfo
    // globallyUniqueString] Its important that it contain a %@
    // somewhere in its name
    // 
    // if we can create the directory, then we'll setup the
    // appWorkdingDirectory and retain the path otherwise, we should
    // quit
    // 
    // actually in the otherwise case we should determine the problem
    // and inform the user.  Its interesting to note that we can't
    // just put up an NSRunAlertPanel at this point, since the
    // runloop isn't running, so we send our ApplicationDelegate
    // object a performSelector:withObject:afterDelay:  so that the
    // NSRunAlertPanel is delayed until the runloop has already
    // started.

    if (!errorString)
      {
        NSString *tempPath=nil;
        NSString *baseFilename;
        NSString *tempFilesDefaultValue;

        baseFilename=[NSString stringWithFormat:@"%@_%%@",[[NSProcessInfo processInfo] processName]];
        tempFilesDefaultValue=[[[NSUserDefaults standardUserDefaults] stringForKey:@"DecompressedFilesDirectory"] stringByStandardizingPath];
        
        tempPath=[[NSFileManager defaultManager] createUniqueDirectoryAtPath:tempFilesDefaultValue
                                                            withBaseFilename:baseFilename
                                                                  attributes:nil];
        if (tempPath)
            appWorkingDirectory = [tempPath retain];
        else
            errorString=[NSString stringWithFormat:NSLocalizedString(@"InitializationFailure",@"Failed to Create Directory in %@"),tempFilesDefaultValue];
      }

    */

    if (errorString)
      {
        [self performSelector:@selector(initializationFailure:) withObject:errorString afterDelay:0];
        return;
      }


    // lets set up some other UI bits now too
    //
    [errorTextView setFont:[NSFont userFixedPitchFontOfSize:10]];
    [errorTextView setBackgroundColor:[NSColor tanTextBackgroundColor]];
    [debugTextView setFont:[NSFont userFixedPitchFontOfSize:10]];
    [debugTextView setBackgroundColor:[NSColor tanTextBackgroundColor]];
    if (!errorString)
        initializationComplete=YES;
}

- (NSString *)decompressTargetDirectory
{
    NSUserDefaults *defaults;

    defaults=[NSUserDefaults standardUserDefaults];
    [defaults synchronize];
    return [[defaults stringForKey:@"DecompressedFilesDirectory"] stringByStandardizingPath];
}

// -(void)applicationDidFinishLaunching:(NSNotification *)aNotification;
//
// Sent when the application is finished loading we set our
// ApplicationDelegate (this object) to be the target for Services
// to notify and register a request for notification on
// windowDidBecomeMain:

-(void)applicationDidFinishLaunching:(NSNotification *)aNotification;
{
    [[[NSWorkspace sharedWorkspace] notificationCenter] addObserver:self
                                                           selector:@selector(applicationWillPowerOff:)
                                                               name:NSWorkspaceWillPowerOffNotification
                                                             object:nil];	
    [NSApp setServicesProvider:self];
    [infoPanel performSelector:@selector(close) withObject:self afterDelay:10];

}

// - (void)setupUserDefaults;
//
// This should probably be added to NSUserDefaults as a category,
// since its universally useful.  It loads a defaults.plist file
// from the app wrapper, and then sets the defaults if they don't
// already exist.

- (void)setupUserDefaults;
{
    NSUserDefaults *defaults;
    NSDictionary *defaultsPlist;
    NSEnumerator *overDefaults;
    id eachDefault;

    // this isn't required, but saves us a few method calls
    defaults = [NSUserDefaults standardUserDefaults];

    // load the defaults.plist from the app wrapper.  This makes it
    // easy to add new defaults just using a text editor instead of
    // hard-coding them into the application
    defaultsPlist=[NSDictionary dictionaryWithContentsOfFile:[[NSBundle mainBundle] pathForResource:@"defaults" ofType:@"plist"]];

    // enumerate over all the keys in the dictionary
    overDefaults=[[defaultsPlist allKeys] objectEnumerator];
    while (eachDefault = [overDefaults nextObject])
      {
        // for each key in the dictionary
        // check if there is a value already registered for it
        // and if there isn't, then register the value that was in the file
        if (![defaults stringForKey:eachDefault])
          {
            [defaults setObject:[defaultsPlist objectForKey:eachDefault] forKey:eachDefault];
          }

      }

    // force the defaults to save to the disk
    [defaults synchronize];
}




// - (void)dealloc;
//
// release the objects that we have explicitly retained, and then call [super dealloc]

- (void)dealloc;
{

    [infoPanelSupportedTypes release];
    [createdDirectories release];
    [fileTypeConfigArray release];
    [servicesDictionary release];
    [super dealloc];
}



// methods common to decompression and compression
//
//

- (NSString *)shellPathUsingConfiguration:(NSDictionary *)fileConfig;
{
    if ([fileConfig objectForKey:@"shell"])
        return [fileConfig objectForKey:@"shell"];
 
    return [[NSUserDefaults standardUserDefaults] stringForKey:@"DefaultShell"];
}

- (NSArray *)shellArgsUsingConfiguration:(NSDictionary *)fileConfig;
{
    if ([fileConfig objectForKey:@"shell_args"])
        return [fileConfig objectForKey:@"shell_args"];

    return [NSArray arrayWithObject:[[NSUserDefaults standardUserDefaults] stringForKey:@"DefaultShellArgs"]];
}


- (NSDictionary *)wrappedProgramsUsingConfiguration:(NSDictionary *)fileConfig;
{
    NSMutableArray *outKeys,*outValues;
    NSArray *keys;
    NSMutableDictionary *outDict;
    NSEnumerator *keysEnumerator;
    NSString *eachKey;

    keys=[fileConfig objectForKey:@"wrapped_programs"];
    outKeys=[NSMutableArray array];
    outValues=[NSMutableArray array];
    
    keysEnumerator=[keys objectEnumerator];
    while ((eachKey = [keysEnumerator nextObject]))
      {
        NSString *pathToProg;
        NSString *theKey;
        
        theKey=[NSString stringWithFormat:@"%%%%WRAPPED_PROGRAM_%@%%%%",[eachKey uppercaseString]];
        pathToProg=[[NSBundle mainBundle] pathForResource:eachKey ofType:@""];

        [outKeys addObject:theKey];
        
        if (pathToProg)
            [outValues addObject:pathToProg];
        else
            [outValues addObject:eachKey];
      }
    outDict=[NSMutableDictionary dictionary];
    [outDict setObject:outKeys forKey:@"keys"];
    [outDict setObject:outValues forKey:@"values"];
    
    return [NSDictionary dictionaryWithDictionary:outDict];
        
    

}

//
// We act as the delegate for the SoundKit when we are playing a sound
// to indicate that we are completed a task.
//
// So, here we autorelease the sound
//
- (void)didPlay:(id)sender;
{
    [sender autorelease];
}



@end


























