/* EOAccessAdditions.m created by lindberg on Mon 20-Dec-1999 */
/*-
 * Copyright (c) 2000
 *      Blacksmith, Inc.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Blacksmith, Inc.
 * 4. The name Blacksmith may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY BLACKSMITH ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL BLACKSMITH BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#import "EOAccessAdditions.h"
#import "FoundationAdditions.h"

@implementation EOEntity (EOGenerator)

- (BOOL)hasParentEntity
{
    return ([self parentEntity] != nil);
}

- (NSString *)frameworkName
{
    return [[self model] frameworkName];
}

/*"
 * Returns <Framework/ClassName.h> if inside a framework, "ClassName.h"
 * otherwise.
"*/
- (NSString *)objcImportString
{
    return [self objcImportStringInRelationToEntity:nil];
}

/*"
 * If sourceEntity is nil, returns <Framework/ClassName.h> if inside a
 * framework, "ClassName.h" otherwise. If sourceEntity is non-nil, the
 * <Framework/ClassName.h> version will only be returned if the receiver's
 * framework and sourceEntity's framework are different.  This depends on
 * the -#frameworkName method doing the right thing.
"*/
- (NSString *)objcImportStringInRelationToEntity:(EOEntity *)sourceEntity
{
    NSString *myClass = [self className];
    NSString *myFramework = [self frameworkName];
    NSString *sourceFramework = [sourceEntity frameworkName];

    if ([myClass hasSuffix:@"EOGenericRecord"]) return nil;

    if (myFramework != nil &&
        (sourceEntity == nil || ![sourceFramework isEqualToString:myFramework]))
    {
        return [NSString stringWithFormat:@"<%@/%@.h>", myFramework, myClass];
    }
    else
    {
        return [NSString stringWithFormat:@"\"%@.h\"", myClass];
    }
}

/*"
 * If there is no parent entity, returns nil. Otherwise, uses
 * -#objcImportStringInRelationToEntity: to determine the string needed to
 * \#import the parent class.
"*/
- (NSString *)parentObjCImportString
{
    return [[self parentEntity] objcImportStringInRelationToEntity:self];
}

/*"
 * If -#parentObjCImportString is nil, returns an empty array, otherwise returns
 * a single-element array with the parent import string.
"*/
- (NSArray *)arrayWithParentObjCImportStringIfNeeded
{
    NSString *import = [self parentObjCImportString];

    if (import != nil)
        return [NSArray arrayWithObject:import];
    else
        return [NSArray array];
}

/*"
 * A parallel for -#referencedClassNames for dealing with import strings. The
 * array will contain a uniqued list of \#import strings for all related
 * entities.
"*/
- (NSArray *)referencedObjCImportStrings
{
    NSArray        *relationships = [self relationships];
    int            i, count = [relationships count];
    NSMutableArray *refImports = [NSMutableArray arrayWithCapacity:count];

    for (i=0; i<count; i++)
    {
        EORelationship *relationship = [relationships objectAtIndex:i];
        EOEntity       *destEntity   = [relationship destinationEntity];

        if (![[destEntity className] hasSuffix:@"EOGenericRecord"])
        {
            NSString *import = [relationship objcImportString];
            if (import && ![refImports containsObject:import])
                [refImports addObject:import];
        }
    }

    return [refImports sortedArrayUsingSelector:@selector(compare:)];
}

@end


@implementation EOModel (EOGenerator)

/*"
 * Tries to find the framework name. First looks to see if a value for
 * "EOGeneratorFrameworkName" exists in the model's -userInfo dictionary, and if
 * not, looks at the model's -path to find the framework the .eomodeld is in.
 * Both PB.project files and path components ending in ".framework" are looked
 * for, so it should work for .eomodels insided installed frameworks or in a
 * development tree. If the framework name could not be found, returns nil.
 * Several other custom methods use this information to generate \#import
 * statements better. If all your EO classes are in the same framework, this
 * doesn't make much difference, but if there are EOModels in several different
 * frameworks with interrelationships, these methods can make life a lot easier.
"*/
- (NSString *)frameworkName
{
    NSDictionary  *userInfo = [self userInfo];
    NSFileManager *manager  = [NSFileManager defaultManager];
    NSString      *name;

    name = [userInfo objectForKey:@"EOGeneratorFrameworkName"];

    if (name == nil)
    {
        NSString *fullPath = [self path];
        NSMutableDictionary *mutableInfo = [[userInfo mutableCopy] autorelease];

        if (fullPath != nil && ![fullPath isAbsolutePath])
            fullPath = [[manager currentDirectoryPath] stringByAppendingPathComponent:fullPath];

        while ([fullPath isAbsolutePath])
        {
            NSString *pbPath;

            fullPath = [fullPath stringByDeletingLastPathComponent];
            pbPath = [fullPath stringByAppendingPathComponent:@"PB.project"];

            if ([manager regularFileExistsAtPath:pbPath])
            {
                NSDictionary *project = [NSDictionary dictionaryWithContentsOfFile:pbPath];

                if ([[project objectForKey:@"PROJECTTYPE"] hasSuffix:@"Framework"])
                {
                    name = [project objectForKey:@"PROJECTNAME"];
                    break;
                }
            }
            else if ([[fullPath pathExtension] isEqual:@"framework"])
            {
                name = [[fullPath lastPathComponent] stringByDeletingPathExtension];
                break;
            }
        }

        [mutableInfo setObject:name? name : @"" forKey:@"EOGeneratorFrameworkName"];
        [self setUserInfo:mutableInfo];
    }

    return [name length] > 0 ? name : nil;
}

@end

@implementation EORelationship (EOGenerator)

- (NSString *)objcImportString
{
    return [[self destinationEntity] objcImportStringInRelationToEntity:[self entity]];
}

@end
