/* EOTemplateGenerator.m created by mgentry on Mon 14-Jun-1999 */
/*-
 * Copyright (c) 2000
 *      Blacksmith, Inc.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Blacksmith, Inc.
 * 4. The name Blacksmith may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY BLACKSMITH ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL BLACKSMITH BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#import "EOTemplateGenerator.h"
#import <EOAccess/EOModel.h>
#import <EOModeler/EOModelExtensions.h>
#import <MiscMerge.subproj/MiscMergeTemplate.h>
#import <MiscMerge.subproj/MiscMergeCommandBlock.h>
#import <MiscMerge.subproj/MiscMergeEngine.h>
#import <MiscFoundation.subproj/NSString+MiscAdditions.h>
#import "FoundationAdditions.h"
#import "SystemType.h"

/* Writes to stdout */
void EOVPrintf(NSString *format, va_list arguments)
{
    NSString *logString = [[NSString alloc] initWithFormat:format arguments:arguments];

    fwrite([logString cString], 1, [logString cStringLength], stdout);
    if (![logString hasSuffix:@"\n"]) fputc('\n', stdout); //hrm

    [logString release];
}

void EOPrintf(NSString *format, ...)
{
    va_list arguments;

    va_start(arguments, format);
    EOVPrintf(format, arguments);
    va_end(arguments);
}

@implementation EOTemplateGenerator

- init
{
    BOOL isEOF4 = (NSClassFromString(@"EOEvent") != nil);  //EOEvent added in 4.5

    [super init];

    javaMode = NO;
    verboseMode = NO;
    destination = @".";

    mainHeaderTemplate = isEOF4? @"ObjCHeader.eotemplate" : @"ObjCHeaderEOF3.eotemplate";
    mainSourceTemplate = isEOF4? @"ObjCSource.eotemplate" : @"ObjCSourceEOF3.eotemplate";
    stubHeaderTemplate = @"ObjCSubclassHeader.eotemplate";
    stubSourceTemplate = @"ObjCSubclassSource.eotemplate";
#ifdef EOF2_ONLY
    mainJavaTemplate   = @"JavaSourceEOF2.eotemplate";
    stubJavaTemplate   = @"JavaSubclassSourceEOF2.eotemplate";
#else
    mainJavaTemplate   = isEOF4? @"JavaSource.eotemplate" : @"JavaSourceEOF3.eotemplate";
    stubJavaTemplate   = @"JavaSubclassSource.eotemplate";
#endif

    models = [[NSMutableArray alloc] init];
    searchPaths = [[NSMutableArray alloc] init];
    entityNames = [[NSMutableArray alloc] init];
    userVariables = [[NSMutableDictionary alloc] init];

    [self processCommandLineArguments];

    return self;
}

- (void)displayHelpAndExit:(int)exitStatus
{
    const char *appName = [[[NSProcessInfo processInfo] processName] cString];

    printf("Usage: %s -model eoModelName.eomodeld [entities ...]\n\n", appName);
    printf("Options:\n"
           "  -java                    Output Java source code instead of ObjC\n"
           "  -model <model>           Output entities from <model>\n"
           "  -refmodel <model>        Load model, do not generate code\n"
           "  -destination <dir>       Output directory for generated files\n"
           "  -subclassDestination     <dir> Directory for stub subclass files\n"
           "  -define-<key> <value>    Add <key> to runtime variable list\n"
           "  -templatedir <dir>       Add <dir> to template search path\n"
           "  -sourceTemplate <file>   Use different template file\n"
           "  -headerTemplate             \"\n"
           "  -subclassSourceTemplate     \"\n"
           "  -subclassHeaderTemplate     \"\n"
           "  -javaTemplate               \"\n"
           "  -subclassJavaTemplate       \"\n"
           "  -verbose                 Verbose output\n"
           "  -version                 Print version number and exit\n"
           "  -help                    This help\n");

    exit(exitStatus);
}

- (void)addModelFromPath:(NSString *)path shouldProcess:(BOOL)shouldProcess
{
    EOModel *model = nil;

    NS_DURING
        model = [[EOModel alloc] initWithContentsOfFile:path];
    NS_HANDLER
        EOPrintf(@"Error loading eomodel file '%@': %@",
            path, [localException reason]);
        return;
    NS_ENDHANDLER

    if ([[EOModelGroup defaultGroup] modelNamed:[model name]] == nil)
        [[EOModelGroup defaultGroup] addModel:model];

    if (model && shouldProcess)
        [models addObject:model];

    [model release];
}

- (void)processCommandLineArguments
{
    NSEnumerator *argEnum = [[[NSProcessInfo processInfo] arguments] objectEnumerator];
    NSString *currArg;

    [argEnum nextObject]; //skip argv[0]

    while (currArg = [argEnum nextObject])
    {
        if ([@"-verbose" hasPrefix:currArg])
        {
            verboseMode = YES;
        }
        else if ([@"-help" hasPrefix:currArg])
        {
            [self displayHelpAndExit:0];
        }
        else if ([@"-model" hasPrefix:currArg])
        {
            [self addModelFromPath:[argEnum nextObject] shouldProcess:YES];
        }
        else if ([@"-java" hasPrefix:currArg])
        {
            javaMode = YES;
        }
        else if ([@"-destination" hasPrefix:currArg])
        {
            destination = [[argEnum nextObject] retain];
        }
        else if ([@"-subclassDestination" hasPrefix:currArg])
        {
            subclassDestination = [[argEnum nextObject] retain];
        }
        else if ([@"-templatedir" hasPrefix:currArg] || [@"-Templatedir" hasPrefix:currArg])
        {
            NSString *path = [argEnum nextObject];
            if (![[NSFileManager defaultManager] directoryExistsAtPath:path])
                EOPrintf(@"Warning: Template search directory %@ does not exist", path);
            [searchPaths addObject:path];
        }
        else if ([@"-sourceTemplate" hasPrefix:currArg])
            mainSourceTemplate = [[argEnum nextObject] retain];
        else if ([@"-headerTemplate" hasPrefix:currArg])
            mainHeaderTemplate = [[argEnum nextObject] retain];
        else if ([@"-subclassSourceTemplate" hasPrefix:currArg])
            stubSourceTemplate = [[argEnum nextObject] retain];
        else if ([@"-subclassHeaderTemplate" hasPrefix:currArg])
            stubHeaderTemplate = [[argEnum nextObject] retain];
        else if ([@"-javaTemplate" hasPrefix:currArg])
            mainJavaTemplate = [[argEnum nextObject] retain];
        else if ([@"-subclassJavaTemplate" hasPrefix:currArg])
            stubJavaTemplate = [[argEnum nextObject] retain];

        else if ([@"-refmodel" hasPrefix:currArg])
        {
            [self addModelFromPath:[argEnum nextObject] shouldProcess:NO];
        }
        else if ([@"-version" hasPrefix:currArg])
        {
            EOPrintf(@"EOGenerator version 1.0");
            exit(0);
        }
        else if ([currArg hasPrefix:@"-define-"])
        {
            NSString *key = [currArg substringFromIndex:8];
            NSString *value = [argEnum nextObject];

            if ([key length] > 0 && value)
                [userVariables setObject:value forKey:key];
        }
        else if ([currArg hasPrefix:@"-"])
        {
            EOPrintf(@"Unknown option: %@", currArg);
            [self displayHelpAndExit:1];
        }
        else
        {
            if ([currArg containsString:@".eomodel"])
                [self addModelFromPath:currArg shouldProcess:YES];
            else
                [entityNames addObject:currArg];
        }		
    }

    if ([models count] == 0)
    {
        EOPrintf(@"Unspecified EOModel file.  Use -h to see help.");
        exit(1);
    }

    if (![[NSFileManager defaultManager] directoryExistsAtPath:destination])
    {
        EOPrintf(@"Destination directory (%@) does not exist.", destination);
        exit(1);
    }

    if (subclassDestination &&
        ![[NSFileManager defaultManager] directoryExistsAtPath:subclassDestination])
    {
        EOPrintf(@"Subclass destination directory (%@) does not exist.", subclassDestination);
        exit(1);
    }

    [self addDefaultSearchPaths];
}

- (void)addDefaultSearchPaths
{
    NSArray *libraryPaths;
    int i;

#ifdef OPENSTEP_ONLY
    libraryPaths = NSStandardLibraryPaths();
#else
    libraryPaths = NSSearchPathForDirectoriesInDomains(
                        NSAllLibrariesDirectory,
                        NSUserDomainMask|NSLocalDomainMask|NSNetworkDomainMask,
                        YES);
#endif

    for (i=0; i<[libraryPaths count]; i++)
    {
        NSString *libPath = [libraryPaths objectAtIndex:i];
        [searchPaths addObject:[libPath stringByAppendingPathComponent:@"EOGenerator"]];
    }

    [searchPaths addObject:[[NSBundle mainBundle] bundlePath]];

    if (verboseMode)
        EOPrintf(@"Search path is:\n%@", searchPaths);
}

- (NSArray *)searchPaths
{
    return searchPaths;
}


/*** GENERATION ***/

- (void)generate
{
    EOEntity *entity;
    NSString *entityName;
    NSEnumerator *enumerator;

    /* Use specified entities instead of entire EOModel(s) if > 0 */
    if ([entityNames count] > 0)
    {
        enumerator = [entityNames objectEnumerator];
        while (entityName = [enumerator nextObject])
        {
            entity = [[EOModelGroup defaultGroup] entityNamed:entityName];
            if (entity == nil)
            {
                EOPrintf(@"Entity %@ not found in model, skipping.", entityName);
            }
            else
            {
                [self generateEntity:entity];
            }
        }
    }
    else
    {
        NSEnumerator *modelEnum = [models objectEnumerator];
        EOModel *model;

        while (model = [modelEnum nextObject])
        {
            enumerator = [[model entities] objectEnumerator];
            while (entity = [enumerator nextObject])
            {
                [self generateEntity:entity];
            }

        }
    }
}

- (void)generateEntity:(EOEntity *) entity
{
    NSAutoreleasePool *pool;

    if ([[entity className] hasSuffix:@"EOGenericRecord"])
        return;

    pool = [[NSAutoreleasePool alloc] init];

    if (javaMode)
    {
        [self generateEntity:entity withTemplate:mainJavaTemplate asPrimary:YES extension:@"java"];
        [self generateEntity:entity withTemplate:stubJavaTemplate asPrimary:NO  extension:@"java"];
    }
    else
    {
        [self generateEntity:entity withTemplate:mainHeaderTemplate asPrimary:YES extension:@"h"];
        [self generateEntity:entity withTemplate:mainSourceTemplate asPrimary:YES extension:@"m"];

        [self generateEntity:entity withTemplate:stubHeaderTemplate asPrimary:NO extension:@"h"];
        [self generateEntity:entity withTemplate:stubSourceTemplate asPrimary:NO extension:@"m"];
    }

    [pool release];
}


- (NSString *)filenameForEntity:(EOEntity *)entity
        primary:(BOOL)primary
        extension:(NSString *)fileExtension
{
    NSString *baseFilename;
    NSString *className = [entity className];

    /* If it is Java, it could have the package path prefixed before it, so we
       must lose it */
    if (javaMode)
    {
        NSRange dotRange = [className rangeOfString:@"." options:NSBackwardsSearch];
        if (dotRange.length > 0)
            className = [className substringFromIndex:NSMaxRange(dotRange)];
    }

    baseFilename = [NSString stringWithFormat:@"%@%@", primary ? @"_" : @"", className];
    baseFilename = [baseFilename stringByAppendingPathExtension:fileExtension];

    /* Now put the path in front of the filename. */
    if (!primary && subclassDestination)
        return [subclassDestination stringByAppendingPathComponent:baseFilename];
    else	
        return [destination stringByAppendingPathComponent:baseFilename];
}

- (MiscMergeTemplate *)templateForName:(NSString *)name
{
    static NSMutableDictionary *templateDict = nil;
    MiscMergeTemplate *template = [templateDict objectForKey:name];

    if (templateDict == nil)
        templateDict = [[NSMutableDictionary alloc] initWithCapacity:4];

    if (template == nil)
    {	
        NSFileManager *fileManager = [NSFileManager defaultManager];
        NSString *filename = nil;

        if ([fileManager regularFileExistsAtPath:name])
            filename = name;
        else
            filename = [fileManager findFile:name inSearchPath:[self searchPaths]];

        if (filename == nil)
        {
            EOPrintf(@"Could not find template file '%@' in search path", name);
        }
        else
        {
            if (verboseMode)
                EOPrintf(@"Using template at path %@", filename);
            template = [[MiscMergeTemplate alloc] init];
            [template setStartDelimiter:@"<$" endDelimiter:@"$>"];
            [template parseContentsOfFile:filename];
            [templateDict setObject:template forKey:name];
            [template release];
        }
    }

    return template;
}

- (void)generateEntity:(EOEntity *)entity
		withTemplate:(NSString *)templateName
		asPrimary:(BOOL)primary
		extension:(NSString *)fileExtension
{
    NSFileManager     *fileManager = [NSFileManager defaultManager];
    MiscMergeEngine   *engine;
    NSString          *newVersion;
    NSString          *oldVersion;
    NSString          *outputFilename;
    MiscMergeTemplate *template;
	NSEnumerator      *userVariableEnum;
	NSString          *userKey;

    outputFilename = [self filenameForEntity:entity primary:primary extension:fileExtension];

    if (verboseMode)
        EOPrintf(@"Generating class file %@ from template %@", outputFilename, templateName);
    
    /* 
     * If it is not a primary output file (an always generated _ClassName
     * entity), then check to see if the secondary (ClassName, which is a
     * subclass of _ClassName) is present. If the secondary file is present, we
     * DO NOT generate a new file. We only generate this once and the user is
     * free to edit the file afterwards to add business logic.
     */
    if (!primary && [fileManager regularFileExistsAtPath:outputFilename])
    {
        if (verboseMode)
        {
            EOPrintf(@"Subclass %@ already exists at %@; not overwritten",
                     [entity className], outputFilename);
        }

        return;  /* we're done. */
    }

	template = [self templateForName:templateName];;
    engine   = [[MiscMergeEngine alloc] initWithTemplate:template];

	/* Add variables defined on the command line */
    userVariableEnum = [userVariables keyEnumerator];
    while (userKey = [userVariableEnum nextObject])
        [engine setEngineValue:[userVariables objectForKey:userKey] forKey:userKey];

    newVersion = [engine executeWithObject:entity sender:nil];
    oldVersion = [NSString stringWithContentsOfFile:outputFilename];

    if (oldVersion == nil || ![oldVersion isEqualToString:newVersion])
    {
		NSData *fileData = [newVersion dataUsingEncoding:[NSString defaultCStringEncoding]];
        if (![fileData writeToFile:outputFilename atomically:YES])
        {
            EOPrintf(@"ERROR: Could not write file '%@'", outputFilename);
        }
        else
        {
            /*
             * If we regenerate the main class, update the timestamp on the
             * subclass so it will be recompiled.
             */
            if (primary)
            {
                NSString *subclassFile = [self filenameForEntity:entity
                                                         primary:NO
                                                       extension:fileExtension];

                if ([fileManager regularFileExistsAtPath:subclassFile])
                    [fileManager touchPath:subclassFile];
            }

            if (verboseMode) {
                EOPrintf(@"Generated output file %@ for entity %@",
                         outputFilename, [entity name]);
            }
            else {
                EOPrintf(@"Generated %@\n", outputFilename);
            }
        }
    }
    else
    {
        if (verboseMode)
            EOPrintf(@"Output file %@ is identical to generated file; not written",
				outputFilename);
    }

    [engine release];
}

@end
