/* 
	AppController.m created by dirk on Fri 10-Sep-1999
*/
#import "AppController.h"
#import "Mailbox.h"
#import "globals.h"
#import "NSMenu+Removing.h"

@interface AppController (PrivateAPI)

- (void)initStatusItem;
- (NSImage *)scaledImage;
- (NSMutableArray *)arrayWithMailboxes:(NSArray *)watchlist;

@end

@implementation AppController

- (id)init
{
    NSArray			*watchlist;
    int				timerInterval;
    NSUserDefaults	*defaults;
    
    [super init];
    [self initStatusItem];
   
    defaults = [NSUserDefaults standardUserDefaults];
    watchlist = [defaults arrayForKey:gvMailboxList];
    if (watchlist == nil)
    {
        // no configuration yet, scan all mailboxes in User's home as default
        watchlist = [Mailbox mailboxesForDirectoryAtPath:[Mailbox mailViewerMailboxPath]];
    }
    ivMailboxes = [self arrayWithMailboxes:watchlist];
    ivMailboxes = [[ivMailboxes sortedArrayUsingSelector:@selector(compare:)] mutableCopy];

    timerInterval = [defaults integerForKey:gvUpdateTimerInterval];
    if (timerInterval == 0)
    {
        // use the default configuration
        timerInterval = UPDATE_TIMER_INTERVAL_DEFAULT;
        [defaults setInteger:timerInterval forKey:gvUpdateTimerInterval];
        [defaults synchronize];
    }

    if ([defaults boolForKey:gvCheckOnActivate] == YES)
    {
        [ivMailboxes makeObjectsPerformSelector:@selector(checkTOCForUnreadMail)];
    }

    [self setTimer:[NSTimer scheduledTimerWithTimeInterval:timerInterval target:self selector:@selector(updateTocs:) userInfo:nil repeats:YES]];

    return self;
}

- (void)initStatusItem
{
    NSImage	*image;

    // set ourselves up in the status bar
    image = [self scaledImage];
    ivStatusItem = [[[NSStatusBar systemStatusBar] statusItemWithLength:[image size].width + 10] retain];
    [ivStatusItem setImage:image];

    // build this menu only once and swap only the menuItems
    [ivStatusItem setMenu:[[[NSMenu allocWithZone:[NSMenu menuZone]] initWithTitle:@"Biff"] autorelease]];

    // hide the statusItem
    [ivStatusItem setLength:0.1];
}

- (void)dealloc
{
    [ivMailboxes release];
    [ivStatusItem release];
    [ivTimer release];
    [super dealloc];
}

//------------------------------------------------------------------------------
// App's delegate methods
//------------------------------------------------------------------------------
- (void)applicationWillTerminate:(NSNotification *)aNotification
{
    [self saveWatchedMailboxes];
}

//------------------------------------------------------------------------------
// Helper methods
//------------------------------------------------------------------------------
- (NSMutableArray *)arrayWithMailboxes:(NSArray *)watchlist
{
    NSEnumerator	*mboxEnum;
    NSString		*element;
    NSMutableArray	*returnArray;

    returnArray = [NSMutableArray arrayWithCapacity:[watchlist count]];
    mboxEnum = [watchlist objectEnumerator];
    while ((element = [mboxEnum nextObject]) != nil)
    {
        Mailbox	*newMailbox;

        newMailbox = [[[Mailbox alloc] initWithMailboxPath:element] autorelease];
        if (newMailbox != nil)
        {
            [returnArray addObject:newMailbox];
        }
        else
        {
            NSRunAlertPanel(@"Invalid Mailbox", [NSString stringWithFormat:@"The mailbox %@ does not exist (removing it from watchlist)", element], nil, nil, nil);
		}
    }
    return returnArray;
}

- (void)updateTocs:(NSTimer *)timer
{
    NSEnumerator	*enumerator;
    Mailbox			*mailbox;
    BOOL			statusItemIsVisible = NO;

#ifdef DEBUG
    NSLog(@"updateTocs ...");
#endif
    // You have to remove all items from an existing NSMenu. Simply providing a new
    // NSMenu instance to NSStatusItem crashes the app :-(
    [[ivStatusItem menu] removeAllItems];
    
    enumerator = [ivMailboxes objectEnumerator];
    while ((mailbox = [enumerator nextObject]) != nil)
    {
        if ([mailbox hasNewMail] == YES)
        {
            NSMenuItem *item;

            item = [[NSMenuItem allocWithZone:[NSMenu menuZone]] initWithTitle:[mailbox relativePath] action:NULL keyEquivalent:@""];
            [item setRepresentedObject:mailbox];
            [item setTarget:self];
            [item setAction:@selector(openMailbox:)];
            [[ivStatusItem menu] addItem:item];
            [item release];
            statusItemIsVisible = YES;
        }
    }
    
    if (statusItemIsVisible == YES)
    {
        [ivStatusItem setLength:[[ivStatusItem image] size].width + 10];
    }
    else
    {
        [ivStatusItem setLength:0.1];
    }
}

- (void)openMailbox:(NSMenuItem *)sender
{
    NSAssert1([sender isKindOfClass:[NSMenuItem class]], @"selector %@ must be invoked with an NSMenuItem as sender", NSStringFromSelector(_cmd));
    [[NSWorkspace sharedWorkspace] openFile:[[sender representedObject] mailboxPath] withApplication:nil andDeactivate:YES];
}

- (void)saveWatchedMailboxes
{
    NSEnumerator	*arrayEnum;
    Mailbox			*mailbox;
    NSMutableArray	*saveArray;

    saveArray = [NSMutableArray array];
    arrayEnum = [ivMailboxes objectEnumerator];
    while ((mailbox = [arrayEnum nextObject]) != nil)
    {
        [saveArray addObject:[mailbox mailboxPath]];
    }

    [[NSUserDefaults standardUserDefaults] setObject:saveArray forKey:gvMailboxList];
    [[NSUserDefaults standardUserDefaults] synchronize];
}

//------------------------------------------------------------------------------
// Accessor methods
//------------------------------------------------------------------------------
- (NSImage *)scaledImage
{
    NSImage			*image;

    image = [NSImage imageNamed:@"Envelope"];
    NSAssert(image != nil, @"image Envelope.tiff not found");
    
    [image setScalesWhenResized:YES];
    [image setSize:NSMakeSize(22.0, 22.0)];
    [image recache];
    return image;
}

- (NSMutableArray *)mailboxArray;
{
    return ivMailboxes;
}

- (void)addToMailboxArray:(Mailbox *)newMailbox
/*"
Add %newMailbox and make sure the array is sorted afterwards.
"*/
{
    NSMutableArray	*newMboxArray;
    
    [ivMailboxes addObject:newMailbox];
    newMboxArray = [[ivMailboxes sortedArrayUsingSelector:@selector(compare:)] mutableCopy];
    [ivMailboxes release];
    ivMailboxes = newMboxArray;
}

- (void)setTimer:(NSTimer *)tmp;
{
    id old = ivTimer;
    [ivTimer invalidate];
    ivTimer = [tmp retain];
    [old release];
}

@end