//
//  MKNDockling.m
//  MulleNewz
//
//  Created by znek on Fri Jul 20 2001.
//  $Id: MKNDockling.m,v 1.6 2001/08/10 13:08:01 znek Exp $
//
//  Copyright (c) 2001 by Marcus Mller <znek@mulle-kybernetik.com>.
//  All rights reserved.
//
//  Permission to use, copy, modify and distribute this software and its documentation
//  is hereby granted under the terms of the GNU General Public License, version 2
//  as published by the Free Software Foundation, provided that both the copyright notice
//  and this permission notice appear in all copies of the software, derivative works or
//  modified versions, and any portions thereof, and that both notices appear in supporting
//  documentation, and that credit is given to Marcus Mller in all documents and publicity
//  pertaining to direct or indirect use of this code or its derivatives.
//
//  This is free software; you can redistribute and/or modify it under
//  the terms of the GNU General Public License, version 2 as published by the Free
//  Software Foundation. Further information can be found on the project's web pages
//  at http://www.mulle-kybernetik.com/software/MulleNewz
//
//  THIS IS EXPERIMENTAL SOFTWARE AND IT IS KNOWN TO HAVE BUGS, SOME OF WHICH MAY HAVE
//  SERIOUS CONSEQUENCES. THE COPYRIGHT HOLDER ALLOWS FREE USE OF THIS SOFTWARE IN ITS
//  "AS IS" CONDITION. THE COPYRIGHT HOLDER DISCLAIMS ANY LIABILITY OF ANY KIND FOR ANY
//  DAMAGES WHATSOEVER RESULTING DIRECTLY OR INDIRECTLY FROM THE USE OF THIS SOFTWARE
//  OR OF ANY DERIVATIVE WORK.
//---------------------------------------------------------------------------------------


#import "MKNDockling.h"
#import <EDCommon/EDCommon.h>
#import "NSString+XMLExtensions.h"
#import "RSSCache.h"
#import "RSSParser.h"
#import "RSSElements.h" // high level RSS elements


@interface MKNDockling (PrivateAPI)
- (void)logWithFormat:(NSString *)format, ...;
- (void)_setup;
- (void)_setupMenu;
- (void)_setNormalImage;
- (void)_setClockImageWithAngle:(float)angle;

- (NSMenuItem *)_itemForDocument:(id)document;
- (void)_insertItemForDocumentDescription:(NSDictionary *)documentDescription atIndex:(int)itemIndex;
- (NSMenuItem *)_findExistingItemForDocumentDescription:(NSDictionary *)documentDescription;
- (void)updateMenuForRSSDocumentDescription:(NSDictionary *)documentDescription preferredIndex:(int)itemIndex;
- (IBAction)openURLForSender:(id)sender;

- (void)addJobForURL:(NSString *)sourceURL preferredIndex:(int)itemIndex;
- (void)processPendingJobs;
- (void)_processPendingJobs;

- (void)_retrieveDefaults;
- (void)_synchronizeDefaults;
@end


@implementation MKNDockling

////////////////////////////////////////////////////
//
//  INIT & DEALLOC
//
////////////////////////////////////////////////////


-(id)initWithBundle:(NSBundle *)bundle window:(NSWindow *)window
{
	[super initWithBundle:bundle window:window];
	[self logWithFormat:@"setting up at: %@", [NSCalendarDate date]];
	[self _setup];
	return self;
}

-(void)release
{
    if([self retainCount] == 1 && updateTimer != nil)
        [self stopUpdateTimer];
    [super release];
}

- (void)dealloc
{
    [self logWithFormat:@"Deallocing at: %@", [NSCalendarDate date]];
    [self _synchronizeDefaults];
    [docklingImageView release];
    [rssSources release];
    [jobQueue release];
    [jobLock release];
    [super dealloc];
}


////////////////////////////////////////////////////
//
//  SETUP
//
////////////////////////////////////////////////////


- (void)_setup
{
    [RSSCache setBundle:[self bundle]];
    [NSString setBundle:[self bundle]];
    rssSources = [[NSMutableArray alloc] init];
    jobQueue = [[NSMutableArray alloc] init];
    jobLock = [[NSLock alloc] init];

    NS_DURING

        [self _retrieveDefaults];
        [self _setNormalImage];
        [self _setupMenu];
        [self checkRSSSources];
        if([self updateInterval] > 0.0)
            [self startUpdateTimer];

    NS_HANDLER
    
        [self logWithFormat:@"Caught exception: %@", [localException reason]];
    
    NS_ENDHANDLER
}

- (NSImage *)normalDocklingImage
{
    if(normalDocklingImage == nil)
         normalDocklingImage = [[NSImage alloc] initWithContentsOfFile:[[self bundle] pathForResource:@"MulleNewz" ofType:@"tiff"]];
    return normalDocklingImage;
}

- (void)_setNormalImage
{
    [self setDocklingImage:[self normalDocklingImage]];
}

- (void)_setClockImageWithAngle:(float)angle
{
    NSImage *image;
    NSBezierPath *bezierPath;
    NSPoint centerPoint;
    float _angle;

    angle = ((int)angle) % 360;

    if(angle < 90)
        _angle = 90 - angle;
    else
        _angle = 360 - (angle - 90);

    image = [[[self normalDocklingImage] copy] autorelease];
    bezierPath = [NSBezierPath bezierPath];

    centerPoint = NSMakePoint(64, 64+8);
    [bezierPath moveToPoint:centerPoint];
    [bezierPath appendBezierPathWithArcWithCenter:centerPoint radius:64 * 0.70 startAngle:90 endAngle:_angle clockwise:YES];
    [bezierPath lineToPoint:centerPoint];

    [image lockFocus];
    [[NSColor colorWithDeviceWhite:NSBlack alpha:0.25] set];
    [bezierPath stroke];
    [[NSColor colorWithDeviceWhite:NSBlack alpha:0.2] set];
    [bezierPath fill];
    [image unlockFocus];
    [self setDocklingImage:image];
}


- (void)setDocklingImage:(NSImage *)anImage
{
    if(docklingImageView == nil)
    {
        docklingImageView = [[NSImageView alloc] initWithFrame:NSMakeRect(0, 0, 128, 128)];
        [docklingImageView setImageAlignment:NSImageAlignCenter];
        [docklingImageView setImageFrameStyle:NSImageFrameNone];
        [docklingImageView setImageScaling:NSScaleToFit];
    
        [[self window] setBackgroundColor:[NSColor clearColor]];
        [[self window] setContentView:docklingImageView];
    }
    [docklingImageView setImage:anImage];
    [[self window] display];
}


////////////////////////////////////////////////////
//
//  RSS DOCUMENTS
//
////////////////////////////////////////////////////


- (NSArray *)rssSources
{
	return rssSources;
}

- (void)checkRSSSources
{
    NSEnumerator *rssSourceURLEnum;
    NSString *aSourceURL;
    
    rssSourceURLEnum = [[self rssSources] objectEnumerator];
    while((aSourceURL = [rssSourceURLEnum nextObject]) != nil)
        [self addJobForURL:aSourceURL preferredIndex:NSNotFound];

    // now perform the jobs. this will take care of visualizing the process.
    [self processPendingJobs];
}


////////////////////////////////////////////////////
//
//  JOBS/FETCHES
//
////////////////////////////////////////////////////


- (void)addJobForURL:(NSString *)sourceURL preferredIndex:(int)itemIndex
{
    EDObjectPair *job;
    
    job = [EDObjectPair pairWithObjects:sourceURL :[NSNumber numberWithInt:itemIndex]];
    [jobLock lock];
    [jobQueue addObject:job];
    [jobLock unlock];
}

- (void)processPendingJobs
{
    [jobLock lock];
#if 0
    [NSThread detachNewThreadSelector:@selector(_processPendingJobs)];
#else
    [self _processPendingJobs];
#endif
    [jobLock unlock];
}

- (void)_processPendingJobs
{
    int count;
    
    count = [jobQueue count];
    if(count > 0)
    {
        EDObjectPair *job;
        float angleIncrementPerEntry = 359 / count;
        float angle = angleIncrementPerEntry;
        int i;

        for(i = 0; i < count; i++)
        {
            NSDictionary *documentDescription;

            job = [jobQueue objectAtIndex:i];
            [self _setClockImageWithAngle:angle];
            documentDescription = [[RSSCache sharedCache] documentDescriptionForURL:[job firstObject]];
            [self updateMenuForRSSDocumentDescription:documentDescription preferredIndex:[[job secondObject] intValue]];
            angle += angleIncrementPerEntry;
        }
        [self _setClockImageWithAngle:359];
        [jobQueue removeAllObjects];
        [self _setNormalImage];
    }
}

////////////////////////////////////////////////////
//
//  MENU STUFF
//
////////////////////////////////////////////////////


- (void)updateMenuForRSSDocumentDescription:(NSDictionary *)documentDescription preferredIndex:(int)itemIndex
{
    NSMenuItem *anItem;
    
    [self logWithFormat:@"UpdatingRSSSource: %@", [[documentDescription objectForKey:RSSDocumentURLKey] absoluteString]];
    
    anItem = [self _findExistingItemForDocumentDescription:documentDescription];
    if(anItem != nil)
    {
        #warning * we shouldn't remove items in all cases!
        // DO ME:
        // if a source was available at some point and now has become unavailable,
        // we should place a note somewhere indicating that this info is outdated,
        // but shouldn't delete it completely
        itemIndex = [[self menu] indexOfItem:anItem];
        [[self menu] removeItemAtIndex:itemIndex];
    }

    // this will construct all we need
    [self _insertItemForDocumentDescription:documentDescription atIndex:itemIndex];
    // announce the change to the dockling server
    [self setMenuChanged:YES];
    // ... and finally synchronize the user defaults
    [self _synchronizeDefaults];
}


- (void)_setupMenu
{
    if([NSBundle loadNibNamed:@"MulleNewz" owner:self] == NO)
        [NSException raise:NSGenericException format:@"Error loading MulleNewz.nib"];
//    [self logWithFormat:@"menu = %@", [self menu]];
}

- (NSMenuItem *)_itemForDocument:(id)document
{
    NSMenuItem *anItem;

    anItem = [[[NSMenuItem alloc] initWithTitle:[document title] action:@selector(openURLForSender:) keyEquivalent:@""] autorelease];
    [anItem setTarget:self];
    [anItem setRepresentedObject:document];
    return anItem;
}

- (void)_insertItemForDocumentDescription:(NSDictionary *)documentDescription atIndex:(int)itemIndex
{
	RSSDocument *rssDocument;
	NSMenuItem *anItem;
	NSMenu *aSubMenu;
	NSString *title;
	
	rssDocument = [documentDescription objectForKey:RSSDocumentObjectKey];
	if(rssDocument != nil)
	{
		NSEnumerator *rssItemEnum;
		RSSItem *rssItem;

		anItem = [self _itemForDocument:rssDocument];
		if(itemIndex != NSNotFound)
			[[self menu] insertItem:anItem atIndex:itemIndex];
		else
			[[self menu] addItem:anItem];
		
		aSubMenu = [[[NSMenu alloc] initWithTitle:[[documentDescription objectForKey:RSSDocumentURLKey] absoluteString]] autorelease];
		[[self menu] setSubmenu:aSubMenu forItem:anItem];
		
		#warning ** localize this!
		anItem = [[[NSMenuItem alloc] initWithTitle:[NSString stringWithFormat:@"Update %@", [[rssDocument channel] title]] action:@selector(updateRSSSourceForSender:) keyEquivalent:@""] autorelease];
		[anItem setTarget:self];
		[anItem setRepresentedObject:[documentDescription objectForKey:RSSDocumentURLKey]];
		[aSubMenu addItem:anItem];
		[aSubMenu addItem:[NSMenuItem separatorItem]];
		
		rssItemEnum = [[rssDocument items] objectEnumerator];
		while((rssItem = [rssItemEnum nextObject]) != nil)
		{
			anItem = [self _itemForDocument:rssItem];
			[aSubMenu addItem:anItem];
		}
	}
	else
	{
		title = [[documentDescription objectForKey:RSSDocumentURLKey] absoluteString];

		anItem = [[[NSMenuItem alloc] initWithTitle:title action:@selector(openURLForSender:) keyEquivalent:@""] autorelease];
		[anItem setTarget:self];
		[anItem setState:NSMixedState]; // visually indicate problem
		[anItem setRepresentedObject:[documentDescription objectForKey:RSSDocumentObjectKey]];

		if(itemIndex != NSNotFound)
			[[self menu] insertItem:anItem atIndex:itemIndex];
		else
			[[self menu] addItem:anItem];

		aSubMenu = [[[NSMenu alloc] initWithTitle:title] autorelease];
		[[self menu] setSubmenu:aSubMenu forItem:anItem];
		
		#warning ** localize this!
		anItem = [[[NSMenuItem alloc] initWithTitle:[NSString stringWithFormat:@"Update %@", title] action:@selector(updateRSSSourceForSender:) keyEquivalent:@""] autorelease];
		[anItem setTarget:self];
		[anItem setRepresentedObject:[documentDescription objectForKey:RSSDocumentURLKey]];
		[aSubMenu addItem:anItem];
		[aSubMenu addItem:[NSMenuItem separatorItem]];

		anItem = [[[NSMenuItem alloc] initWithTitle:[NSString stringWithFormat:@"Error: %@", [documentDescription objectForKey:RSSDocumentErrorKey]] action:NULL keyEquivalent:@""] autorelease];
		[anItem setEnabled:NO];
		[aSubMenu addItem:anItem];
	}
}

- (NSMenuItem *)_findExistingItemForDocumentDescription:(NSDictionary *)documentDescription
{
    NSString *documentURL;
    int count = [[self menu] numberOfItems], i = 3;

    documentURL = [[documentDescription objectForKey:RSSDocumentURLKey] absoluteString];

    // the next step is to iterate over all menuitems and compare the submenus' titles with that url
    for(; i < count; i++)
    {
        NSMenuItem *menuItem = [[self menu] itemAtIndex:i];
        if([[[menuItem submenu] title] isEqualToString:documentURL])
            return menuItem;
    }
    return nil;
}


////////////////////////////////////////////////////
//
//  ACTIONS
//
////////////////////////////////////////////////////


- (IBAction)openPreferences:(id)sender
{
    if(prefWindow == nil)
    {
        if([NSBundle loadNibNamed:@"Preferences" owner:self] == NO)
            [NSException raise:NSGenericException format:@"Error loading Preferences.nib"];
    }
    [rssSourcesTableView reloadData];
    [updateIntervalField setFloatValue:[self updateInterval] / 60];
    [prefWindow display];
    [prefWindow makeKeyAndOrderFront:self];
}

// Preferences action
- (IBAction)addRSSSource:(id)sender
{
	[self logWithFormat:@"addRSSSource:"];

	#warning !! FIXME (GUI update problems)
	[rssSources addObject:@"NewRSSSource"];
	[rssSourcesTableView reloadData];
	[rssSourcesTableView selectRow:[rssSources count] - 1 byExtendingSelection:NO];
//	[[[rssSourcesTableView superview] superview] display];
//	[rssSourcesTableView display];
	[prefWindow display];
	[self _synchronizeDefaults];
}

- (IBAction)removeSelectedRSSSources:(id)sender
{
	NSArray *indices;
	NSEnumerator *iEnum;
	NSNumber *removeIndex;

	#warning !! FIXME (GUI update problems)

NS_DURING

	indices = [[rssSourcesTableView selectedRowEnumerator] allObjects];
	[self logWithFormat:@"removeSelectedRSSSources: SHOULD ix=%@ count=%d", indices, [rssSourcesTableView numberOfSelectedRows]];

	indices = [NSArray arrayWithObject:[NSNumber numberWithInt:0]];
	[self logWithFormat:@"removeSelectedRSSSources: INSTEAD ix=%@ count=%d", indices, [rssSourcesTableView numberOfSelectedRows]];

	// Guarantee that indices are sorted in ascending order
	indices = [indices sortedArrayUsingSelector:@selector(compare:)];
	iEnum = [indices reverseObjectEnumerator];

	while((removeIndex = [iEnum nextObject]) != nil)
	{
		[rssSources removeObjectAtIndex:[removeIndex intValue]];
		// remove appropriate entries from menu as well
		[[self menu] removeItemAtIndex:[removeIndex intValue] + 3];
	}

	[rssSourcesTableView reloadData];
//	[[[rssSourcesTableView superview] superview] setNeedsDisplay:YES];
	[rssSourcesTableView setNeedsDisplay:YES];
	[prefWindow display];
	[self _synchronizeDefaults];
NS_HANDLER
	[self logWithFormat:@"removeSelectedRSSSources: Caught exception: %@", [localException reason]];
NS_ENDHANDLER
}


- (IBAction)updateRSSSourceForSender:(id)sender
{
    NSString *sourceURL;
    
    [[RSSCache sharedCache] flushCaches];
    sourceURL = [(NSURL *)[sender representedObject] absoluteString];
    [self addJobForURL:sourceURL preferredIndex:NSNotFound];
    [self processPendingJobs];
}

- (IBAction)updateAllRSSSources:(id)sender
{
	[[RSSCache sharedCache] flushCaches];
	[self checkRSSSources];
}

- (IBAction)setUpdateInterval:(id)sender
{
    NSMutableDictionary *defaults;
    float _interval = [updateIntervalField floatValue];
    
    defaults = [self defaults];
    [defaults setObject:[NSNumber numberWithFloat:_interval] forKey:@"updateInterval"];
    [self storeDefaults:defaults];
    [self _synchronizeDefaults];

    if(_interval > 0.0)
    {
        [self logWithFormat:@"Setting update interval to %f", _interval];
        [self stopUpdateTimer];
        [self startUpdateTimer];
    }
    else
    {
        [self logWithFormat:@"Disabling updates now"];
        [self stopUpdateTimer];
    }
}


// triggered by NSMenuItem
- (IBAction)openURLForSender:(id)sender
{
//    [self logWithFormat:@"openURLForSender called by sender: %@", sender];
    
    NS_DURING

        NSString *urlString = [[sender representedObject] link];

        [self logWithFormat:@"Telling Workspace %@ to open URL: %@", [NSWorkspace sharedWorkspace], urlString];
        if([[NSWorkspace sharedWorkspace] openURL:[NSURL URLWithString:urlString]] == NO)
            [self logWithFormat:@"Failed to open URL: %@", urlString];

    NS_HANDLER
    
        [self logWithFormat:@"INTERNAL ERROR: %@", [localException reason]];

    NS_ENDHANDLER
}


////////////////////////////////////////////////////
//
//  UPDATE TIMER
//
////////////////////////////////////////////////////


- (void)startUpdateTimer
{
	updateTimer = [[NSTimer scheduledTimerWithTimeInterval:[self updateInterval] target:self selector:@selector(updateAllRSSSources:) userInfo:nil repeats:YES] retain];
}

- (void)stopUpdateTimer
{
		[updateTimer invalidate];
		[updateTimer release];
		updateTimer = nil;
}

- (NSTimeInterval)updateInterval
{
	return [[[self defaults] objectForKey:@"updateInterval"] floatValue] * 60;
}



////////////////////////////////////////////////////
//
//  USER PREFERENCES
//
////////////////////////////////////////////////////


- (void)_retrieveDefaults
{
	NSMutableDictionary *defaults;
	
	defaults = [self defaults];
	if([defaults objectForKey:@"_run"] == nil)
	{
		NSDictionary *factorySettings;
		NSString *factoryPlist;
		
		factoryPlist = [NSString stringWithContentsOfFile:[[self bundle] pathForResource:@"FactorySettings" ofType:@"plist"]];
		factorySettings = [factoryPlist propertyList];
		[defaults takeValuesFromDictionary:factorySettings];
		[defaults setObject:[NSCalendarDate date] forKey:@"_run"];
		[self storeDefaults:defaults];
	}
	[rssSources removeAllObjects];
	[rssSources addObjectsFromArray:[defaults objectForKey:@"sources"]];
	[[RSSCache sharedCache] setExpireInterval:[[defaults objectForKey:@"expireInterval"] floatValue] * 60];
}


- (void)_synchronizeDefaults
{
	NSMutableDictionary *defaults;
	
	defaults = [self defaults];
	[defaults setObject:rssSources forKey:@"sources"];
	[defaults setObject:[NSNumber numberWithFloat:[[RSSCache sharedCache] expireInterval] / 60] forKey:@"expireInterval"];
	[self storeDefaults:defaults];
}

- (NSMutableDictionary *)defaults
{
	NSString *identifier;
	NSMutableDictionary *defaults;

	identifier = [[self bundle] bundleIdentifier];
	defaults = [NSMutableDictionary dictionaryWithDictionary:[[NSUserDefaults standardUserDefaults] persistentDomainForName:identifier]];
	return defaults;
}

- (void)storeDefaults:(NSDictionary *)defaults
{
	[[NSUserDefaults standardUserDefaults] setPersistentDomain:defaults forName:[[self bundle] bundleIdentifier]];
}


////////////////////////////////////////////////////
//
//  DELEGATE / DATASOURCE
//
////////////////////////////////////////////////////


// DELEGATE

- (void)tableViewSelectionDidChange:(NSNotification *)notification
{
	[removeRSSSourcesButton setEnabled:[rssSourcesTableView numberOfSelectedRows] > 0];
	[removeRSSSourcesButton setNeedsDisplay:YES];
	[prefWindow display];
}


// DATASOURCE

- (int)numberOfRowsInTableView:(NSTableView *)tableView
{
	return [rssSources count];
}

- (id)tableView:(NSTableView *)tableView objectValueForTableColumn:(NSTableColumn *)tableColumn row:(int)row
{
	return [rssSources objectAtIndex:row];
}


- (void)tableView:(NSTableView *)tableView setObjectValue:(id)object forTableColumn:(NSTableColumn *)tableColumn row:(int)row
{
	[rssSources removeObjectAtIndex:row];
	[rssSources insertObject:object atIndex:row];
	// delete old menu item
	[[self menu] removeItemAtIndex:row + 3]; // Preferences, Update All and Separator are the offset

	// retrieve new description and insert into menu
  [self addJobForURL:object preferredIndex:row + 3];
  [self processPendingJobs];
}


////////////////////////////////////////////////////
//
//  DEBUGGING
//
////////////////////////////////////////////////////


- (void)logWithFormat:(NSString *)format, ...
{
    NSFileHandle *fh;
    va_list   	args;
    NSMutableString	*buffer;

    fh = [NSFileHandle fileHandleForUpdatingAtPath:@"/tmp/mulleNewz.log"];
    [fh seekToEndOfFile];

    va_start(args, format);
    buffer = [[NSMutableString alloc] initWithFormat:format arguments:args];
    [buffer appendString:@"\n"];
    [buffer fprintf:fh];
    [buffer release];
    va_end(args);

}

@end
