//---------------------------------------------------------------------------------------
// created as RSSCache.m by znek on Sun 29-Oct-2000
// $Id: RSSCache.m,v 1.3 2001/07/27 10:25:36 znek Exp $
//
//  Copyright (c) 2000 by Marcus Mueller <znek@mulle-kybernetik.com>.
//  All rights reserved.
//
//  Permission to use, copy, modify and distribute this software and its documentation
//  is hereby granted under the terms of the GNU General Public License, version 2
//  as published by the Free Software Foundation, provided that both the copyright notice
//  and this permission notice appear in all copies of the software, derivative works or
//  modified versions, and any portions thereof, and that both notices appear in supporting
//  documentation, and that credit is given to Marcus Mueller in all documents and publicity
//  pertaining to direct or indirect use of this code or its derivatives.
//
//  This is free software; you can redistribute and/or modify it under
//  the terms of the GNU General Public License, version 2 as published by the Free
//  Software Foundation. Further information can be found on the project's web pages
//  at http://www.mulle-kybernetik.com/software/MKPortal
//
//  THIS IS EXPERIMENTAL SOFTWARE AND IT IS KNOWN TO HAVE BUGS, SOME OF WHICH MAY HAVE
//  SERIOUS CONSEQUENCES. THE COPYRIGHT HOLDER ALLOWS FREE USE OF THIS SOFTWARE IN ITS
//  "AS IS" CONDITION. THE COPYRIGHT HOLDER DISCLAIMS ANY LIABILITY OF ANY KIND FOR ANY
//  DAMAGES WHATSOEVER RESULTING DIRECTLY OR INDIRECTLY FROM THE USE OF THIS SOFTWARE
//  OR OF ANY DERIVATIVE WORK.
//---------------------------------------------------------------------------------------


#import "RSSCache.h"
#import <EDCommon/EDCommon.h>
#import "RSSParser.h"
#import "RSSDocument.h"

@interface RSSCache (Private)

- (NSDictionary *)getDocumentDescriptionFromURL:(NSString *)urlString;

@end

// Keys for the DocumentDescription
NSString *RSSDocumentObjectKey = @"rootObject";
NSString *RSSDocumentKey = @"document";
NSString *RSSDocumentURLKey = @"documentURL";
NSString *RSSDocumentErrorKey = @"error";
NSString *RSSDocumentTimestampKey = @"timestamp";

@implementation RSSCache

// class var
static NSBundle *_bundle = nil;

////////////////////////////////////////////////////
//
//  INIT & DEALLOC
//
////////////////////////////////////////////////////


- init
{
  [super init];
  rssDocumentLUT = [[NSMutableDictionary allocWithZone:[self zone]] init];
  expireInterval = 30.0 * 60; // 30 minutes
  return self;
}

- (void)dealloc
{
  [rssDocumentLUT release];
  [super dealloc];
}


////////////////////////////////////////////////////
//
//  FACTORY
//
////////////////////////////////////////////////////


+ (id)sharedCache
{
  static id sharedInstance = nil;

  if(sharedInstance == nil)
    sharedInstance = [[RSSCache alloc] init];

  return sharedInstance;
}

+ (void)setBundle:(NSBundle *)aBundle
{
    [aBundle retain];
    [_bundle release];
    _bundle = aBundle;
}


////////////////////////////////////////////////////
//
//  RESETTING
//
////////////////////////////////////////////////////


- (void)flushCaches
{
  [rssDocumentLUT removeAllObjects];
}


////////////////////////////////////////////////////
//
//  EXPIRY SETTINGS
//
////////////////////////////////////////////////////


// it's not possible to set the interval at a value below 60 seconds!
- (void)setExpireInterval:(NSTimeInterval)anInterval
{
	expireInterval = anInterval;
	if(expireInterval < 60.0)
		expireInterval = 60.0;
}

- (NSTimeInterval)expireInterval
{
	return expireInterval;
}


////////////////////////////////////////////////////
//
//  DOCUMENT RETRIEVAL
//
////////////////////////////////////////////////////


- (RSSDocument *)documentObjectForURL:(NSString *)urlString
{
  return [[self documentDescriptionForURL:urlString] objectForKey:RSSDocumentObjectKey];
}

- (NSDictionary *)documentDescriptionForURL:(NSString *)urlString
{
	NSDictionary *documentTable;
	BOOL isDone = NO;
	
	while(isDone == NO)
	{
		NSCalendarDate *expiryDate;

		if((documentTable = [rssDocumentLUT objectForKey:urlString]) == nil || [documentTable objectForKey:RSSDocumentErrorKey] != nil)
		{
			// retrieve the document
			documentTable = [self getDocumentDescriptionFromURL:urlString];
			if(documentTable == nil)
				return nil;
			[rssDocumentLUT setObject:documentTable forKey:urlString];
			isDone = YES;
		}
		else
		{
			expiryDate = [[documentTable objectForKey:RSSDocumentTimestampKey] addTimeInterval:[self expireInterval]];
			if([expiryDate precedesDate:[NSCalendarDate date]])
				[rssDocumentLUT removeObjectForKey:urlString];
			else
				isDone = YES;
		}
	}
	return documentTable;
}


- (NSDictionary *)getDocumentDescriptionFromURL:(NSString *)urlString
{
	NSURL *url;
	NSMutableDictionary *documentTable;
	NSData *content;
	RSSDocument *rootObject;
	
	
	url = [NSURL URLWithString:urlString];
	
	documentTable = [[[NSMutableDictionary allocWithZone:[self zone]] initWithCapacity:4] autorelease];
	[documentTable setObject:url forKey:RSSDocumentURLKey];
	
	if([url isFileURL])
	{
		content = [[[NSData allocWithZone:[self zone]] initWithContentsOfFile:[url path]] autorelease];
		if(content == nil)
		{
			[documentTable setObject:@"File not found" forKey:RSSDocumentErrorKey];
			[documentTable setObject:[NSCalendarDate date] forKey:RSSDocumentTimestampKey];
			return documentTable;
		}
	}
	else
	{
		NSURLHandle *urlHandle;
		
		urlHandle = [url URLHandleUsingCache:NO];
		content = [urlHandle resourceData];
		if(content == nil)
		{
			[documentTable setObject:@"No response from server" forKey:RSSDocumentErrorKey];
			[documentTable setObject:[NSCalendarDate date] forKey:RSSDocumentTimestampKey];
			return documentTable;
		}
	}

	[documentTable setObject:content forKey:RSSDocumentKey];
	// update the timestamp
	[documentTable setObject:[NSCalendarDate date] forKey:RSSDocumentTimestampKey];


	NS_DURING
	
		rootObject = [[self rssParser] documentFromXMLData:content];
		[documentTable setObject:rootObject forKey:RSSDocumentObjectKey];
	
	NS_HANDLER
	
		[documentTable setObject:[NSString stringWithFormat:@"%@ (%@)", [localException name], [localException reason]] forKey:RSSDocumentErrorKey];
	//	NSLog(@"Error while reading document at URL %@: %@ (%@)", url, [localException name], [localException reason]);
	
	NS_ENDHANDLER
	
	return documentTable;
}


////////////////////////////////////////////////////
//
//  RSS PARSING
//
////////////////////////////////////////////////////


- (RSSParser *)rssParser
{
  if(rssParser == nil)
  {
    NSDictionary *rssDefinitions;
    NSString *rssDefinitionsPath;
    NSBundle *bundle;
    
    if(_bundle == nil)
        bundle = [NSBundle mainBundle];
    else
        bundle = _bundle;

    rssDefinitionsPath = [bundle pathForResource:@"RSSTags" ofType:@"plist"];
    if(rssDefinitionsPath == nil)
      [NSException raise:NSGenericException format:@"RSSDefinitions not found!"];

    rssDefinitions = (NSDictionary *)[[NSString stringWithContentsOfFile:rssDefinitionsPath] propertyList];
    rssParser = [[RSSParser parserWithTagDefinitions:rssDefinitions] retain];
    [rssParser setAcceptsUnknownAttributes:YES];
  }
  return rssParser;
}


@end
