/* QTSpriteManager.m created by seb on Thu 02-Apr-1998 */

#import "QTSpriteManager.h"
#import "QDPictHandle.h"

#include <TextUtils.h>

#include <Endian.h>
#include <FixMath.h>
#include <Fonts.h>
#include <Movies.h>
#include <Processes.h>
#include <QuickTimeComponents.h>
#include <Resources.h>
#include <Script.h>

#ifndef _SPRITEUTILITIES_
#include "sprites/SpriteUtilities.h"
#endif

#ifndef __ENDIANUTILITIES__
#include "sprites/EndianUtilities.h"
#endif

#ifndef _IMAGECOMPRESSIONUTILITIES_
#include "sprites/ImageCompressionUtilities.h"
#endif

#define kSpriteTrackWidth			640
#define kSpriteTrackHeight			480

#define kIconPictID				129
#define kWorldPictID				130
#define kBackgroundPictID		        158
#define kFirstSpaceShipPictID			(kBackgroundPictID + 1)

#define kIconImageIndex				1
#define kWorldImageIndex				2
#define kBackgroundImageIndex			3
#define kFirstSpaceShipImageIndex		4
#define kNumSpaceShipImages			24
#define kLastSpaceShipImageIndex			(kFirstSpaceShipImageIndex + kNumSpaceShipImages - 1)

#define kSpriteMediaTimeScale			600
#define kSpriteMediaFrameDuration		8
#define kSpriteMediaFramesPerSecond		(kSpriteMediaTimeScale / kSpriteMediaFrameDuration)

#define kNumOverrideSamples			199

// global variables
Boolean 					gUseBackgroundPicture = true; // do we display a background picture?

@interface QTSpriteManager (PrivateMethods)
  
  // create a sprite track and add it to a QuickTime movie
  -(OSErr)AddSpriteTrackToMovie:(Movie)theMovie;
  
  -(PicHandle)getPictFromFile:(NSString*)filename imageDesc:(ImageDescriptionHandle *)desc;

  // compress a PICT with animation compressor and add image data to a sprite key sample's images container atom
  -(OSErr)AddPICTImageToKeyFrameSample:(QTAtomContainer)keySample pictID:(short)pictID keyColor:(RGBColor *)keyColor withID:(short)theID;
                                        
@end



@implementation QTSpriteManager

-(id)init {
   
  [super init];

  sprite_ = [[NSMutableArray alloc] init];

  [sprite_ insertObject:[QDPictHandle makeWithPicHandle:[self getPictFromFile:@"C:\\images\\icon.pct" imageDesc:0]] atIndex:0];
   [sprite_ insertObject:[QDPictHandle makeWithPicHandle:[self getPictFromFile:@"C:\\images\\world.pct" imageDesc:0]] atIndex:1];
   [sprite_ insertObject:[QDPictHandle makeWithPicHandle:[self getPictFromFile:@"C:\\images\\bkg.pct" imageDesc:0]] atIndex:2];
   [sprite_ insertObject:[QDPictHandle makeWithPicHandle:[self getPictFromFile:@"C:\\images\\ship.pct" imageDesc:0]] atIndex:3];
  //NSLog(@"%@",sprite_);
  return self;
}

-(void)dealloc {
  if (sprite_) {
    int i;
    for (i=0;i<[sprite_ count];i++) [[sprite_ objectAtIndex:i] killPicture];
    [sprite_ release];
  }
}
// private methods

// create a sprite movie
-(Movie)CreateSampleSpriteMovie:(NSString*)theFilename {

  OSErr 			myErr = noErr;
  FSSpec 			fileSpec;
  short 			myResRefNum = 0;
  Movie			        theMovie = 0;
  long				myFlags = createMovieFileDeleteCurFile | createMovieFileDontCreateResFile;
  char                          fullPath[255];

  strcpy(fullPath,[theFilename cString]);
  c2pstr(fullPath);

  FSMakeFSSpec(0,0L,fullPath,&fileSpec);

  // create a movie file for the destination movie
  myErr = CreateMovieFile(&fileSpec, FOUR_CHAR_CODE('TVOD'), 0, myFlags, &myResRefNum, &theMovie);
  if (myErr != noErr)
          goto bail;

  myErr = [self AddSpriteTrackToMovie:theMovie];

  if (myErr != noErr)
    goto bail;

  // add the movie resource to the movie file
  AddMovieResource(theMovie,myResRefNum,0,fileSpec.name);

bail:
    if ( myResRefNum ) CloseMovieFile ( myResRefNum );

  return theMovie;

}

// create a sprite track and add it to a QuickTime movie
-(OSErr)AddSpriteTrackToMovie:(Movie)theMovie {

  OSErr 				err = noErr;
  QTAtomContainer			sample = 0;
  Track 				newTrack;
  Media 				newMedia;
  QTAtomContainer			spriteData;
  RGBColor				keyColor;
  Point					location, iconLocation;
  short					visible, layer, imageIndex,i, iconDelta, iconMinH, iconMaxH;

  // create the track, add it to theMovie, and then add samples to tracks media
  newTrack = NewMovieTrack( theMovie, ((long)kSpriteTrackWidth << 16), ((long)kSpriteTrackHeight << 16), kNoVolume );
  newMedia = NewTrackMedia( newTrack, SpriteMediaType, kSpriteMediaTimeScale,(Handle)0,(OSType)0);

  // create an empty key frame sample
  err = QTNewAtomContainer( &sample );
  if (err!=noErr)
    goto bail;

  keyColor.red = keyColor.green = keyColor.blue = 0xFFFF;		// white

  // add images to the key frame sample
  err = [self AddPICTImageToKeyFrameSample:sample pictID:kIconPictID keyColor:&keyColor withID:1];
  if (err!=noErr)
    goto bail;

  err = [self AddPICTImageToKeyFrameSample:sample pictID:kWorldPictID keyColor:&keyColor withID:2];
  if (err!=noErr)
    goto bail;

  err = [self AddPICTImageToKeyFrameSample:sample pictID:kBackgroundPictID keyColor:&keyColor withID:3];
  if (err!=noErr)
    goto bail;

  for ( i = 1; i <= kNumSpaceShipImages; i++ )
    err = [self AddPICTImageToKeyFrameSample:sample pictID:(kFirstSpaceShipPictID + i - 1) keyColor:&keyColor withID:(i + 3)];


  err = BeginMediaEdits( newMedia );
  if (err!=noErr)
      goto bail;

  // add sprites to the key frame
  err = QTNewAtomContainer( &spriteData );
  if (err!=noErr)
    goto bail;

  if ( gUseBackgroundPicture ) {
    // background
    location.h 	= 0;
    location.v 	= 0;
    visible 	= true;
    layer	= kBackgroundSpriteLayerNum;			// this makes the sprite a background sprite
    imageIndex	= kBackgroundImageIndex;
    
    //err = [self SetSpriteData:spriteData location:&location visible:&visible layer:&layer imageIndex:&imageIndex];
    err = SetSpriteData(spriteData,&location,&visible,&layer,&imageIndex,0,0,0);    
    if (err!=noErr)
      goto bail;

    //[self AddSpriteToSample:sample sprite:spriteData withID:1];
    AddSpriteToSample(sample,spriteData,1);
  }

  // space ship sprite
  location.h 	= 0;
  location.v 	= 60;
  visible 	= true;
  layer		= -1;
  imageIndex	= kFirstSpaceShipImageIndex;
  
  //err = [self SetSpriteData:spriteData location:&location visible:&visible layer:&layer imageIndex:&imageIndex];
  err = SetSpriteData(spriteData,&location,&visible,&layer,&imageIndex,0,0,0);
  if (err!=noErr)
    goto bail;

  //[self AddSpriteToSample:sample sprite:spriteData withID:2];
  AddSpriteToSample(sample,spriteData,2);
  
  // world sprite
  location.h 	= (kSpriteTrackWidth / 2) - 24;
  location.v 	= (kSpriteTrackHeight / 2) - 24;
  visible 	= true;
  layer		= 1;
  imageIndex	= kWorldImageIndex;
  
  //err  = [self SetSpriteData:spriteData location:&location visible:&visible layer:&layer imageIndex:&imageIndex];
  err = SetSpriteData(spriteData,&location,&visible,&layer,&imageIndex,0,0,0);
  if (err!=noErr)
    goto bail;

  //[self AddSpriteToSample:sample sprite:spriteData withID:3];
  AddSpriteToSample(sample,spriteData,3);
  
  // icon sprite
  iconDelta             = 2;
  iconMinH              = (kSpriteTrackWidth / 2) - 116;
  iconMaxH              = iconMinH + 200;
  iconLocation.h 	= iconMinH;
  iconLocation.v 	= (kSpriteTrackHeight / 2) - (24 + 12);
  layer			= 0;
  imageIndex		= kIconImageIndex;
  
  //err = [self SetSpriteData:spriteData location:&iconLocation visible:&visible layer:&layer imageIndex:&imageIndex];
  err = SetSpriteData(spriteData,&iconLocation,&visible,&layer,&imageIndex,0,0,0);
  if (err!=noErr)
    goto bail;

  //[self AddSpriteToSample:sample sprite:spriteData withID:4];
  AddSpriteToSample(sample,spriteData,4);
  
  //[self AddSpriteSampleToMedia:newMedia sample:sample duration:kSpriteMediaFrameDuration isKeyFrame:true];	
  AddSpriteSampleToMedia(newMedia,sample,kSpriteMediaFrameDuration,true,0);
  
  // now add a bunch of override frames to the media which make the space ship spin and move
  imageIndex = kFirstSpaceShipImageIndex;
  location.h = 0;
  location.v = 80;

  visible = true;

  for ( i = 1; i < kNumOverrideSamples; i++ ) {
    QTRemoveChildren( sample, kParentAtomIsContainer );
    QTRemoveChildren( spriteData, kParentAtomIsContainer );

    // bump the image index every third frame to spin
    if ( (i % 3) == 0 ) {
      imageIndex++;
      if ( imageIndex > kLastSpaceShipImageIndex )
        imageIndex = kFirstSpaceShipImageIndex;
    }

    // bump the location by one pixel vertically and two horizontally every frame to make it move
    location.h += 2;
    location.v += 1;

    if (visible)
      //[self SetSpriteData:spriteData location:&location visible:0 layer:0 imageIndex:&imageIndex];
      SetSpriteData(spriteData,&location,0,0,&imageIndex,0,0,0);
    else {
      visible = true;
      //[self SetSpriteData:spriteData location:&location visible:&visible layer:0 imageIndex:&imageIndex];
      SetSpriteData(spriteData,&location,&visible,0,&imageIndex,0,0,0);
    }

    //[self AddSpriteToSample:sample sprite:spriteData withID:2];
    AddSpriteToSample(sample,spriteData,2);
    
    // make the icon move and change layer
    QTRemoveChildren( spriteData, kParentAtomIsContainer );
    iconLocation.h += iconDelta;
    
    if ( iconLocation.h >= iconMaxH ) {
      iconLocation.h = iconMaxH;
      iconDelta = - iconDelta;
    }
    
    if ( iconLocation.h <= iconMinH ) {
      iconLocation.h = iconMinH;
      iconDelta = - iconDelta;
    }
    
    if ( iconDelta > 0 )
      layer = 0;
    else
      layer = 3;
    
    //[self SetSpriteData:spriteData  location:&iconLocation visible:0 layer:&layer imageIndex:0];
    SetSpriteData(spriteData,&iconLocation,0,&layer,0,0,0,0);
    
    //[self AddSpriteToSample:sample sprite:spriteData withID:4];
    AddSpriteToSample(sample,spriteData,4);
    
    //[self AddSpriteSampleToMedia:newMedia sample:sample duration:kSpriteMediaFrameDuration isKeyFrame:false];
    AddSpriteSampleToMedia(newMedia,sample,kSpriteMediaFrameDuration,false,0);	
  }

  EndMediaEdits( newMedia );

  // add the media into track
  InsertMediaIntoTrack( newTrack, 0, 0, GetMediaDuration( newMedia ), fixed1 );

  // set the sprite track properties
  
  // add a background color to the sprite track
  if ( !gUseBackgroundPicture ) {
    
    QTAtomContainer		trackProperties;
    RGBColor			backgroundColor;

    backgroundColor.red   = EndianU16_NtoB(0x8000);
    backgroundColor.green = EndianU16_NtoB(0);
    backgroundColor.blue  = EndianU16_NtoB(0xffff);

    QTNewAtomContainer( &trackProperties );
    QTInsertChild( trackProperties, 0, kSpriteTrackPropertyBackgroundColor, 1, 1, sizeof(RGBColor), &backgroundColor, 0 );

    err = SetMediaPropertyAtom( newMedia, trackProperties );
    if (err != noErr)
      goto bail;
    
    QTDisposeAtomContainer( trackProperties );
  }

bail:
    if ( sample )		QTDisposeAtomContainer( sample );
    if ( spriteData )		QTDisposeAtomContainer( spriteData );

  return err;
}


-(PicHandle)getPictFromFile:(NSString*) filename imageDesc:(ImageDescriptionHandle *)desc {

  char fullPath[255];
  FSSpec fileSpec;
  PicHandle picture =0;
  GraphicsImportComponent gi;
  OSErr err = noErr;  
  
  [filename getCString:fullPath];
  c2pstr(fullPath);

  FSMakeFSSpec(0,0L,fullPath,&fileSpec);
    
  err = GetGraphicsImporterForFile(&fileSpec, &gi);

  if (err!=noErr)
    printf("GetGraphicsImporterForFile -> err = %d\n",err);

  err = GraphicsImportGetAsPicture(gi,&picture);
  if (err!=noErr)
    printf("GraphicsImportGetAsPicture -> err = %d\n",err);
   
  if (desc) GraphicsImportGetImageDescription(gi,desc);
  
  CloseComponent(gi);
  
  return picture;
}

// compress a PICT with animation compressor and add image data to a sprite key sample's images container atom
-(OSErr)AddPICTImageToKeyFrameSample:(QTAtomContainer)keySample pictID:(short)pictID keyColor:(RGBColor *)keyColor withID:(short)theID {

  OSErr			  err = noErr;
  PicHandle		  picture;
  Handle		  compressedPicture;
  ImageDescriptionHandle	  idh;

  switch(pictID) {
    case kIconPictID:
      picture = [[sprite_ objectAtIndex:0] pictHandle];
      break;
    case kWorldPictID:			
      picture = [[sprite_ objectAtIndex:1] pictHandle];
      break;
    case kBackgroundPictID:
      picture = [[sprite_ objectAtIndex:2] pictHandle];
      break;
    default:
      // de kFirstSpaceShipPictID  kFirstSpaceShipPictID + kNumSpaceShipImages
      picture = [[sprite_ objectAtIndex:3] pictHandle];
  }

  // convert it to image data compressed by the animation compressor
  err = RecompressPictureWithTransparency( picture, keyColor, 0, &idh, &compressedPicture );
  if(err) goto bail;

  // add it to the keySample
  HLock( compressedPicture );
  err = AddCompressedImageToKeyFrameSample( keySample, idh, GetHandleSize( compressedPicture ), *compressedPicture, theID, 0, 0 );
    
bail:
  if ( compressedPicture )	DisposeHandle( compressedPicture );	
  if ( idh )			DisposeHandle( (Handle)idh );

  return err;
}

@end
