//
//  This file is a part of StevesClock.app, a simple dock clock application.
//  Copyright (C) 1998  Steve Nygard
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
//  You may contact the author by:
//     e-mail:  nygard@telusplanet.net
//

#import "SNClockView.h"
#import "SNUtility.h"
#import "NSImage+SNExtensions.h"

static NSImage *_clockBitsImage = nil;

#if 0
#define BOTTOM_MARGIN (6.0)
#else
#define BOTTOM_MARGIN (0.0)
#endif

// The optimal size of this view appears to be about 64x70.

static NSRect _calendarRect = { {64, 72}, {64, 64} };

static NSRect _dowRects[7] =
{
    { {0, 29}, {19, 5} },
    { {0, 65}, {19, 5} },
    { {0, 59}, {19, 5} },
    { {0, 53}, {19, 5} },
    { {0, 47}, {19, 5} },
    { {0, 41}, {19, 5} },
    { {0, 35}, {19, 5} },
};

static NSRect _monthRects[12] =
{
    { {40, 66}, {22, 5} },
    { {40, 60}, {22, 5} },
    { {40, 54}, {22, 5} },
    { {40, 48}, {22, 5} },
    { {40, 42}, {22, 5} },
    { {40, 36}, {22, 5} },
    { {40, 30}, {22, 5} },
    { {40, 24}, {22, 5} },
    { {40, 18}, {22, 5} },
    { {40, 12}, {22, 5} },
    { {40,  6}, {22, 5} },
    { {40,  0}, {22, 5} },
};

static NSRect _dayOfMonthNumberRects[10] =
{
    { {172, 14}, {10, 17} },
    { { 65, 14}, { 8, 17} },
    { { 76, 14}, {10, 17} },
    { { 88, 14}, { 9, 17} },
    { {100, 14}, {10, 17} },
    { {112, 14}, {10, 17} },
    { {124, 14}, {10, 17} },
    { {136, 14}, {10, 17} },
    { {148, 14}, {10, 17} },
    { {160, 14}, {10, 17} },
};

static NSRect _timeNumberRects[10] =
{
    { {166, 0}, {9, 11} },
    { { 89, 0}, {4, 11} },
    { { 93, 0}, {9, 11} },
    { {104, 0}, {8, 11} },
    { {112, 0}, {9, 11} }, // four
    { {121, 0}, {8, 11} },
    { {130, 0}, {8, 11} },
    { {141, 0}, {7, 11} }, // seven
    { {148, 0}, {9, 11} },
    { {157, 0}, {9, 11} },
};

static NSRect _yearNumberRects[10] =
{
    { {12,  8}, {5, 5} },
    { { 0, 20}, {1, 5} },
    { { 2, 20}, {4, 5} },
    { { 7, 20}, {4, 5} },
    { {12, 20}, {5, 5} },
    { { 0, 14}, {4, 5} },
    { { 5, 14}, {5, 5} },
    { {11, 14}, {4, 5} },
    { { 0,  8}, {5, 5} },
    { { 6,  8}, {5, 5} },
};

static NSRect _colonRect = { {175, 0}, { 4, 11} };
static NSRect _amRect    = { {179, 0}, {12, 6} };
static NSRect _pmRect    = { {192, 0}, {11, 6} };

static NSPoint _dowDestinationPoint =   {21, 34 + BOTTOM_MARGIN};
static NSPoint _monthDestinationPoint = {19,  9 + BOTTOM_MARGIN};
static NSPoint _ampmDestinationPoint =  {44, 48 + BOTTOM_MARGIN};

//======================================================================
//
//
//
//======================================================================

#define SNClockView_VERSION 1

@implementation SNClockView

+ (void) initialize
{
    if (self == [SNClockView class])
    {
        [self setVersion:SNClockView_VERSION];

        // It appears that there is no guarantee that we can get the bundle
        // in this method, so we have to work around this.

        if ([NSBundle bundleForClass:self] == nil)
        {
            [[NSNotificationCenter defaultCenter] addObserver:self
                                                  selector:@selector (loadClassImages)
                                                  name:NSApplicationDidFinishLaunchingNotification
                                                  object:NSApp];
        }
        else
        {
            [self loadClassImages];
        }
    }
}

//----------------------------------------------------------------------

+ (void) loadClassImages
{
    NSBundle *thisBundle;
    
    if (self == [SNClockView class])
    {
        thisBundle = [NSBundle bundleForClass:self];
        NSAssert (thisBundle != nil, @"Could not get bundle.");

        _clockBitsImage = [[NSImage imageNamed:@"clockbits.tiff" fromBundle:thisBundle] retain];
    }

    [[NSNotificationCenter defaultCenter] removeObserver:self];
}

//----------------------------------------------------------------------

- initWithFrame:(NSRect)frameRect
{
    if ([super initWithFrame:frameRect] == nil)
        return nil;

    date = [[NSCalendarDate calendarDate] retain];

    militaryTime = NO;
    timer = nil;

    return self;
}

//----------------------------------------------------------------------

- (void) dealloc
{
    SNRelease (date);
    if (timer != nil)
    {
        [timer invalidate];
        SNRelease (timer);
    }

    [super dealloc];
}

//----------------------------------------------------------------------

- (NSCalendarDate *) date
{
    return date;
}

//----------------------------------------------------------------------

- (void) setDate:(NSCalendarDate *)newDate
{
    if (timer != nil)
    {
        [timer invalidate];
        SNRelease (timer);
    }        

    SNRelease (date);
    date = [newDate retain];
    [self setNeedsDisplay:YES];
}

//----------------------------------------------------------------------

- (BOOL) militaryTime
{
    return militaryTime;
}

//----------------------------------------------------------------------

- (void) setMilitaryTime:(BOOL)flag
{
    militaryTime = flag;

    [self setNeedsDisplay:YES];
}

//----------------------------------------------------------------------

- (void) showCurrentTime
{
    if (timer != nil)
    {
        [timer invalidate];
        SNRelease (timer);
    }

    timer = [[NSTimer scheduledTimerWithTimeInterval:15
                      target:self
                      selector:@selector (updateCurrentTime:)
                      userInfo:nil
                      repeats:YES] retain];
}

//----------------------------------------------------------------------

- (void) updateCurrentTime:(NSTimer *)userInfo
{
    SNRelease (date);
    date = [[NSCalendarDate calendarDate] retain];
    [self setNeedsDisplay:YES];
}

//----------------------------------------------------------------------

- (void) drawRect:(NSRect)rect
{
    int d1, d2, d3, d4;
    NSPoint aPoint;
    float w;

    int hour, minute;
    int year, month, day, dayOfWeek;

    aPoint.x = 0.0;
    aPoint.y = BOTTOM_MARGIN;

    [_clockBitsImage compositeToPoint:aPoint fromRect:_calendarRect operation:NSCompositeSourceOver]; // Was just copy

#if 0
    PSsetgray (NSLightGray);
    PSrectfill (0, 0, rect.size.width, 7.0);
#endif
    if (date == nil)
        return;

    year = [date yearOfCommonEra];
    month = [date monthOfYear];
    day = [date dayOfMonth];
    dayOfWeek = [date dayOfWeek];

    hour = [date hourOfDay];
    minute = [date minuteOfHour];

    // Time.

    if (militaryTime == NO && hour > 12)
    {
        d1 = (hour - 12) / 10;
        d2 = (hour - 12) % 10;
    }
    else if (militaryTime == NO && hour == 0)
    {
        // 0 am.
        d1 = 1;
        d2 = 2;
    }
    else
    {
        d1 = hour / 10;
        d2 = hour % 10;
    }

    d3 = minute / 10;
    d4 = minute % 10;

    w = 0.0;

    if (militaryTime == NO && d1 == 0)
        w = 0.0;
    else
        w = _timeNumberRects[d1].size.width;

    w += _timeNumberRects[d2].size.width + _colonRect.size.width + _timeNumberRects[d3].size.width + _timeNumberRects[d4].size.width;

    if (militaryTime == YES)
        aPoint.x = 6.0 + ((57.0 - 5.0) - w) / 2.0;
    else
        aPoint.x = 6.0 + ((43.0 - 5.0) - w) / 2.0;

    aPoint.y = 46.0 + BOTTOM_MARGIN;

    if (militaryTime == YES || d1 != 0)
    {
        [_clockBitsImage compositeToPoint:aPoint fromRect:_timeNumberRects[d1] operation:NSCompositeSourceOver];
        aPoint.x += _timeNumberRects[d1].size.width;
    }

    [_clockBitsImage compositeToPoint:aPoint fromRect:_timeNumberRects[d2] operation:NSCompositeSourceOver];
    aPoint.x += _timeNumberRects[d2].size.width;

    [_clockBitsImage compositeToPoint:aPoint fromRect:_colonRect operation:NSCompositeSourceOver];
    aPoint.x += _colonRect.size.width;

    [_clockBitsImage compositeToPoint:aPoint fromRect:_timeNumberRects[d3] operation:NSCompositeSourceOver];
    aPoint.x += _timeNumberRects[d3].size.width;

    [_clockBitsImage compositeToPoint:aPoint fromRect:_timeNumberRects[d4] operation:NSCompositeSourceOver];
    aPoint.x += _timeNumberRects[d4].size.width;

    if (militaryTime == NO)
    {
        if (hour < 12)
            [_clockBitsImage compositeToPoint:_ampmDestinationPoint fromRect:_amRect operation:NSCompositeSourceOver];
        else
            [_clockBitsImage compositeToPoint:_ampmDestinationPoint fromRect:_pmRect operation:NSCompositeSourceOver];
    }

    //

    [_clockBitsImage compositeToPoint:_dowDestinationPoint fromRect:_dowRects[dayOfWeek] operation:NSCompositeSourceOver];
    [_clockBitsImage compositeToPoint:_monthDestinationPoint fromRect:_monthRects[month-1] operation:NSCompositeSourceOver];

    // Month day.

    d1 = day / 10;
    d2 = day % 10;
    w = _dayOfMonthNumberRects[d2].size.width;
    if (d1 != 0)
        w += 1.0 + _dayOfMonthNumberRects[d1].size.width;

    aPoint.x = 14.0 + ((46.0 - 14.0) - w) / 2.0;
    aPoint.y = 15.0 + BOTTOM_MARGIN;
    if (d1 != 0)
    {
        [_clockBitsImage compositeToPoint:aPoint fromRect:_dayOfMonthNumberRects[d1] operation:NSCompositeSourceOver];
        aPoint.x += 1.0 + _dayOfMonthNumberRects[d1].size.width;
    }
    [_clockBitsImage compositeToPoint:aPoint fromRect:_dayOfMonthNumberRects[d2] operation:NSCompositeSourceOver];

#if 0
    // Year.
  
    d1 = (year / 1000) % 10;
    d2 = (year / 100) % 10;
    d3 = (year / 10) % 10;
    d4 = year % 10;

    w = _yearNumberRects[d1].size.width + _yearNumberRects[d2].size.width + _yearNumberRects[d3].size.width + _yearNumberRects[d4].size.width + 3.0;

    aPoint.x = (64.0 - w) / 2.0;
    aPoint.y = 1.0;
    [_clockBitsImage compositeToPoint:aPoint fromRect:_yearNumberRects[d1] operation:NSCompositeSourceOver];

    aPoint.x += _yearNumberRects[d1].size.width + 1.0;
    [_clockBitsImage compositeToPoint:aPoint fromRect:_yearNumberRects[d2] operation:NSCompositeSourceOver];

    aPoint.x += _yearNumberRects[d2].size.width + 1.0;
    [_clockBitsImage compositeToPoint:aPoint fromRect:_yearNumberRects[d3] operation:NSCompositeSourceOver];

    aPoint.x += _yearNumberRects[d3].size.width + 1.0;
    [_clockBitsImage compositeToPoint:aPoint fromRect:_yearNumberRects[d4] operation:NSCompositeSourceOver];
#endif
}

@end
