/* CardPileController.m created by veilljf on Wed 25-Nov-1998 */

#import "CardPileController.h"
#import "CardPile.h"
#import "CardPileView.h"
#import <AppKit/NSView.h>
#import <AppKit/NSWindow.h>
#import <AppKit/NSImage.h>
#import <AppKit/NSCustomImageRep.h>
#import <AppKit/NSPasteboard.h>
#import "CardGameProtocol.h"
#import <Foundation/Foundation.h>
#import "CardPileHolder.h"

@implementation CardPileController

/*" CardPileController is the Controller part of the famous triad (MVC) where the
View is the CardPileView and the model is CardPile.

This controller implement user interaction as drag and drop, moving card using
the NSDrag mechanism.



#{Notifying the CardPileController Object's Delegate}

The logic of a card game is primarily implemented through the delegate
mechanism.  Here delegation make more sense than subclassing since the game logic really reside somewhere inside  the model.  Having this logic object as a delegate enhance encapsulation and modularity.
The delegate has the option of implementing the following methods:

clickedCard:in:
doubleClickedCard:in:
canAcceptPile:from:in:

Knowing the sequence in which these delegate messages are sent is critical
in using the CardSet objects to create card games.  If there is no delegate
(or the delegate does not implement the above methods) the cards on a pile
may not be dragged, flipped, or have cards dropped on them.

When the user single clicks on a CardPileView, the message
#{clickedCard:}%{aCard} #{in:}%{aCardPileView} is sent to the delegate.
Note that the parameter %{aCard} will be nil if the pile was empty, or if
the user clicked in an area of the pile where there is no Card displayed.
This message is sent even if the user keeps holding down the mouse button
(ie. dragging) after clicking.

If the user holds down the mouse button, and is pointing at a Card, the
delegate receives the message #{draggedHolder:}%{aCardPile}
#{from:}%{aCardPileView}.  %{aCardPile} is a temporary pile containing the
Card that the user clicked on.

If #draggedHolder:from: returns YES, CardPileView performs the dragging
animation with the cards on the temporary CardPile.  If the user releases
the dragged pile on any CardPileView (other than the source pile), that 	
CardPileView's delegate receives the #{canAcceptPile:}%{dropPile}
#{from:}%{sender} #{in:}%{cardPileView} message, where %dropPile is the
pile that wants to be dropped, %cardPileView is the pile receiving the
cards, and %sender is the CardPileView that the cards were dragged from.
If this method returns NO, the dragged cards return to the source pile.

If a CardPileView agrees to accept some dragged cards, they are
automatically added to that pile.  Then the delegate receives the
#{acceptPile:}%{dropPile} #{in:}%{self} message, which lets the pile know
which cards were added to it.

The cards are now removed from the original CardPileView, and its delegate
receives the #{removedPile:}%{aCardPile} #{from:}%{aCardPileView} message.

"*/


- init
{
    viewShouldDrag = NO;
    valid = YES;
    return [super init];
}

- (void)dealloc
{
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    [draggedHolder release];
//    [view release];
    [model release];
    [lastMouseDownEvent release];
    [super dealloc];
}

- view
    /*" return the 'M' part of the famous 'MVC' triad."*/
{
    return view;
}

- (void)setView:(id)v
    /*" Set the 'V' part of the famous 'MVC' triad."*/
{
    // do not retain the view to avoid retain cycles
    // the view will retain the controller.
    view = v;
//    if (v != view ) {
//        [view release];
//        view = [v retain];
//    }
}

- model
    /*" return the 'M' part of the famous 'MVC' triad."*/
{
    return model;
}

- (void)setModel:m
    /*" Set the 'M' part of the famous 'MVC' triad."*/
{
    if(m != model ) {
        [[NSNotificationCenter defaultCenter] removeObserver:self name:nil object:model];

        [model release];
        model = [m retain];
        [[NSNotificationCenter defaultCenter] addObserver:self
                                                 selector:@selector(invalidateModel:)
                                                     name:CARDHOLDER_INVALIDATE
                                                   object:model];
        [[NSNotificationCenter defaultCenter] addObserver:self
                                                 selector:@selector(validateModel:)
                                                     name:CARDHOLDER_VALIDATE
                                                   object:model];
    }
}

- draggedHolder
    /*" a drag holder is simply a list holder that will hold temporary data.  This become usefull in the Undo context where the data store temporarity in a bigger sequence."*/
{
    if(draggedHolder == nil) {
        draggedHolder = [[CardPileHolder alloc] init];
        [draggedHolder setGameModel:[self model]];
    }
    return draggedHolder;
}

- (void)setDraggedHolderValue:(id)value
    /*" set the value data stored in the draggedHolder."*/
{
    [value moveAtLastOf:[self draggedHolder]];
}

- (void)setDraggedHolder:(id)p
/*" setter method for the draggedHolder instance variable."*/
{
    id old = draggedHolder;
    draggedHolder = [p retain];
    [old release];
}

- lastMouseDownEvent
    /*" This method return the last mouseDown event that the view received.  Since we do not implement our own event loop to check mouse activity, we need to know the mouseDown event when we receive the mouseDraggedEvent in order to use the NSView/NSWindow drag services."*/
{
    return lastMouseDownEvent;
}

- (void)setLastMouseDownEvent:(NSEvent*)ev
    /*" This method set the last mouseDown event that the view received.  Since we do not implement our own event loop to check mouse activity, we need to know the mouseDown event when we receive the mouseDraggedEvent in order to use the NSView/NSWindow drag services."*/
{
    id old = lastMouseDownEvent;
    lastMouseDownEvent = [ev retain];
    [old release];
}

- (void)validateModel:(NSNotification*)no
    /*"
    notify when our model get re-validated.  We have to be able to revert an invalidation for Undo capability.
     "*/
{
    [self setValid:YES];
}

- (void)invalidateModel:(NSNotification*)no
    /*" notify when our model get invalidated.  We have to be able to revert this invalidation for Undo capability.  So when a card Pile get invalidated, we simply move it frame it to {0,0,0,0} in window's coordinate. "*/
{
    [self setValid:NO];
}

- (BOOL)valid
    /*" a controller can get invalid if the model send the notification that it is invalidated.  An invalid controller mean that it won't respond to any mouse event.  This is done by moving the view see invalidateModel: documentation."*/
{
    return valid;
}

- (void)setValid:(BOOL)v
/*" a controller can get invalid if the model send the notification that it is invalidated.  An invalid controller mean that it won't respond to any mouse event.  This is done by moving the view see invalidateModel: documentation."*/
{
    if(v != valid) {
        if(v == YES) {
            [view setFrame:oldFrame];
            [view display];
        } else {
            NSRect fr = {  {0.0, 0.0} , {0.0, 0.0} };
            oldFrame = [view frame];
            [view display];
            [view setFrame:fr];
        }
        valid = v;
    }
}

/*" Mouse Management "*/

- (void) mouseDragged:(NSEvent *)theEvent
    /*"received when a drag operation is started.  Since NSDragging mecanism is actualy expecting a specific behavior for this method, then we have to flag our view to use it's superclass behavior."*/
{
    if(valid) {
        if(viewShouldDrag) {
            [view superMouseDragged:theEvent];
        } else {

            NSPoint    	thePoint = [theEvent locationInWindow];
            CardPile*	thePile;

            viewShouldDrag = YES;

            /*-----------------------------------------------------------------------
                |    Determine which card was clicked on
                \-----------------------------------------------------------------------*/

            thePoint = [view convertPoint:thePoint fromView:nil];
            thePile = [view findPileAtPoint:thePoint];

            if (thePile) {
                if ([model draggedPile:thePile from:[self model]])
                {
                    NSPasteboard*   thePasteboard;
                    NSRect     	cardRect;
                    NSImage*   	cardImage;
                    NSPoint	winCardPoint;

                    // BEGIN ONLY an undo grouping.
                    [[model undoManager] beginUndoGrouping];

                    // If it can be dragged, calculate the size of the image to be dragged
                    cardRect = [view getRectForPile:thePile];
                    winCardPoint = [view convertPoint:cardRect.origin toView:nil];
                    // remove the cardPile from his holder, keep the pile as the draggedHolder.
                    [self setDraggedHolderValue:thePile];

                    //  Create the image and pasteboard
                    cardImage = [[NSImage allocWithZone:[view zone]]
                            initWithSize:cardRect.size];
                    [cardImage addRepresentation:[[[NSCustomImageRep alloc]
                            initWithDrawSelector:@selector(drawDragCard:)
                                        delegate:view] autorelease]];
                    
                    // We have to retain the pasteboard until after the dragging.
                    thePasteboard = [[NSPasteboard
                            pasteboardWithName:@"NXDragPBoard"] retain];
                    [thePasteboard declareTypes:[NSArray arrayWithObject:CardPilePBoardType] owner:self];
                    [thePasteboard setData:[model firstObject] forType:CardPilePBoardType];

                    //   Drag it
                    [[view window] dragImage:cardImage
                                 at:winCardPoint
                             offset:NSMakeSize(0,0)
                              event:lastMouseDownEvent
                         pasteboard:thePasteboard
                             source:self
                          slideBack:YES];

                    //  Destroy the image and pasteboard.
                    [thePasteboard autorelease];
                    [cardImage release];
                }
            }
        }
    } else {  // if the model is invalidated, then pass the event down the responder chain
        [view superMouseDragged:theEvent];
    }
}

- (void)mouseUp:(NSEvent *)theEvent
    /*" Mouse Up received,  Send a perform Click of can be."*/
{
    if (valid) {
        NSPoint    	thePoint = [theEvent locationInWindow];
        CardPile*	thePile;
            //   Determine which card was clicked on
        thePoint = [view convertPoint:thePoint fromView:nil];
        thePile = [view findPileAtPoint:thePoint];
            //   enclose the behavior inside an undo group.
        [[model undoManager] beginUndoGrouping];
        [model clickedCard:thePile in:model];
        [[model undoManager] endUndoGrouping];
    } else {  // if the model is invalidated, then pass the event down the responder chain
        [view superMouseUp:theEvent];
    }
}

- (void) mouseDown:(NSEvent *)thisEvent
    /*"  "*/
{
    if(valid) {
        if([thisEvent clickCount] == 2) {
            NSPoint    	thePoint = [thisEvent locationInWindow];
            CardPile*	thePile;
                //   Determine which card was clicked on
            thePoint = [view convertPoint:thePoint fromView:nil];
            thePile = [view findPileAtPoint:thePoint];

                // enclose the behavior inside an undo group.
            [[model undoManager] beginUndoGrouping];
            [model doubleClickedCard:thePile in:model];
            [[model undoManager] endUndoGrouping];
        }
        viewShouldDrag = NO;
        [self setLastMouseDownEvent:thisEvent];
    } else {  // if the model is invalidated, then pass the event down the responder chain
        [view superMouseDown:thisEvent];
    }
}

- (unsigned int) draggingSourceOperationMaskForLocal:(BOOL)flag
    /*"
    Let the dragging mechanism know that only generic dragging is available,
     | and then only within the same application.
     "*/
{
    if (flag)
    {
        return NSDragOperationGeneric;
    }
    return NSDragOperationNone;
}

- (void) draggedImage:(NSImage *)image beganAt:(NSPoint)screenPoint
    /* "
    After the dragged image has been displayed, redraw ourselves without the
    cards being dragged.
    "*/
{
    [view display];
}

- (void) draggedImage:(NSImage *)image endedAt:(NSPoint)screenPoint
            deposited:(BOOL)flag
    /*"
    After the cards have been dragged successfully, notify our delegate,
     if appropriate.  Redraw ourselves whether cards were removed or not.
     "*/
{
    id dropPile = [[self draggedHolder] firstObject];
    if (flag == NO)
    {
        // dragged failed, card must return to self
        [dropPile moveAtLastOf:model];
    } else {
        // advise the delegate that we just removed card from it's pile
        [model removedPile:dropPile from:model];
        // name this action in the undo manager
        [[model undoManager] setActionName:@"drag & drop"];
    }
    // END ONLY an undo grouping.  The Begin drag operation will begin the group.
    [[model undoManager] endUndoGrouping];
    viewShouldDrag = NO;
}

- (unsigned int) draggingEntered:sender
    /*"
    Ask our delegate if the cards dragged in can be dropped.
     "*/
{
    unsigned int 	theOperation = NSDragOperationNone;
    id dragCardPile = [[[sender draggingSource] draggedHolder] firstObject];

    if  (dragCardPile) {
        if([model canAcceptPile:dragCardPile from:sender in:model])
        {
            theOperation = NSDragOperationGeneric;
        }
    }
    return theOperation;
}

- (BOOL) prepareForDragOperation:sender
    /*"
    Add cards dropped on our pile to the top of the pile and notify our
     delegate, if possible.  Redisplay the pile afterwards in any case.
     "*/
{
    id dropPile = [[[sender draggingSource] draggedHolder] firstObject];

    [dropPile moveAtLastOf:model];
    [model acceptPile:dropPile in:model];

    return YES;
}

- (BOOL)performDragOperation:sender
    /*"
    Returns YES.
     "*/
{
    return YES;
}

@end
