/* indent:4  tabsize:8  font:fixed-width */

#import "CardPile.h"
#import <Solitaire/CardSet.h>
#import <Foundation/Foundation.h>

#if defined (WIN32)
#define random() rand()
#define srandom(x) srand(x)
#else
#import <libc.h>
#endif


NSString *CARDPILE_CHANGED = @"CardPile_Changed";

@implementation CardPile

/*"
	Instances of this class provide low-level functionality for card games.
 A CardPile is a node of a DblLinkedList where the value of the node is the card and we have added a faceUp instance variable to keep track of the orientation of the card.
 
	A CardPile is a Card in the context of a group of card. So a CardPile has a faceUp attribute that define if the pile is faceUp or down within the pile. 
"*/

+ (void) initialize
/*"
        Sets a random seed based on the time before initializing the class.
"*/
{
        if (self == [CardPile class]) {
            srandom(time(0));
        }
}

+ (CardPile*) fullDeck
    /*" Adds a full deck of cards to the top of the pile, neatly ordered. all face down.
        Return an autorelease CardPile object (which is a linked list of CardPile*)"*/
{
    static id fullDeck = nil;
    if(fullDeck == nil) {
        int suitCounter, valueCounter;
        CardPile *prev = [[CardPileHolder alloc] init];
        CardPile *hold = prev;

        for (valueCounter = CS_LOWVALUE; valueCounter <= CS_HIGHVALUE; valueCounter++)
        {
            for (suitCounter = CS_LOWSUIT; suitCounter <= CS_HIGHSUIT; suitCounter++)
            {
                CardPile *newCard;
                newCard = [[[self class] alloc] initWithCard:[[[Card allocWithZone:[self zone]] initSuit:suitCounter value:valueCounter] autorelease]
                                                      faceUp:NO];
                [prev appendObjects:newCard];
                [newCard release];
                prev = newCard;
            }
        }
        fullDeck = [[hold nextObject] retain];
        [hold release];
    }
    return fullDeck;
}

- init
/*"
        Calls our designated initializer #initWithCard:nil faceUp:NO.
"*/
{
    return [self initWithCard:nil faceUp:NO];
}

- initWithCard:(Card*)crd faceUp:(BOOL)fu
/*"
        Designated initializer with the CardObject (stored as value) and the faceUp boolean.
"*/
{
    [super init];
    [self setValue:crd];
    isFaceUp = fu;
    return self;
}

- (BOOL)isFaceUp
    /*" return isFaceUp instance variable."*/
{
    return isFaceUp;
}

- (void)setFaceUp:(BOOL)bl
    /*" set the isFaceUp instance variable."*/
{
    if(bl != isFaceUp) {
        isFaceUp = bl;
        [self changed];
        [[self undoManager] registerUndoWithTarget:self
                                     selector:@selector(setFaceUp:)
                                       object:(bl != (BOOL)nil) ? nil : self ];
    }
}

- undoManager
{
    return [holder undoManager];
}

- (void)moveAtLastOf:(ListHolder*)list
    /*" Move the list at the end of the other list.  Adjust holder, and so on."*/
{
    id ho = [self holder];
    [super moveAtLastOf:list];
    [[self undoManager] registerUndoWithTarget:self
                                 selector:@selector(moveAtLastOf:)
                                   object:ho];
}


- (Card*)bottomCard
    /*" return the first card, next to the holder."*/
{
    return [[self firstHolderObject] value];
}

- (void)dealLastObjectFlipedFaceTo:(ListHolder*)dest
    /*" move the last CardPile object from the list to the destination CardPile and flip the faceUp. "*/
{
    [self dealLastObjectTo:dest faceUp:![[self lastObject] isFaceUp]];
}

- (void)dealLastObjectTo:(ListHolder*)dest faceUp:(BOOL)up
    /*" move the last CardPile object from the list to the destination CardPile and set the faceUp to 'up'. "*/
{
    [[self lastObject] setFaceUp:up];
    [self dealLastObjectTo:dest];
}

- (void)dealLastObjectFaceUpTo:(ListHolder*)dest
    /*" move the last CardPile object from the list to the destination CardPile and set the faceUp to YES. "*/
{
    [self dealLastObjectTo:dest faceUp:YES];
}

- (void)dealLastObjectFaceDownTo:(ListHolder*)dest
    /*" move the last CardPile object from the list to the destination CardPile and set the faceUp to NO. "*/
{
    [self dealLastObjectTo:dest faceUp:NO];
}

- (int)numberOfContiniusHolderVisibleCard
    /*" return the number of top visible cards. From now on, using the nextObject mecanism, return the number of continuous card where isFaceUp == YES   "*/
{
    if([self isFaceUp]) {
        return 1 + [holder numberOfContiniusHolderVisibleCard];
    }
    return 0;
}

- (BOOL)isAllNextInCardSuit:(CardSuit)s
    /*"return a BOOL that testifies that all next CardPile are is the suit _s_."*/
{
    BOOL result = ([self cardSuit] == s);
    if((result == NO) || ([self nextObject] == nil)) {
        return result;
    }
    return [nextObject isAllNextInCardSuit:s];
}

- (BOOL)isAllNextInSameCardSuit
    /*"return a BOOL that testifies that all next CardPile are is the same suit as self."*/
{
    return [self isAllNextInCardSuit:[self cardSuit]];
}

- (BOOL)isAllNextInDecreasingCardValue
    /*"return a BOOL that testifies that all next CardPile are decreasing their Card value by 1."*/
{
    return [self isAllNextInDecreasingCardValue:[self cardValue]];
}

- (BOOL)isAllNextInDecreasingCardValue:(int)v
    /*"return a BOOL that testifies that all next CardPile are the value v and next object is of value v -1."*/
{
    BOOL result = ([self cardValue] == v);
    if((result == NO) || ([self nextObject] == nil)) {
        return result;
    }
    return [nextObject isAllNextInDecreasingCardValue:v-1];
}

- (BOOL)isAllNextInAlternateColor
    /*"return a BOOL that testifies that all next CardPile are alternating color."*/
{
    return [self isAllNextInAlternateColor:[self cardColor]];
}

- (BOOL)isAllNextInAlternateColor:(CardColor)col
/*" is self in the expecting card suit color, if not return NO, if yes is the next object in my inverse color"*/
{
    CardColor myColor = [[self value] cardColor];

    if(myColor != col) {
        return NO;
    }
    if (nextObject == nil) {
        return YES;
    }
    return [nextObject isAllNextInAlternateColor:[Card colorInverseOf:myColor]];
}

- (BOOL)isAllNextFaceUp
    /*"return YES if all nextObject are isFaceUp == YES"*/
{
    return [self isAllNextFace:YES];
}

- (BOOL)isAllNextFace:(BOOL)faceUp
    /*"return YES if all nextObject are isFaceUp == faceUp"*/
{
    if([self isFaceUp] != faceUp) {
        return NO;
    }
    if([self nextObject] == nil) {
        return YES;
    }
    return [nextObject isAllNextFace:faceUp];
}
    /*"Copying"*/
- (CardPile*)copyWithZone:(NSZone*)zone
    /*"Copy the Pile but not the card itself."*/
{
    id cpy = [super copyWithZone:zone];
    [cpy setFaceUp:[self isFaceUp]];
    return cpy;
}

- (void) addDeck
/*"
   Adds a full deck of cards to the top of the pile, neatly ordered.
"*/
{
    id cpyDeck = [[[self class] fullDeck] copy];
    [self appendObjects:cpyDeck];
    [cpyDeck release];
}

- (void) flipCard
    /*" flip the face of the current card only"*/
{
    [self setFaceUp:(![self isFaceUp])];
}

- (void) flip
/*"
   Flip the orientation of all nextObject's cardPile.
"*/
{
    [self flipCard];
    [[self nextObject] flip];
}

- (void)swapCardWithPile:(CardPile*)otherPile
    /*" swap the cardPile object value from a CardPile to the other. "*/
{
    [self swapValueWithLink:otherPile];
}

- (void)swapValueWithLink:otherPile
    /*" swap position of 2 card "*/
{
    BOOL myFace = [self isFaceUp];
    [super swapValueWithLink:otherPile];
    [self setFaceUp:[otherPile isFaceUp]];
    [otherPile setFaceUp:myFace];
}

- (void) empty
    /*" set the current card to nil, and nextPile to nil "*/
{
    [super empty];
    [self setFaceUp:NO];
}

- (void)removeFromHolderList
    /*" remove self from the holder's list"*/
{
    [[self holder] setNextObject:nil];
    [self changed];
}

- (id) initWithCoder:(NSCoder *)aDecoder
/*"
   Unarchives a CardPile instance.
"*/
{
    const char *boolType = @encode(BOOL);
    [super initWithCoder:aDecoder];
    [aDecoder decodeValueOfObjCType:boolType at:&isFaceUp];
    return self;
}

- (void) encodeWithCoder:(NSCoder *)aCoder
/*"
   Archives a CardPile instance.
"*/
{
    const char *boolType = @encode(BOOL);
    [super encodeWithCoder:aCoder];
    [aCoder encodeValueOfObjCType:boolType at:&isFaceUp];
}

@end

@implementation CardPile (Compatibility)


- (int)indexOfCard:(Card*)crd
    /*" return the index from the current pile "*/
{
    return [self indexOfObjectValue:crd];
}

- (int)absoluteIndexOfCard:(Card*)crd
    /*" return the index from the bottom pile. "*/
{
    return [self absoluteIndexOfObjectValue:crd];
}


- (Card*)card
    /*"return the current card"*/
{
    NSLog(@"Deprecated method: [CardPile card]");
    return (Card*)[self value];
}

- (void)setCard:crd
    /*" set the current card. "*/
{
    NSLog(@"Deprecated method: [CardPile setCard]");
    [self setValue:crd];
}

- (CardPile*)nextPile
    /*"return the next pile"*/
{
    NSLog(@"Deprecated method: [CardPile nextPile]");
    return [self nextObject];
}

- (void)setNextPile:(CardPile*)nextCards
    /*" set the next pile "*/
{
    NSLog(@"Deprecated method: [CardPile setNextPile:(CardPile*)cp]");
    [self setNextObject:nextCards];
}

- (CardSuit)cardSuit
{
    return [[self value] cardSuit];
}

- (void)setCardSuit:(CardSuit)aSuit
{
    [[self value] setCardSuit:aSuit];
}

- (CardValue)cardValue
{
    return [[self value] cardValue];
}

- (void)setCardValue:(CardValue)aValue
{
    [[self value] setCardValue:aValue];
}

- (CardColor)cardColor;
{
    return [[self value] cardColor];
}

- initForCardSize:(int)size
{
    return [self init];
}

// - (void)setCardSize:(int)sze  { };

- (void)addCopyOfPile:(CardPile*)p
{
    [self appendObjects:[p copy]];
}

@end

@interface CardPile (private)
- (NSString*)description;
- (NSString*)listDescription;
@end

@implementation CardPile (private)

- (NSString*)description
{
    return [NSString stringWithFormat:@"<%@:%d> %@", [self class], self, [self listDescription]];
}

- (NSString*)listDescription
{
    id rest;
    if([self nextObject] == nil) {
        rest = @"";
    } else {
        rest = [nextObject listDescription];
    }
    return [NSString stringWithFormat:@"(%d)%@,\n\t\t %@",
        [self isFaceUp],
        [[self value] description],
        rest];
}
@end

