/* GameModel.m created by veilljf on Wed 02-Dec-1998 */

#import "GameModel.h"
#import "CardPile.h"
#import <Foundation/Foundation.h>
#import <AppKit/NSApplication.h>

NSString * GAME_WIN = @"GameModel WIN";
NSString * STATISTIC_CHANGED = @"GameStatisticChanged";

@implementation GameModel

/*" GameModel is an Abstract superclass that define the primary interface between a GameModel and a GameGui.
Every GameModel have it's own undo manager, that is saved and restored with the rest of the game model.

Subclass _MUST_ overide the +objectDict class method to return a dictionary of obejct that the game model know of and can give to the GameGui object, or any other View/Controlller-like object.
"*/

- init
    /*" initialize the Model object with a new undo manager."*/
{
    id undo;
    [super init];
    undo = [[NSUndoManager alloc] init];
    [self setUndoManager:undo];
    [undo release];
    return self;
}

- (void)dealloc
{
    [undoManager release];
    [super dealloc];
}

- undoManager
{
    return undoManager;
}

- (void)setUndoManager:(NSUndoManager*)um
{
    if(um != undoManager) {
        [undoManager release];
        undoManager = [um retain];
    }
}

- objectForName:(NSString*)name
/*" return the object named:  This method is used by GUI object to get references over various part of the model."*/
{
    // Subclass responsability
    return nil;
}

- (CardPile*)simpleDeck
    /*" Convenience method that return an ordered single deck of card ."*/
{
    id deal = [[[CardPile alloc] init] autorelease];
    [deal addDeck];
    [[deal nextObject] shuffleValue];
    return [deal nextObject];
}

- (CardPile*)doubleDeck
    /*" Convenience method that return an ordered double deck of card ."*/
{
    id deal = [[[CardPile alloc] init] autorelease];
    [deal addDeck];
    [[deal nextObject] shuffleValue];
    [deal addDeck];
    [[deal nextObject] shuffleValue];
    return [deal nextObject];
}

- (CardPile*)createNewDeal
    /*" by default return a simple full deck shuffled. "*/
{
    id deal = [self simpleDeck];
    [deal shuffleValue];
    return deal;
}

- (void) newGame
{
    [self setupGameUsingDeal:[self createNewDeal]];
}

- (void)restartGame
{
    id undo = [self undoManager];
    while([undo canUndo]) {
        [undo undo];
    }
}

- (void)startGame
/* Do nothing, subclass possibility to overwride */
{
}

- (BOOL) hasBeenPlayed
/* To return YES if the game has been played.  To know that we check if the Undo/Redo stack is empty or not.*/
{
    return ([undoManager canUndo] || [undoManager canRedo]);
}

- (void)endGame:(NSNotification*)notif
    /* Accumulate the number of game lost. */
{
    if([self hasBeenPlayed]) { // if the game has been played
        if(![self isWinning]) { // if it's not a won game
                                // then record as a played game
            [self accumulatePlayed];
        }
    }
}

- (void)win
    /* Post a GAME_WIN notification, and accumulate the number of game won. */
{
    [[NSNotificationCenter defaultCenter]
        postNotificationName:GAME_WIN object:self];
    [self accumulateWin];
}

- (void)setupGameUsingDeal:(CardPile*)deal
/*" subclass responsability "*/
{
    // subclass behavior should be here, then call super after self initialisation
    [[self undoManager] removeAllActions];
    [self startGame];
}

- (BOOL)isWinning
    /*" return YES if the current state define a winning game !
    Always override (unless your game is impossible to win). "*/
{
    return NO;
}

- (void) checkForWin
    /*"    Called to check the state of the game.  If found that the game is currently won -(BOOL)isWinning, then we show the Winning panel, or wathever you overide with -(void)win "*/
{
    if ([self isWinning])
    {
        [self win];
    }
}

- (void) accumulateWin
{
    NSUserDefaults *d = [NSUserDefaults standardUserDefaults];
    id prefName = [self gamePrefWithName:@"LifetimeWins"];
    [d setInteger:[d integerForKey:prefName] + 1 forKey:prefName];

    [self statisticsChanged];
}

- (void) accumulatePlayed
{
    NSUserDefaults *d = [NSUserDefaults standardUserDefaults];
    id prefName = [self gamePrefWithName:@"LifetimePlayed"];
    [d setInteger:[d integerForKey:prefName] + 1 forKey:prefName];

    [self statisticsChanged];
}

- (void) statisticsChanged
{
    [[NSNotificationCenter defaultCenter]
        postNotificationName:STATISTIC_CHANGED object:self];
}
    
- (NSString *)gamePrefWithName:(NSString *)prefName
{
    id modelName = [[self class] description];
    NSRange range = [modelName rangeOfString:@"Model"];
    id clName = [modelName substringToIndex:range.location];
    return [NSString stringWithFormat:@"%@-%@", clName, prefName];
}

- (id) initWithCoder:(NSCoder *)aDecoder
    /*"
    Unarchives a GameModel instance.  This will only archive the NSUndo list.
     "*/
{
    [self setUndoManager:[[aDecoder decodeObject] retain]];
    return self;
}

- (void) encodeWithCoder:(NSCoder *)aCoder
    /*"
    Archives a GameModel instance.  This will only archive the NSUndo list.
     Will archive as a lost game if not already win.
     "*/
{
    [aCoder encodeObject:[self undoManager]];
}

@end
