/* LinkList.m created by veilljf on Mon 09-Nov-1998 */

#import "LinkList.h"
#import <Foundation/Foundation.h>


#if defined (WIN32)
#define random() rand()
#define srandom(x) srand(x)
#else
#import <libc.h>
#endif


@implementation LinkList

/*"
	A basic linked list. Each node has a value and a nextObject, basicaly
 	pointing to the next link.
        
"*/

- (void)dealloc
    /*" release the value and the nextObject. "*/
{
    [value release];
    [super dealloc];
}

- value
    /*" each node has a value, return the current value."*/
{
    return value;
}

- (void)setValue:(id)v
    /*" set the value of the current node."*/
{
    if(value != v ) {
        [value release];
        value = [v retain];   
    }
}

- (void)empty
    /*" empty the list, nil the current value, and the nextObject."*/
{
    [self setValue:nil];
    [super empty];
}

/*" Manipulating list "*/
- (unsigned)count
    /*" return the size of the list. return 0 if empty."*/
{
    return (nextObject == nil) ? 1 : 1 + [nextObject count];
}

- (LinkList*)dealLastObject
    /*" return the last object, and remove it from the list. if self is the last object, then simply return it."*/
{
    id result = self;
    if ([nextObject nextObject] == nil) {
        result = (nextObject == nil) ? self : nextObject;
        [result retain];
        [result autorelease];
        [self setNextObject:nil];
    } else {
        result = [nextObject dealLastObject];
    }
    return result;
}

- (LinkList*)lastObject
    /*" return the last node of the list.  the node remain in the list."*/
{
    return ([self nextObject] == nil) ? self : [nextObject lastObject];
}

- (void) shuffleValue
    /*" Randomly re-order all objects value contained in the list "*/
{
    LinkList*	swap1;
    LinkList*	swap2;
    int		counter;
    int		swapIndex;
    int		max = [self count];

    for (counter = 0; counter < max - 1; counter++)
    {
        // changed by erikk for more true randomness to the shuffle...
        swapIndex = random() % (max);
        if (swapIndex == counter)
            continue;

        swap1 = [self objectAtIndex:counter];
        swap2 = [self objectAtIndex:swapIndex];
        [swap1 swapValueWithLink:swap2];
    }
}

- (void)swapValueWithLink:(LinkList*)otherLink
    /*" swap value from two nodes: self and otherLink.  This is used to shuffle the list. "*/
{
    id myValue = [[self value] retain];
    [self setValue:[otherLink value]];
    [otherLink setValue:myValue];
    [myValue release];
}


- (int)indexOfObjectValue:val
    /*" return the index from the current node where the value isEqual: to val.  
        Where index of 0 is the current node.

                EXCEPTION:
        raise an exception if the val is not in the list.
    "*/
{
    if(![val isEqual:[self value]]) {
        if([self nextObject] != nil) {
            return 1 + [[self nextObject] indexOfObjectValue:val];
        } else {
            [NSException raise:@"NOT_FOUND" format:@"Object not found: %@", val];
        }
    }
    return 0;
}

- (NSArray*)asNSArray
/*" return the list as an nsarray."*/
{
    id array = [NSMutableArray array];
    id iterator = self;
    do {
        [array addObject:[iterator value]];
        iterator = [iterator nextObject];
    } while(iterator != nil);
    return array;
}

/*"Copying"*/
- copyWithZone:(NSZone*)zone
    /*"Copy the node but not the value itself."*/
{
    id cpy = [super copyWithZone:zone];
    [cpy setValue:[self value]];
    return cpy;
}

- (id) initWithCoder:(NSCoder *)aDecoder
/*"
   Unarchives a node instance.
"*/
{
    [super initWithCoder:aDecoder];
    [self setValue:[[aDecoder decodeObject] retain]];
    return self;
}

- (void) encodeWithCoder:(NSCoder *)aCoder
/*"
   Archives a node instance.
"*/
{
    [super encodeWithCoder:aCoder];
    [aCoder encodeObject:value];
}

@end
