/* CardImager.m created by veilljf on Wed 04-Nov-1998 */

#import "TiffCardsImager.h"
#import "Card.h"

#import <Foundation/Foundation.h>
#import "CardPile.h"


/*---------------------------------------------------------------------------
|
|    CardImager Globals
|
|    The arrays _STCardsPipPositions and _STCardsCardPips provide information about
|    the positioning of pips on cards.  There are 17 possible positions,
|    described in pipPositions, which are selectively refrerred to for each
|    of 13 possible card values by a character string.
|
|    The letters A-Q are used to represent pips present in the given card.
|
|    The id's _CardFront, _CardBack etc... are images shared by all instances
|    of Card
|
\----------------------------------------------------------------------------*/

static NSPoint _STCardsPipPositions[] = {
{13, 91}, {30, 91}, {47, 91},
{30, 78},
{30, 71},
{13, 64}, {47, 64},
{13, 52}, {30, 52}, {47, 52},
{13, 39}, {47, 39},
{30, 32},
{30, 25},
{13, 13}, {30, 13}, {47, 13}
};

static char *_STCardsCardPips[] = {
    "I",
    "BP",
    "BIP",
    "ACOQ",
    "ACIOQ",
    "ACHJOQ",
    "ACEHJOQ",
    "ACEHJMOQ",
    "ACFGIKLOQ",
    "ACDFGKLNOQ",
    "CO",
    "CO",
    "AQ"
};




// static NSImage* cardCache[52];



@implementation TiffCardsImager
/*"
   The CardImager is an object that can image a Card object.

   The card images can either be drawn by compositing the partial card images
   contained in the TIFF files in the CardSet, or the 52 cards in a deck can
   be pre-drawn and stored in shared NSImage caches (which are private class
   variables) for reuse. The shared NSImage caches increase performance, but
   with the penalty of increased memory usage.

     ToDo: - reorganize the images initialization, remove them from
             the +initialize method, so that the memory deallocation
             will be easier if we dynamicaly change from different cardImager

"*/

- init
/*" initialize the CardImager.  "*/
{
    return [super init];
}

- (double)cardWidth
{
    return   80.0;
}

- (double)cardHeight
{
    return  126.0;
}


- (void) drawContentsOfCard:(CardPile*)card at:(NSPoint)thePoint
{
    [self drawContentsForSuit:[card cardSuit]
                        value:[card cardValue]
                       faceUp:[card isFaceUp]
                           at:thePoint];
}


- (void) drawCard:(CardPile*)card at:(NSPoint)thePoint
/*"
   Draws the card, both outline and contents, at thePoint, by calling
   #drawOutlineAt: and #drawContentsAt:. If the 52 cards of the deck have been
   pre-drawn, the shared NSImage cache is used. Otherwise, the card is drawn
   by compositing the partial images.
"*/
{
    if(_preDrawnImagesAvailable && [card isFaceUp]) {
        [cardCache[([card cardSuit]*13)+[card cardValue]] compositeToPoint:thePoint
                operation:NSCompositeSourceOver];
        return;
    }
    [self drawOutlineOfCard:card at:thePoint];
    [self drawContentsOfCard:card at:thePoint];
}


- (void) drawOutlineOfCard:(CardPile*)card at:(NSPoint)thePoint
/*"
   Draws the card's border only at thePoint.  This method do not draw the content.
"*/
{
    [_CardFrontImage compositeToPoint:thePoint operation:NSCompositeSourceOver];
}

@end




@implementation TiffCardsImager(private)

//-------------------------------------------------------------------
//	Initializing the class
//-------------------------------------------------------------------
- subImagesDict
{
    return [[NSMutableDictionary alloc] init];
}

- (void)cacheSubImages
/*"
   Initializes the class object. The partial card images from CardSet TIFFS
   are loaded here.
"*/
{
    NSBundle* bundle = [NSBundle bundleForClass:[self class]];
    NSString* imagePath = nil;

    id subImgDict = [self subImagesDict];
    id dictEnum = [subImgDict keyEnumerator];
    id current;

    while (current = [dictEnum nextObject]) {
        id resourceName = [subImgDict objectForKey:current];
        imagePath = [bundle pathForResource:resourceName ofType:@"tiff"];
         [[NSImage alloc] initWithContentsOfFile:imagePath];
    }

    imagePath = [bundle pathForResource:@"sCard" ofType:@"tiff"];
    _CardFrontImage = [[NSImage alloc] initWithContentsOfFile:imagePath];

    imagePath = [bundle pathForResource:@"sBack" ofType:@"tiff"];
    _CardBackImage = [[NSImage alloc] initWithContentsOfFile:imagePath];

    imagePath = [bundle pathForResource:@"sKingClubs" ofType:@"tiff"];
    _CardKingClubsImage = [[NSImage alloc] initWithContentsOfFile:imagePath];

    imagePath = [bundle pathForResource:@"sKingDiamonds" ofType:@"tiff"];
    _CardKingDiamondsImage = [[NSImage alloc] initWithContentsOfFile:imagePath];

    imagePath = [bundle pathForResource:@"sKingHearts" ofType:@"tiff"];
    _CardKingHeartsImage = [[NSImage alloc] initWithContentsOfFile:imagePath];

    imagePath = [bundle pathForResource:@"sKingSpades" ofType:@"tiff"];
    _CardKingSpadesImage = [[NSImage alloc] initWithContentsOfFile:imagePath];

    imagePath = [bundle pathForResource:@"sQueenClubs" ofType:@"tiff"];
    _CardQueenClubsImage = [[NSImage alloc] initWithContentsOfFile:imagePath];

    imagePath = [bundle pathForResource:@"sQueenDiamonds" ofType:@"tiff"];
    _CardQueenDiamondsImage = [[NSImage alloc] initWithContentsOfFile:imagePath];

    imagePath = [bundle pathForResource:@"sQueenHearts" ofType:@"tiff"];
    _CardQueenHeartsImage = [[NSImage alloc] initWithContentsOfFile:imagePath];

    imagePath = [bundle pathForResource:@"sQueenSpades" ofType:@"tiff"];
    _CardQueenSpadesImage = [[NSImage alloc] initWithContentsOfFile:imagePath];

    imagePath = [bundle pathForResource:@"sJackClubs" ofType:@"tiff"];
    _CardJackClubsImage = [[NSImage alloc] initWithContentsOfFile:imagePath];

    imagePath = [bundle pathForResource:@"sJackDiamonds" ofType:@"tiff"];
    _CardJackDiamondsImage = [[NSImage alloc] initWithContentsOfFile:imagePath];

    imagePath = [bundle pathForResource:@"sJackHearts" ofType:@"tiff"];
    _CardJackHeartsImage = [[NSImage alloc] initWithContentsOfFile:imagePath];

    imagePath = [bundle pathForResource:@"sJackSpades" ofType:@"tiff"];
    _CardJackSpadesImage = [[NSImage alloc] initWithContentsOfFile:imagePath];

    imagePath = [bundle pathForResource:@"sAceOfSpades" ofType:@"tiff"];
    _CardAceOfSpadesImage = [[NSImage alloc] initWithContentsOfFile:imagePath];

    imagePath = [bundle pathForResource:@"sSymbols" ofType:@"tiff"];
    _CardSymbolsImage = [[NSImage alloc] initWithContentsOfFile:imagePath];
}

- (void) drawCardImages
/*"
   Creates shared NSImage caches of all 52 cards in the deck. Uses the instance
   method #preDrawCards: to do the actual drawing.
"*/
{
    int 	i,j;
    NSSize 	aSize;

    if(_preDrawnImagesAvailable) return;	// don't draw again
    aSize = [_CardFrontImage size];
    for(i=0; i<4; i++){
        for(j=0; j<13; j++){
            cardCache[(i*13)+j] = [[self imageForSuit:i value:j] retain];
        }
    }
    _preDrawnImagesAvailable = YES;
}

- (NSImage*)imageForSuit:(int)suit value:(int)value
/*"Return an image for the card defined by the suit and value"*/
{
    id image = [_CardFrontImage copy];
    NSPoint thePoint={0.,0.};
    [image lockFocus];
    [self drawContentsForSuit:suit  value:value faceUp:YES at:thePoint];
    [image unlockFocus];
    return [image autorelease];
}

- (void) freeCardImages
/*"
   Frees the shared NSImage caches of all 52 cards.
"*/
{
    int i;

    if(!_preDrawnImagesAvailable) return;	// don't free again
    for(i=0; i<52; i++){
        [cardCache[i] release];
    }
    _preDrawnImagesAvailable = NO;
}


- (void) setCardBackImage:theImage
/*"
   Sets a custom image used to draw the back of the card.
"*/
{
    if(theImage){
        [_CardBackImage autorelease];
        _CardBackImage = [theImage copyWithZone:[self zone]];
    }
}


- (void) setCardBack:(CardBack)aBack
/*"
   Sets the card back used to draw the back of the card.
"*/
{
    NSBundle* bundle = [NSBundle bundleForClass:[self class]];
    NSString* imagePath = nil;

    imagePath = [bundle pathForResource:@"sBack" ofType:@"tiff"];

    if (imagePath != nil) {
        NSImage* newImage = [[NSImage alloc] initWithContentsOfFile:imagePath];
        if (newImage != nil) {
            [self setCardBackImage:newImage];
        }
    }
}

- (void) drawContentsForSuit:(CardSuit)suit
                       value:(CardValue)value
                      faceUp:(BOOL)faceUp
                          at:(NSPoint)thePoint
/*"
   Draws the contents of the card (but not the border) at thePoint.
"*/
{
    if (!faceUp)
    {

        /*------------------------------------------------------
        |
        |    Draw the back of the card
        |
        \------------------------------------------------------*/

        NSPoint destPoint = {thePoint.x + 6, thePoint.y + 5};

        [_CardBackImage compositeToPoint:destPoint
                               operation:NSCompositeSourceOver];

    } else {

        /*---------------------------------------------------------
        |
        |    Draw the upper left / lower right value indicator
        |
        \---------------------------------------------------------*/

    {
        NSRect sourceRect = {{value * 16, 60}, {8, 15}};
        NSPoint destPoint = {thePoint.x + 3, thePoint.y + 106};

        if (([Card colofOfSuit:suit] == CS_RED)) {
            sourceRect.origin.x += 8;
        }
        [_CardSymbolsImage compositeToPoint:destPoint fromRect:sourceRect operation:NSCompositeSourceOver];

        sourceRect.origin.y -= 16;
        destPoint.x = thePoint.x + 69;
        destPoint.y = thePoint.y + 6;

        [_CardSymbolsImage compositeToPoint:destPoint fromRect:sourceRect operation:NSCompositeSourceOver];
    }

        /*----------------------------------------------------------
        |
        |    Draw the upper left / lower right suit indicator
        |
        \----------------------------------------------------------*/

        {
            NSRect sourceRect = {{suit * 10, 10}, {10, 12}};
            NSPoint destPoint = {thePoint.x + 2, thePoint.y + 94};

            [_CardSymbolsImage compositeToPoint:destPoint fromRect:sourceRect		operation:NSCompositeSourceOver];
        sourceRect.origin.x += 40;
        destPoint.x = thePoint.x + 69;
        destPoint.y = thePoint.y + 21;
            [_CardSymbolsImage compositeToPoint:destPoint fromRect:sourceRect 	operation:NSCompositeSourceOver];
        }


        /*---------------------------------------------------------
        |
        |    Draw special card images for face cards and the
        |    ace of spades
        |
        \---------------------------------------------------------*/

        {
            NSPoint destPoint = {thePoint.x + 12, thePoint.y + 12};

            switch ( value )
            {
                case CS_JACK:
                    switch ( suit )
                    {
                        case CS_CLUBS:
                            [_CardJackClubsImage compositeToPoint:destPoint operation:NSCompositeSourceOver];
                            break;
                        case CS_DIAMONDS:
                            [_CardJackDiamondsImage compositeToPoint:destPoint operation:NSCompositeSourceOver];
                            break;
                        case CS_HEARTS:
                            [_CardJackHeartsImage compositeToPoint:destPoint operation:NSCompositeSourceOver];
                            break;
                        case CS_SPADES:
                            [_CardJackSpadesImage compositeToPoint:destPoint operation:NSCompositeSourceOver];
                            break;
                    }
                    return;

                case CS_QUEEN:
                    switch ( suit )
                    {
                        case CS_CLUBS:
                            [_CardQueenClubsImage compositeToPoint:destPoint operation:NSCompositeSourceOver];
                            break;
                        case CS_DIAMONDS:
                            [_CardQueenDiamondsImage compositeToPoint:destPoint operation:NSCompositeSourceOver];
                            break;
                        case CS_HEARTS:
                            [_CardQueenHeartsImage compositeToPoint:destPoint operation:NSCompositeSourceOver];
                            break;
                        case CS_SPADES:
                            [_CardQueenSpadesImage compositeToPoint:destPoint operation:NSCompositeSourceOver];
                            break;
                    }
                    return;

                case CS_KING:
                    switch ( suit )
                    {
                        case CS_CLUBS:
                            [_CardKingClubsImage compositeToPoint:destPoint operation:NSCompositeSourceOver];
                            break;
                        case CS_DIAMONDS:
                            [_CardKingDiamondsImage compositeToPoint:destPoint operation:NSCompositeSourceOver];
                            break;
                        case CS_HEARTS:
                            [_CardKingHeartsImage compositeToPoint:destPoint operation:NSCompositeSourceOver];
                            break;
                        case CS_SPADES:
                            [_CardKingSpadesImage compositeToPoint:destPoint operation:NSCompositeSourceOver];
                            break;
                    }
                    return;

                default:
                    break;
            }

            /*----------------------------------------------------
                |
                |    Not only draw the special image for the ace of
                |    spades, but avoid the standard suit pip drawing
                |
                \----------------------------------------------------*/

            if (( value == CS_ACE) && ( suit == CS_SPADES)) {
                [_CardAceOfSpadesImage compositeToPoint:destPoint 		operation:NSCompositeSourceOver];
                return;
            }
        }

        /*------------------------------------------------------
        |
        |    Draw suit pips on the card based on patterns
        |    defined by the _CardCardSymbols and
        |    _STCardsCardPips arrays
        |
        \------------------------------------------------------*/

        {
            NSRect suitRect = {{suit * 20, 23}, {20, 21}};
            NSRect upsideSuitRect = {{80 + suit * 20, 23}, {20, 21}};
            char *pipString = _STCardsCardPips[value];

            while (*pipString) {
                NSPoint tempPoint = _STCardsPipPositions[*pipString - 'A'];
                tempPoint.x += thePoint.x;
                tempPoint.y += thePoint.y;
                [_CardSymbolsImage compositeToPoint:tempPoint                                       fromRect:(*pipString < 'K') ? suitRect : upsideSuitRect	operation:NSCompositeSourceOver];
                pipString++;
            }
        }
    }
}


@end
