/* indent:4  tabsize:8  font:fixed-width */

#import "SpiderModel.h"
#import "localstrings.h"
#import "SpiderDrawLogic.h"
#import "SpiderSuitLogic.h"
#import "SpiderGameLogic.h"
#import <Solitaire/CardPileHolder.h>
#import <Solitaire/CardPile.h>

@implementation SpiderModel


- init
{
    int i = 0;
    // create logic objects
    id gamePileLogic = [[SpiderGameLogic alloc] init];
    id suitPileLogic = [[SpiderSuitLogic alloc] init];
    id drawPileLogic = [[SpiderDrawLogic alloc] init];

    [super init];

    [gamePileLogic setGameModel:self];
    [suitPileLogic setGameModel:self];
    [drawPileLogic setGameModel:self];

    // create the draw pile holder
    drawPileHolder = [[CardPileHolder alloc] initWithModel:self logic:drawPileLogic];
    
    // then create the gamePileHolders
    for(i = 0 ; i < 10 ; i++ ) {
        gamePileHolders[i] = [[CardPileHolder alloc] initWithModel:self logic:gamePileLogic];
    }
    // then create the suitPileHolders
    for(i = 0 ; i < 8 ; i++ ) {
        suitPileHolders[i] = [[CardPileHolder alloc] initWithModel:self logic:suitPileLogic];
    }

    [self newGame];
    [gamePileLogic release];
    [suitPileLogic release];
    [drawPileLogic release];

    return self;
}

- (void)dealloc
{
    int i = 0;
    [drawPileHolder release];
    for(i = 0; i < 8 ; i++ ) {
        [suitPileHolders[i] release];
    }
    for(i = 0; i < 10 ; i++ ) {
        [gamePileHolders[i] release];
    }
    [super dealloc];
}

- (CardPileHolder**)gamePileHolders
{
    return gamePileHolders;
}


- (CardPileHolder**)suitPileHolders
{
    return suitPileHolders;
}

- (CardPileHolder*)drawPileHolder
{
    return drawPileHolder;
}

- (CardPile*)createNewDeal
    /*" A Spider deal is 2 full deck shuffled. "*/
{
    id deal = [super doubleDeck];
    [deal shuffleValue];
    return deal;
}

- (void)setupGameUsingDeal:(CardPile*)deal
    /*" Setup a new using this specified Deal. "*/
{
    int 			cycle;
    [drawPileHolder empty];
    [drawPileHolder setNextObject:deal];

    //  Initialize and deal cards to the 10 "user piles"
    for (cycle = 0; cycle < 10; cycle++)
    {
        [gamePileHolders[cycle] empty];
    }
    for (cycle = 0; cycle < 8; cycle++)
    {
        [suitPileHolders[cycle] empty];
    }

    for(cycle = 0; cycle < 44 ; cycle++ ) {
        [deal dealLastObjectFaceDownTo:gamePileHolders[cycle % 10]];
    }

    for (cycle = 0; cycle < 10; cycle++)
    {
        [[gamePileHolders[cycle] lastObject] flip];
    }
    [super setupGameUsingDeal:deal];
}

- (BOOL)isWinning
    /*" Called to check the state of the game. "*/
{
    int i;
    int sum = 0;
    for(i = 0 ; i < 8 ; i++ ) {
        sum = sum + [suitPileHolders[i] count];
    }
    return (sum == 104);
}

- (void) tryToMoveCardPile:(CardPile*)aCard fromPile:(CardPileHolder*)cardPile
    /*" try to find a good match for that pile.  Look at cardPile next (assuming a sequence) to the one clicked on.
    It will first try to send the pile to the suit, if it's a completed suit.
    Then try to best match a cardPile whith the same suit, then any suit, then any empty suit. "*/
{
    id dest = nil;
    int destLevel = 0;
    int i = 0;
    int startPileIdx;

    // first is it a King ?, is so and the full suit is there, then move it to a suit
    if([aCard count] == 13)
    {
        // move to the next empty suit
        for(i = 0; i < 8 ; i++ )
        {
            if([suitPileHolders[i] count] == 0)
            {
                [aCard moveAtLastOf:suitPileHolders[i]];
                [self checkForWin];
                return;
            }
        }
    }
    startPileIdx = [self indexOfGamePileHolder:cardPile];
    for(i = (startPileIdx + 9) % 10; i != startPileIdx ; i=( i + 9 )%10 ) {
        CardPile* lcard = [gamePileHolders[i] lastObject];
        if(destLevel < 2) {
            if(destLevel < 1) {
                // first look for anything empty pile or anything else.
                if(lcard == nil)
                {
                    dest = gamePileHolders[i];
                }
            }
            // Good match, look for any following card value
            if([lcard isFaceUp]) {
                if( ([lcard cardValue] - 1) == [aCard cardValue])
                {
                    dest = gamePileHolders[i];
                    destLevel = 1;
                }
            }
        }
        // Best match :  look for same suit.
        if([lcard isFaceUp]) {
            if( ([lcard cardSuit] == [aCard cardSuit]) &&
                (([lcard cardValue] - 1) == [aCard cardValue]) )
            {
                dest = gamePileHolders[i];
                destLevel = 2;
            }
        }
    }
    if(dest) {
        id ho = [aCard holder];
        [aCard moveAtLastOf:dest];
        [[ho lastObject] setFaceUp:YES]; // [ListHolder lastObject] will return nil !
    }
}

- (int)indexOfGamePileHolder:(CardPileHolder*)ho
    /* this is just to know which pile we are talking to, considering an order from left to right.*/
{
    int idx;
    for(idx = 0 ; idx < 10 ; idx++) {
        if (gamePileHolders[idx] == ho) {
            break;
        }
    }
    return idx;
}

- (id) initWithCoder:(NSCoder *)aDecoder
    /*"
    Unarchives a GameModel instance. archive the state of all pileHolders
     "*/
{
    int i = 0;
    [super initWithCoder:aDecoder];
    drawPileHolder = [[aDecoder decodeObject] retain];

    for(i = 0 ; i< 8 ; i++) {
        suitPileHolders[i] = [[aDecoder decodeObject] retain];
    }
    for(i = 0 ; i< 10 ; i++) {
        gamePileHolders[i] = [[aDecoder decodeObject] retain];
    }
    return self;
}

- (void) encodeWithCoder:(NSCoder *)aCoder
    /*"
    Archives a GameModel instance.  Unarchive the state of all pileHolders
     "*/
{
    int i = 0;
    [super encodeWithCoder:aCoder];
    [aCoder encodeObject:drawPileHolder];
    for(i=0 ; i < 8 ; i++ ) {
        [aCoder encodeObject:suitPileHolders[i]];
    }
    for(i=0 ; i < 10 ; i++ ) {
        [aCoder encodeObject:gamePileHolders[i]];
    }
}

@end

