//
//  MiscMergeCommand.m
//            Written by Don Yacktman and Carl Lindberg
//        Copyright 1998 by Don Yacktman and Carl Lindberg.
//                     All rights reserved.
//      This notice may not be removed from this source code.
//
//	This header is included in the MiscKit by permission from the author
//	and its use is governed by the MiscKit license, found in the file
//	"License.rtf" in the MiscKit distribution.  Please refer to that file
//	for a list of all applicable permissions and restrictions.
//	

#import "MiscMergeCommand.h"
#import <Foundation/NSCharacterSet.h>
#import <Foundation/NSUtilities.h>
#import "NSScanner+MiscMerge.h"

@implementation MiscMergeCommand
/*"
 * The MiscMergeCommand class implements a merge command.  Since the
 * MiscKit merge engine can dynamically add new commands, it is possible to
 * create custom subclasses of MiscMergeCommand to implement new
 * functionality in the engine.  The merge engine already implements most
 * of the commands a user would want, but certain applications may wish to
 * override those commands, replace them, or add new commands.
 * 
 * To create a subclass, two methods, -#{executeForMerge:} and either
 * -#{parseFromString:template} or -#parseFromScanner:template: need to be
 * implemented.  The parse method is expected to break up a text string
 * into whatever arguments a particular MiscMergeCommand subclass needs in
 * order to function.  The execute method performs, during a merge,
 * whatever special task the command is designed to do.
 * 
 * The other methods in this object may be used by subclasses to aid in
 * parsing the command string.  They can grab key words, conditional
 * operators, an arguments (single word or quoted string).  A special kind
 * of argument, promptable, is also supported.  A promptable argument is
 * expected to have its actual value determined at run time.
 * 
 * When implementing commands, the full API of the MiscMergeEngine object
 * is available.  This allows the programmer to store information in the
 * engine, manipulate the symbol tables used for resolving fields, and
 * alter the output being created by the merge.
 * 
 * The MiscKit source code is a good place to look for examples of how to
 * implement various MiscMergeCommand subclasses.
"*/

/*"
 * This method is called while parsing a merge template.  The text of the
 * full merge command is contained in %{aString}.  The default
 * implementation creates an NSScanner with aString and calls
 * -#parseFromScanner:template:.  If there is no need to parse the string,
 * then override this method; otherwise override the scanner method.
"*/
- (BOOL)parseFromString:(NSString *)aString template:(MiscMergeTemplate *)template
{
	return [self parseFromScanner:[NSScanner scannerWithString:aString] template:template];
}

/*"
 * This method is called while parsing a merge template, typically by
 * -#parseFromString:template:. This method should break %{aScanner}'s
 * string up into keywords, conditionals, and arguments as needed and store
 * the results in instance variables for later use during merges.  Note
 * that returning YES tells the template parsing machinery that all is
 * well.  Return NO if there is an error or the command cannot be properly
 * initialized.  The default implementation returns NO; either this method
 * or -#parseFromString:template: needs to be overridden in subclasses.
"*/
- (BOOL)parseFromScanner:(NSScanner *)aScanner template:(MiscMergeTemplate *)template
{
	return NO;
}

/*"
 * This method is called by the merge engine while it is performing a
 * merge.  The command is expected to perform its specified function when
 * this call is received.
"*/
- (void)executeForMerge:(MiscMergeEngine *)aMerger
{
	// subclass responsibility
}


/*"
 * Attempts to scan past of %{aKeyWord} in %{scanner}. If %{optional} is
 * YES, then no complaint will be made if %{aKeyWord} is missing.  YES or
 * NO is returned to tell the caller if the required key word was found or
 * not, no matter what the value of %{optional} was.
"*/
- (BOOL)eatKeyWord:(NSString *)aKeyWord fromScanner:(NSScanner *)scanner
	isOptional:(BOOL)optional
{
	BOOL wasCaseSensitive = [scanner caseSensitive];
	BOOL foundKeyword;
	
	[scanner setCaseSensitive:NO];
	foundKeyword = [scanner scanString:aKeyWord];

	if (!foundKeyword && !optional) [self error_keyword:aKeyWord];
	[scanner setCaseSensitive:wasCaseSensitive];

	return foundKeyword;
}

- _getQuotedArgumentStringFromScanner:(NSScanner *)scanner toEnd:(BOOL)endFlag
{
	NSMutableString *parsedString = [[[NSMutableString alloc] init] autorelease];
	NSCharacterSet *stopSet = [NSCharacterSet characterSetWithCharactersInString:@"\\\""];
	NSString *workString = nil;
	
	[scanner scanCharacter:'"']; //scan past open quote if its there

	while (![scanner isAtEnd])
	{
		if ([scanner scanUpToCharactersFromSet:stopSet intoString:&workString])
			[parsedString appendString:workString];
		
		if ([scanner scanCharacter:'\\'])
		{
			if ([scanner scanLetterIntoString:&workString])
			{
				/*" Leave the backslash if not quoting the '"' character */
				// Need to allow quoted delimiters here as well?
				if (![workString isEqualToString:@"\""])
					[parsedString appendString:@"\\"];

				[parsedString appendString:workString];
			}
		}
		else if ([scanner scanCharacter:'"'])
		{
			if (endFlag && ![scanner isAtEnd])
			{
				[self error_closequote];
				[parsedString appendString:[scanner remainingString]];
			}

			return parsedString;
		}
	}
	
	[self error_closequote];
	return parsedString;
}

/*"
 * Attempts to parse an argument from %{scanner}.  If %{endFlag} is set,
 * then the remaining string of %{scanner} will be assumend to be the
 * required argument.  Otherwise, if an argument contains whitespace, it
 * should be surrounded by quotation marks ("). The parsed argument will
 * scanned past in %{scanner}.
"*/
- getArgumentStringFromScanner:(NSScanner *)scanner toEnd:(BOOL)endFlag
{
	if ([scanner peekNextCharacter] == '"') {
		return [self _getQuotedArgumentStringFromScanner:scanner toEnd:endFlag];
	}
	else {
		if (endFlag)
		{
			return [scanner remainingString];
		}
		else
		{
			NSCharacterSet *stopSet;
			NSString *workString;

			stopSet = [NSCharacterSet characterSetWithCharactersInString:@" \t\r\n<>!="];
			if ([scanner scanUpToCharactersFromSet:stopSet intoString:&workString])
				return workString;
		}
	}

	return nil;
}

/*"
 * Attempts to parse a promptable argument from %{scanner}.  If the
 * argument begins with a "?" then the argument is a "prompt".  Scans past
 * the parsed argument in %{scanner} and returns it.  Returns nil if the
 * wrong kind of argument was found.  It is expected that a promptable
 * argument's value will be determined at run time by asking the user for
 * the value that should be stored for it.
"*/
- getPromptFromScanner:(NSScanner *)scanner toEnd:(BOOL)endFlag
{
	if ([scanner scanCharacter:'?'])
	{
		return [self getArgumentStringFromScanner:scanner toEnd:endFlag];
	}
	else
	{
		[self error_noprompt];
		return nil;
	}
}

/*"
 * Attempts to parse an argument, which could be promptable, from
 * %{scanner}. If the argument begins with a "?" then the argument is a
 * "prompt".  Otherwise, a regular argument is parsed.  Scans past the
 * parsed argument in %{scanner} and returns it.  %{prompt} is set to YES
 * or NO depending upon what was parsed.
"*/
- getPromptableArgumentStringFromScanner:(NSScanner *)scanner
		wasPrompt:(BOOL *)prompt toEnd:(BOOL)endFlag
{
	if ([scanner peekNextCharacter] == '?') {
		*prompt = YES;
		return [self getPromptFromScanner:scanner toEnd:endFlag];
	} else {
		*prompt = NO;
		return [self getArgumentStringFromScanner:scanner toEnd:endFlag];
	}
	return nil;
}

/*"
 * Attempts to parse a conditional from %{scanner}.  Currently recognized
 * conditionals are:  <>, ><, !=, <=, =<, >=, =>, <, >, ==, =.  Returns the
 * type of conditional found or MiscMergeOperatorNone if an unrecognized
 * conditional is found.  Scans past the parsed conditional in %{scanner}.
"*/
- (MiscMergeConditionalOperator)getConditionalFromScanner:(NSScanner *)scanner
{
//	NSCharacterSet *condSet = [NSCharacterSet characterSetWithCharactersInString:@"=!<> \t\r\n"];
	NSCharacterSet *condSet = [NSCharacterSet characterSetWithCharactersInString:@"=!<>"];
	NSString *condString = nil;

	[scanner scanCharactersFromSet:condSet intoString:&condString];

//  Used to allow spaces/tabs/newlines between chars of the operator (like "< >").
//  I don't like that, so for the moment it's not implemented that way.
//	[condString replaceEveryOccurrenceOfChars:" \t\r\n" with:""];

	if ([condString length] > 2) {
		[self error_conditional:condString];
		return MiscMergeOperatorNone;
	}
	if ([condString length] < 1) return MiscMergeOperatorNone;

	if      ([condString isEqualToString:@"<>"])   return MiscMergeOperatorNotEqual;
	else if ([condString isEqualToString:@"><"])   return MiscMergeOperatorNotEqual;
	else if ([condString isEqualToString:@"!="])   return MiscMergeOperatorNotEqual;
	else if ([condString isEqualToString:@"<="])   return MiscMergeOperatorLessThanOrEqual;
	else if ([condString isEqualToString:@">="])   return MiscMergeOperatorGreaterThanOrEqual;
	else if ([condString isEqualToString:@"=>"])   return MiscMergeOperatorGreaterThanOrEqual;
	else if ([condString isEqualToString:@"=<"])   return MiscMergeOperatorLessThanOrEqual;
	else if ([condString isEqualToString:@">"])    return MiscMergeOperatorGreaterThan;
	else if ([condString isEqualToString:@"<"])    return MiscMergeOperatorLessThan;
	else if ([condString isEqualToString:@"=="])   return MiscMergeOperatorEqual;
	else if ([condString isEqualToString:@"="])    return MiscMergeOperatorEqual;
	else
	{
		[self error_conditional:condString];
		return MiscMergeOperatorNone;
	}
}

/*"
 * This method is called if, while parsing, it is discovered that a
 * conditional is unrecognized.  Prints the name of the merge command
 * class, the text "Unrecognized conditional:", and the -#description of
 * %{theCond} to the console using NSLog().
"*/
- (void)error_conditional:(NSString *)theCond
{
	NSLog(@"%@:  Unrecognized conditional:  \"%@\".", [self class], theCond);
}

/*"
 * This method is called if, while parsing, it is discovered that a
 * required key word is missing.  Prints the name of the merge command
 * class, the text "Missing key word:", and %{aKeyWord} to the console
 * using NSLog().
"*/
- (void)error_keyword:(NSString *)aKeyWord
{
	NSLog(@"%@:  Missing key word:  \"%@\".", [self class], aKeyWord);
}

/*"
 * This method is called if, while parsing, it is discovered that the
 * required prompt is missing.  (Referring to arguments that are
 * promptable.)  Prints the name of the merge command class and the text
 * "Missing prompt." to the console using NSLog().
"*/
- (void)error_noprompt
{
	NSLog(@"%@:  Missing prompt.", [self class]);
}

/*"
 * This method is called if, while parsing, it is discovered that
 * quotations are not matched up properly.  Prints the name of the merge
 * command class and the text "Closing quote missing or spurious extra
 * argument added." to the console using NSLog().
"*/
- (void)error_closequote
{
	NSLog(@"%@:  Closing quote missing or spurious extra argument added.", [self class]);
}

@end
