//
//  MiscMergeTemplate.m
//            Written by Don Yacktman and Carl Lindberg
//        Copyright 1998 by Don Yacktman and Carl Lindberg.
//                     All rights reserved.
//      This notice may not be removed from this source code.
//
//	This header is included in the MiscKit by permission from the author
//	and its use is governed by the MiscKit license, found in the file
//	"License.rtf" in the MiscKit distribution.  Please refer to that file
//	for a list of all applicable permissions and restrictions.
//	

#import "MiscMergeTemplate.h"
#import <Foundation/NSCharacterSet.h>
#import <Foundation/NSArray.h>
#import <Foundation/NSUtilities.h>
#import <Foundation/NSAutoreleasePool.h>
#import <objc/objc-runtime.h>
#import <MiscFoundation.subproj/NSString+MiscAdditions.h>
#import "NSScanner+MiscMerge.h"
#import "MiscMergeCommand.h"
#import "MiscMergeCommandBlock.h"
#import "_MiscMergeFieldCommand.h"
#import "_MiscMergeCopyCommand.h"
#import "_MiscMergeDelayedParseCommand.h"

@implementation MiscMergeTemplate
/*"
 * This class contains the template that is used by a merge engine. It
 * performs two functions:  (1) parse a string or text file into the
 * commands required by a merge ending and (2) act as a container for the
 * commands once they have been parsed, providing them to a merge engine as
 * needed.
 * 
 * Typically, MiscMergeTemplate objects are used in a very simple way: they
 * are instantiated, given the ASCII text or string to parse, and then
 * passed to MiscMergeEngine instances as needed.  That's it!
 * 
 * It should be noted that template text which is simply copied from the
 * template into the merged output (i.e. any text outside of a merge
 * command) is actually turned into a special "copy" command by the parsing
 * algorithm. This allows the merge engine to deal exclusively with
 * MiscMergeCommand subclasses to perform a merge.  This implementation
 * detail should not affect anything that would normally be done with this
 * object, but it is important to understand this fact if attempting to
 * understand the data structure created by the parsing routines.
 * 
 * If a command string contains merge commands inside itself, then a
 * special "delayed command" class will used.  That class will, during a
 * merge, create an engine, perform a merge on its text, and then parse
 * itself into the correct type of command.  This allows merges to contain
 * commands that change depending upon the data records.
 * 
 * Commands created while parsing are always added to the "current" command
 * block.  By using -#pushCommandBlock: and -#popCommandBlock,
 * MiscMergeCommand subclasses can temporarily substitute their own command
 * block to be the "current" block during the parsing process.  This way,
 * container-type commands such as if clauses and loops can know which
 * commands they hold.
"*/

/*"
 * Returns the default character used to start a merge command, ''.  A
 * subclass of MiscMergeTemplate could override this method.
"*/
+ (char)defaultStartDelimiter
{
//	return '(';
	return '';
}

/*"
 * Returns the default character used to end a merge command, ''.  A
 * subclass of MiscMergeTemplate could override this method.
"*/
+ (char)defaultEndDelimiter
{
//	return ')';
	return '';
}

/*" Creates a new, autoreleased MiscMergeTemplate. "*/
+ template
{
	return [[[self alloc] init] autorelease];
}

/*"
 * Creates a new, autoreleased MiscMergeTemplate, and parses aString.
"*/
+ templateWithString:(NSString *)aString
{
	return [[[self alloc] initWithString:aString] autorelease];
}

/*"
 * Initializes the MiscMergeTemplate instance and returns self.  This is
 * the designated initializer.  The start and end delimiters are set to the
 * values returned by +#defaultStartDelimiter and +#defaultEndDelimiter.
"*/
- init
{
	[super init];
	topLevelCommands = [[MiscMergeCommandBlock alloc] init];
	commandStack = [[NSMutableArray arrayWithObject:topLevelCommands] retain];
	startDelimiter = [[self class] defaultStartDelimiter];
	endDelimiter = [[self class] defaultEndDelimiter];
	return self;
}

/*"
 * Initializes the MiscMergeTemplate, then parses string.
"*/
- initWithString:(NSString *)string
{
	[self init];
	[self parseString:string];
	return self;
}

/*"
 * Loads the contents of filename, then calls -#initWithString:.
"*/
- initWithContentsOfFile:(NSString *)filename
{
	NSString *fileString = [[[NSString alloc] initWithContentsOfFile:filename] autorelease];
	if (fileString == nil) NSLog(@"%@: Could not read template file %@", [self class], filename);
	return [self initWithString:fileString];
}

- (void)dealloc
{
	[commandStack release];
	[topLevelCommands release];
	[_parseStopChars release];
	[_startDelimiterString release];
	[_endDelimiterString release];
	[super dealloc];
}

/*" Returns the character used to start a merge command. "*/
- (char)startDelimiter
{
	return startDelimiter;
}

/*" Returns the character used to end a merge command. "*/
- (char)endDelimiter
{
	return endDelimiter;
}

/*"
 * Returns the start delimiter string as derived from the start delimiter
 * character.  The parsing methods always use this string to search for the
 * start delimiter, so this method could be overridden in a subclass. The
 * string must always be one character long, however.
"*/
- (NSString *)startDelimiterString
{
	if (_startDelimiterString == nil)
	{
		char start = [self startDelimiter];
		_startDelimiterString = [[NSString alloc] initWithCString:&start length:1];
	}
	
	return _startDelimiterString;
}

/*"
 * Returns the end delimiter string as derived from the end delimiter
 * character.  The parsing methods always use this string to search for the
 * end delimiter, so this method could be overridden in a subclass. The
 * string must always be one character long, however.
"*/
- (NSString *)endDelimiterString
{
	if (_endDelimiterString == nil)
	{
		char end = [self endDelimiter];
		_endDelimiterString = [[NSString alloc] initWithCString:&end length:1];
	}
	
	return _endDelimiterString;
}

/*"
 * Sets the characters used to start and end a merge command.
 * %{startDelim} and %{endDelim} must be different characters.
"*/
- (void)setStartDelimiter:(char)startDelim endDelimiter:(char)endDelim
{
	startDelimiter = startDelim;
	endDelimiter = endDelim;
	
	[_startDelimiterString release];
	[_endDelimiterString release];
	_startDelimiterString = nil;
	_endDelimiterString = nil;
}

/*"
 * Pushes aBlock on the command stack.  aBlock becomes the current command
 * block until popped off (or another block is placed on top of it).
"*/
- (void)pushCommandBlock:(MiscMergeCommandBlock *)aBlock
{
	[commandStack addObject:aBlock];
}

/*"
 * Pops the command block aBlock off of the command stack, so the previous
 * command block will again be the "current" command block.  If aBlock is
 * not at the top of the command stack, logs an error and does nothing.
 * Basically the same as -#popCommandBlock except it does the extra sanity
 * check.
"*/
- (void)popCommandBlock:(MiscMergeCommandBlock *)aBlock
{
	if (aBlock && [commandStack lastObject] != aBlock)
	{
		NSLog(@"Error, command stack mismatch");
		return;
	}
	
	[self popCommandBlock];
}

/*"
 * Pops the top command block off the command stack, so the previous
 * command block will again be the "current" command block.
"*/
- (void)popCommandBlock
{
	if ([commandStack count] <= 1)
	{
		NSLog(@"Error, cannot pop last command block");
		return;
	}

	[commandStack removeLastObject];
}

/*"
 * Returns the "current" command block, i.e. the command block at the top
 * of the command stack.  As they are parsed, commands are always added to
 * the command block returned by this method.
"*/
- (MiscMergeCommandBlock *)currentCommandBlock
{
	return [commandStack lastObject];
}

/*"
 * Returns the "top level" command block of the MiscMergeTemplate, which is
 * basically the series of commands to be executed to generate the merge
 * file.  The top level block is always at the bottom of the command stack.
"*/
- (MiscMergeCommandBlock *)topLevelCommandBlock
{
	return topLevelCommands;
}

/*"
 * Given the command string %{aCommand}, this method determines which
 * MiscMergeCommand subclass implements the merge command.  It returns the
 * class object needed to create instances of the MiscMergeCommand
 * subclass.
 * 
 * This method works by asking the runtime if it can find Objective-C
 * classes with specific names.  The name that is looked up is build from
 * the first word found in %{aCommand}.  The first word is turned to all
 * lower case, with the first letter upper case, and then sandwiched
 * between "Merge" and "Command".  For example, the merge command "if xxx
 * = y" has the word "if" as the first word.  Thus, the class
 * "MergeIfCommand" will be searched for. If the desired class cannot be
 * found, then it is assumed that the merge command is giving the name of a
 * field which should be inserted into the output document.
 * 
 * To avoid name space conflicts, all internal merge commands actually use
 * a slightly different name.  Thus, there really is no "MergeIfCommand" to
 * be found.  This method, when it doesn't find the "MergeIfCommand" class,
 * will search for another class, with a private name.  That class will be
 * found. (If it wasn't found, then the default "field" command class would
 * be returned.)  This allows a programmer to override any built in
 * command. To override the "if" command, simply create a "MergeIfCommand"
 * class and it will be found before the built in class.  If a programmer
 * wishes to make a particular command, such as "omit", inoperative, this
 * technique may be used to override with a MiscMergeCommand subclass that
 * does nothing.
"*/
- (Class)classForCommand:(NSString *)aCommand
{
	NSString *realCommand = [[aCommand firstWord] capitalizedString];
	NSString *className;
	Class theClass;
	
	className = [NSString stringWithFormat:@"MiscMerge%@Command", realCommand];
	theClass = objc_lookUpClass([className cString]);
	
	if (theClass == Nil)
	{
		className = [NSString stringWithFormat:@"_MiscMerge%@Command", realCommand];
		theClass = objc_lookUpClass([className cString]);
	}
	
	if (theClass == Nil)
	{
		theClass = [_MiscMergeFieldCommand class];
	}
	
	return theClass;
}

- (void)_addCommand:(MiscMergeCommand *)command
{
	[[self currentCommandBlock] addCommand:command];
}

- (void)_addBetweenString:(NSString *)betweenString
{
	id command = [[_MiscMergeCopyCommand alloc] init];
	[command parseFromRawString:betweenString];
	[self _addCommand:command];
	[command release];
}

- (void)_addCommandString:(NSString *)commandString
{
	Class commandClass = [self classForCommand:commandString];
	id command = [[commandClass alloc] init];
	[self _addCommand:command];
	[command parseFromString:commandString template:self];
	[command release];
}

/*"
 * Loads the contents of filename and calls -#parseString:.
"*/
- (void)parseContentsOfFile:(NSString *)filename
{
	NSString *string = [[NSString alloc] initWithContentsOfFile:filename];
	if (string == nil) NSLog(@"%@: Could not read template file %@", [self class], filename);
	[self parseString:string];
	[string release];
}

/*"
 * Parses the template in %{string}.
"*/
- (void)parseString:(NSString *)string
{
	NSAutoreleasePool *localPool = [[NSAutoreleasePool alloc] init];
	NSScanner *scanner = [NSScanner scannerWithString:string];
	NSMutableString *accumString = [[NSMutableString alloc] init];
	NSString *startDelimiterString = [self startDelimiterString];
	NSString *endDelimiterString = [self endDelimiterString];
	NSString *currString;
	int nestingLevel = 0;
	int maxNestingLevel = 0;
	BOOL inQuotes = NO;

	if (_parseStopChars == nil)
	{
		NSString *charString = [NSString stringWithFormat:@"\"\\%@%@",
			startDelimiterString, endDelimiterString];
		_parseStopChars = [NSCharacterSet characterSetWithCharactersInString:charString];
		[_parseStopChars retain];
	}
	
	[scanner setCharactersToBeSkipped:nil];
	
	// may want to flush localPool every 50 loops or so...
	while (![scanner isAtEnd])
	{
		if ([scanner scanUpToCharactersFromSet:_parseStopChars intoString:&currString])
			[accumString appendString:currString];
		
		if ([scanner scanLetterIntoString:&currString])
		{
			if ([currString isEqualToString:@"\\"])
			{
				NSString *letterString = nil;
				[scanner scanLetterIntoString:&letterString];
	
                /*
                 * Leave the backslash if we are not quoting a delimiter
                 * character -- we would wreak havoc on RTF template files
                 * otherwise.  Also leave the backslash if we are deeply
                 * nesting or inside quotes, as those sections will get
                 * reduced later.
                 */
				if (nestingLevel > 1 ||
				    (nestingLevel == 1 && inQuotes) ||
					(![letterString isEqualToString:startDelimiterString] &&
					 ![letterString isEqualToString:endDelimiterString]))
				{
					[accumString appendString:@"\\"];
				}

				[accumString appendString:letterString];
			}
			else if ([currString isEqualToString:@"\""])
			{
				[accumString appendString:currString];
				if (nestingLevel == 1) inQuotes = !inQuotes;
			}
			else if (nestingLevel > 0 && [currString isEqualToString:endDelimiterString])
			{
				if (nestingLevel > 1)
				{
					[accumString appendString:currString];
				}
				else
				{
					/* Special hack for the delayed parsing stuff. Hm. */
					if (maxNestingLevel > 1)
					{
						id command = [[_MiscMergeDelayedParseCommand alloc] init];
						[command parseFromString:accumString template:self];
						[self _addCommand:command];
						[command release];
					}
					else
					{
						if ([accumString length] > 0)
							[self _addCommandString:[[accumString copy] autorelease]];
					}
					[accumString setString:@""];
					inQuotes = NO;
					maxNestingLevel = 0;
				}
				
				nestingLevel--;
			}
			else if ([currString isEqualToString:startDelimiterString])
			{
				if (nestingLevel > 0)
				{
					[accumString appendString:currString];
				}
				else
				{
					if ([accumString length] > 0)
						[self _addBetweenString:[[accumString copy] autorelease]];
					[accumString setString:@""];
				}
				
				nestingLevel++;
				if (nestingLevel > maxNestingLevel) maxNestingLevel = nestingLevel;
			}
			else
			{
				// This an error; an end delimiter with no corresponding start.
				[accumString appendString:currString];
			}
		}
	}
	
	if ([accumString length] > 0)
	{
		[self _addBetweenString:[[accumString copy] autorelease]];
	}
	
	[localPool release];
}

@end
