/* ResourceLocator.m created by tthomas on Fri 10-Dec-1999 */

#import <Foundation/Foundation.h>

#import "ResourceLocator.h"


@implementation ResourceLocator

/*"
**	Generic locator of files. Give it some paths to search and use
**	#findResourceNamed: to find a file. The search paths are searched in
**	the order they appear in the searchPaths array, with the search in
**	the main bundle (if #searchInMainBundle is YES) happening last if
**	not found anywhere else.
"*/

- (id) init
{
    return [self initWithSearchPaths:nil];
}

- (id) initWithSearchPaths:(NSArray*)paths;
{
    // NOTE: This method doesn't check yet to see that all paths exist..
    [super init];
    _searchPaths = [paths mutableCopy];
    if (_searchPaths == nil) {
        _searchPaths = [[NSMutableArray alloc] init];
    }

    _searchInMainBundle = YES;

    return self;
}

- (void) dealloc;
{
    [_searchPaths release];
    [super dealloc];
}

- (BOOL) searchInMainBundle { return _searchInMainBundle; }
- (void) setSearchInMainBundle:(BOOL)nowSearchInMainBundle
{
    _searchInMainBundle = nowSearchInMainBundle;
}

- (NSArray*) searchPaths
{
    return _searchPaths;
}

- (void) addToSearchPaths:(NSString*)path
{
    if (path != nil) {
        NSFileManager* fileManager = [NSFileManager defaultManager];
        if ([fileManager fileExistsAtPath:path]) {
            [_searchPaths addObject:path];
        }
#if DEBUG
        else {
            NSLog (@"ResourceLocator - Did not add search path %@ because it doesn't exist.", path);
        }
#endif
    }
}

- (void) addArrayOfPathsToSearchPaths:(NSArray*)paths
{
    // NOTE: This method doesn't check yet to see that all paths exist..
    if (paths != nil) {
        [_searchPaths addObjectsFromArray:paths];
    }
}

- (void) addToSearchPathsUsingUserDefault:(NSString*)defaultName
{
    // NOTE: This method doesn't check yet to see that all paths exist..
    NSUserDefaults* standardUDefaults = [NSUserDefaults standardUserDefaults];
    NSArray* defaultsPaths = [standardUDefaults arrayForKey:defaultName];

    if (defaultsPaths != nil) {
        [self addArrayOfPathsToSearchPaths:defaultsPaths];
    }
}

- (void) removeFromSearchPaths:(NSString*)path
{
    [_searchPaths removeObject:path];
}

- (NSString*) findResourceNamed:(NSString*)resourceName
/*"
**	Tries to find the resource named resourceName and, if found, returns the
**	full path to the resource so the caller can load it.
"*/
{
    NSString* resourcePath = nil;
    BOOL error = NO;

    // Check arguments.
    if (!error) {
        error = (resourceName == nil);
#if DEBUG
        NSLog (@"Tried to find a nil resource.");
#endif
    }

    // Try actually finding the resource.
    if (!error) {
        NSEnumerator* searchPathsEnum = [[self searchPaths] objectEnumerator];
        NSString* searchPath = nil;
        NSFileManager* fileManager = [NSFileManager defaultManager];

        while ((searchPath = [searchPathsEnum nextObject]) != nil) {
            NSString* possibleResourcePath = [searchPath stringByAppendingPathComponent:resourceName];

            if ([fileManager fileExistsAtPath:possibleResourcePath]) {
                resourcePath = possibleResourcePath;
                break;
            }
        }

        // Try the main bundle if it wasn't found via the search paths.
        if (resourcePath == nil && [self searchInMainBundle]) {
            resourcePath = [[NSBundle mainBundle] pathForResource:resourceName ofType:nil];
        }
    }

#if DEBUG
    if (resourcePath != nil) {
        NSLog (@"Resource %@ was found at %@", resourceName, resourcePath);
    }
    else {
        NSLog (@"Resource %@ was not found in the following paths:%@", resourceName, [self searchPaths]);
    }
#endif          

    return resourcePath;
}
    
@end
