/*	MiscLayoutTree.h

	Copyright 1996 Uwe Hoffmann.

	This notice may not be removed from this source code.
	The use and distribution of this software is governed by the
	terms of the MiscKit license agreement.  Refer to the license
	document included with the MiscKit distribution for the terms.

	Version 2 (August 1996)
*/

#import <Foundation/NSObject.h>
#import <Foundation/NSGeometry.h>

/*" MiscPolyline is a linked list representing a polygon line."*/
typedef struct _MiscPolyline {
     float dx,dy;
     struct _MiscPolyline *link;
} MiscPolyline;

/*" MiscPolygon is made out of an upper (left) and a lower (right) polygon line."*/
typedef struct _MiscPolygon {
    struct {
	MiscPolyline *head,*tail;
     } lower,upper;
} MiscPolygon;

/*" This types constants define the layout of the tree: horizontal or vertical."*/
typedef enum _MiscLayoutTreeType {
	MiscHorizontalTreeType,
	MiscVerticalTreeType
} MiscLayoutTreeType;

/*" This types constants define the orientation of the tree (i.e. the position of the root node)."*/
typedef enum _MiscLayoutTreeOrientation {
	MiscStandardOrientation,
	MiscInverseOrientation
} MiscLayoutTreeOrientation;

@interface MiscLayoutTree:NSObject <NSCoding, NSCopying>
{
@private
    	MiscLayoutTree *parent;						/*"Parent node"*/
	MiscLayoutTree *child;						/*"Child node"*/
	MiscLayoutTree *sibling;					/*"Sibling node"*/
   	NSPoint pos;							/*"Position of the node"*/
	NSPoint offset;							/*"Position offset from predecessor node"*/
	NSSize size;							/*"Size of node"*/
	float distanceToParent;						/*"Horizontal distance to parent"*/ 
	float border;							/*"Defines a buffer rectangle around node"*/
    	MiscPolygon contour;						/*"Contour of subtree rooted in node"*/
	BOOL updateLayout;						/*"YES, if contour needs recalculating"*/
	BOOL collapsed;							/*"YES, if subtree rooted in node is collapsed"*/
	MiscLayoutTreeType layoutType;					/*"Horizontal or vertical"*/
	MiscLayoutTreeOrientation orientation;				/*"Position of root node"*/
	int tag;							/*"node tag, can be used by client"*/
	int DGNodeTag;							/*"DG!2"*/
	int DGVertexTag[4];						/*"DG!2"*/
}

/*"Allocating and initializing an MiscLayoutTree instance"*/
+ tree;
+ treeWithSize:(NSSize)aSize;
- initWithSize:(NSSize)aSize;
- copyNodeWithZone:(NSZone *)zone;

/*"Laying out"*/
- (void)layout;
- (BOOL)needsUpdateLayout;
- (void)invalidateLayout;
- (void)layoutNode;

/*"Accessing and manipulating geometric structure"*/
- (void)setPos:(NSPoint)aPos;
- (void)setSize:(NSSize)aSize;
- (void)setDistanceToParent:(float)aDistance;
- (void)setBorder:(float)aBorder;
- (NSPoint)pos;
- (NSPoint)offset;
- (NSSize)size;
- (float)distanceToParent;
- (float)border;
- (float)width;
- (float)height;
- (NSRect)nodeBounds;
- (NSRect)branchBounds;
- (MiscLayoutTreeType)layoutType;
- (void)setLayoutType:(MiscLayoutTreeType)aType;
- (MiscLayoutTreeOrientation)orientation;
- (void)setOrientation:(MiscLayoutTreeOrientation)anOrientation;

/*"Accessing and manipulating tree structure"*/
- (MiscLayoutTree *)sibling;
- (MiscLayoutTree *)child;
- (MiscLayoutTree *)parent;
- (NSEnumerator *)depthEnumerator;
- (NSEnumerator *)breadthEnumerator;
- (MiscLayoutTree *)branchAt:(unsigned)index;
- (unsigned)numberOfChildren;
- (unsigned)indexOfChild:(MiscLayoutTree *)aChild;
- (void)addBranch:(MiscLayoutTree *)branch;
- (void)insertBranch:(MiscLayoutTree *)branch at:(unsigned)index;
- (void)removeBranch:(MiscLayoutTree *)branch;
- (void)removeBranchAt:(unsigned)index;
- (void)moveBranchAt:(unsigned)index to:(unsigned)otherIndex;
- (BOOL)isCollapsed;
- (void)collapse;
- (void)expand;
- (void)toggleCollapse;

/*"Accessing and manipulating node contents"*/
- (void)setTag:(int)aTag;
- (int)tag;

/*"Interactive resizing in a view"*/ 
- (void)setTempNodeBounds:(NSRect)newBounds;

@end
