// MORegexFormatter.m
// MOKit - Yellow Box
//
// Copyright 1996-1999, Mike Ferris.
// All rights reserved.

// ABOUT MOKit
//
// MOKit is a collection of useful and general objects.  Permission is
// granted by the author to use MOKit in your own programs in any way
// you see fit.  All other rights to the kit are reserved by the author
// including the right to sell these objects as part of a LIBRARY or as
// SOURCE CODE.  In plain English, I wish to retain rights to these
// objects as objects, but allow the use of the objects as pieces in a
// fully functional program.  NO WARRANTY is expressed or implied.  The author
// will under no circumstances be held responsible for ANY consequences to
// you from the use of these objects.  Since you don't have to pay for
// them, and full source is provided, I think this is perfectly fair.

#import <MOKit/MOKit.h>

typedef enum {
    MOInitialVersion = 1,
} MOClassVersion;

static const MOClassVersion MOCurrentClassVersion = MOInitialVersion;

@implementation MORegexFormatter

+ (void)initialize {
    // Set the version.  Load classes, and init class variables.
    if (self == [MORegexFormatter class])  {
        [self setVersion:MOCurrentClassVersion];
    }
}

- (id)initWithRegularExpressions:(NSArray *)expressions {
    self = [super init];
    if (self) {
        if (expressions) {
            _expressions = [[NSMutableArray allocWithZone:[self zone]] initWithArray:expressions];
        } else {
            _expressions = nil;
        }
        _lastMatchedExpressionIndex = NSNotFound;
        _rfFlags.allowsEmptyString = YES;
        [self setFormatPattern:@"%0"];
    }

    return self;
}

- (id)initWithRegularExpression:(MORegularExpression *)expression {
    return [self initWithRegularExpressions:(expression ? [NSArray arrayWithObject:expression] : nil)];
}

- (id)init {
    return [self initWithRegularExpressions:nil];
}

- (void)dealloc {
    [_expressions release];
    [super dealloc];
}

- (id)copyWithZone:(NSZone *)zone {
    MORegexFormatter *copyObj = [super copyWithZone:zone];
    if (copyObj) {
        copyObj->_expressions = [_expressions copyWithZone:zone];
        copyObj->_lastMatchedExpressionIndex = NSNotFound;
        copyObj->_rfFlags.allowsEmptyString = _rfFlags.allowsEmptyString;
        copyObj->_formatPattern = [_formatPattern copyWithZone:zone];
    }

    return copyObj;
}

- (void)encodeWithCoder:(NSCoder *)coder {
    BOOL tempBool;
    
    [super encodeWithCoder:coder];

    [coder encodeObject:_expressions];
    tempBool = _rfFlags.allowsEmptyString;
    [coder encodeValueOfObjCType:"C" at:&tempBool];
    [coder encodeObject:_formatPattern];
}

- (id)initWithCoder:(NSCoder *)coder {
    unsigned classVersion = [coder versionForClassName:@"MORegexFormatter"];

    if (classVersion > MOCurrentClassVersion)  {
        NSLog(@"%@: class version %u cannot read instances archived with version %u",
              MOFullMethodName(self, _cmd), MOCurrentClassVersion, classVersion);
        [self dealloc];
        return nil;
    }

    self = [super initWithCoder:coder];
    if (classVersion == MOInitialVersion) {
        BOOL tempBool;
        _expressions = [[coder decodeObject] retain];
        [coder decodeValueOfObjCType:"C" at:&tempBool];
        _rfFlags.allowsEmptyString = tempBool;
        [self setFormatPattern:[coder decodeObject]];

        _lastMatchedExpressionIndex = NSNotFound;
    }

    return self;
}

- (NSArray *)regularExpressions {
    return _expressions;
}

- (void)insertRegularExpression:(MORegularExpression *)expression atIndex:(unsigned)index {
    if (!_expressions) {
        _expressions = [[NSMutableArray allocWithZone:[self zone]] initWithCapacity:1];
    }
    [_expressions insertObject:expression atIndex:index];
}

- (void)addRegularExpression:(MORegularExpression *)expression {
    [self insertRegularExpression:expression atIndex:(_expressions ? [_expressions count] : 0)];
}

- (void)removeRegularExpressionAtIndex:(unsigned)index {
    if (_expressions) {
        [_expressions removeObjectAtIndex:index];
    }
}

- (void)replaceRegularExpressionAtIndex:(unsigned)index withRegularExpression:(MORegularExpression *)expression {
    if (!_expressions) {
        [NSException raise:NSRangeException format:@"*** %@: array index beyond end of array.", MOFullMethodName(self, _cmd)];
    }
    [_expressions replaceObjectAtIndex:index withObject:expression];
}

- (BOOL)allowsEmptyString {
    return _rfFlags.allowsEmptyString;
}

- (void)setAllowsEmptyString:(BOOL)flag {
    _rfFlags.allowsEmptyString = flag;
}

- (NSString *)formatPattern {
    return _formatPattern;
}

- (void)setFormatPattern:(NSString *)pattern {
    NSParameterAssert(pattern);
    [_formatPattern autorelease];
    _formatPattern = [pattern copyWithZone:[self zone]];
}

- (NSString *)stringForObjectValue:(id)obj {
    return [obj description];
}

- (BOOL)validateString:(NSString *)string matchedExpressionIndex:(unsigned *)matchedIndex {
    if (_expressions) {
        unsigned i, c = [_expressions count];
        for (i=0; i<c; i++) {
            if ([[_expressions objectAtIndex:i] matchesString:string]) {
                if (matchedIndex) {
                    *matchedIndex = i;
                }
                return YES;
            }
        }
        return NO;
    } else {
        return NO;
    }
}

- (id)objectForValidatedString:(NSString *)string matchedExpressionIndex:(unsigned)matchedIndex {
    if ([self allowsEmptyString] && [string isEqualToString:@""]) {
        return [[string copy] autorelease];
    } else {
        NSRange searchRange = NSMakeRange(0, [_formatPattern length]);
        NSRange percentRange;

        percentRange = [_formatPattern rangeOfString:@"%" options:0 range:searchRange];

        if (percentRange.length > 0) {
            NSMutableString *buffer = [_formatPattern mutableCopy];
            NSArray *subexpressions = nil;
            unsigned subexpressionsCount = 0;
            int locationOffset = 0;
            
            while (percentRange.length == 1) {
                if ((percentRange.location > 0) && ([_formatPattern characterAtIndex:percentRange.location - 1] == (unichar)'\\')) {
                    // Ignore it.
                } else {
                    // Might be a subexpression format sequence
                    if (percentRange.location + 1 < NSMaxRange(searchRange)) {
                        unichar digit = [_formatPattern characterAtIndex:percentRange.location + 1];
                        if ((digit >= (unichar)'0') && (digit <= (unichar)'9')) {
                            // It is a subexpression.
                            unsigned subexpIndex = digit - (unichar)'0';
                            if (subexpIndex == 0) {
                                [buffer replaceCharactersInRange:NSMakeRange(percentRange.location + locationOffset, 2) withString:string];
                                locationOffset += [string length] - 2;
                                percentRange.length = 2;
                            } else {
                                // Subexpressions are one-based (0 is whole string), and it just so happens that subexpIndex is now one-based.
                                if (!subexpressions) {
                                    subexpressions = [[[self regularExpressions] objectAtIndex:matchedIndex] subexpressionsForString:string];
                                    subexpressionsCount = [subexpressions count];
                                }
                                if (subexpIndex < subexpressionsCount) {
                                    NSString *subexp = [subexpressions objectAtIndex:subexpIndex];
                                    [buffer replaceCharactersInRange:NSMakeRange(percentRange.location + locationOffset, 2) withString:subexp];
                                    locationOffset += [subexp length] - 2;
                                    percentRange.length = 2;
                                }
                            }
                        }
                    }
                }

                searchRange = NSMakeRange(NSMaxRange(percentRange), NSMaxRange(searchRange) - NSMaxRange(percentRange));
                percentRange = [_formatPattern rangeOfString:@"%" options:0 range:searchRange];
            }

            return [buffer autorelease];
        } else {
            return [[_formatPattern copy] autorelease];
        }
    }
    return [[string copy] autorelease];
}

- (BOOL)getObjectValue:(id *)obj forString:(NSString *)string errorDescription:(NSString **)error {
    unsigned matchedIndex;
    BOOL retval;
    
    // Reset the last match ivar
    _lastMatchedExpressionIndex = NSNotFound;

    if (error) {
        *error = nil;
    }

    if ([self allowsEmptyString] && (!string || [string isEqualToString:@""])) {
        retval = YES;
    } else if ([self validateString:string matchedExpressionIndex:&matchedIndex]) {
        _lastMatchedExpressionIndex = matchedIndex;
        retval = YES;
    } else {
        if (error) {
            if ([_expressions count] > 0) {
                *error = [NSString localizedStringWithFormat:NSLocalizedStringFromTableInBundle(@"Proposed value '%@' failed to format because it did not match any of the formatter's regular expressions.", @"MOKit", [NSBundle bundleForClass:[self class]], @"Error string."), string];
            } else {
                *error = [NSString localizedStringWithFormat:NSLocalizedStringFromTableInBundle(@"Proposed value '%@' failed to format because formatter has no expressions to match.", @"MOKit", [NSBundle bundleForClass:[self class]], @"Error string."), string];
            }
        }
        retval = NO;
    }

    // Format the string
    if (retval && obj) {
        NSString *formattedObj = [self objectForValidatedString:string matchedExpressionIndex:_lastMatchedExpressionIndex];
        *obj = formattedObj;
    }
    
    return retval;
}

- (MORegularExpression *)lastMatchedExpression {
    if (_lastMatchedExpressionIndex != NSNotFound) {
        return [_expressions objectAtIndex:_lastMatchedExpressionIndex];
    } else {
        return nil;
    }
}

@end
