#import "TWCell.h"
#import "TWCellDraw.h"
#import "ThumbWheel.h"
#ifdef WIN32
// I don't know where this thing is under windows
#define M_PI	3.14159265
#endif

/******************************************************************************
    TWCell - by Jeff Martin
    
ThumbWheel offers the functionality of Slider plus the features that you would expect from a real thumbwheel (including 2 3/4 D Graphics!).

ThumbWheel has a linear display mode and a radial display mode and offers the ability to assign a value to the visible region of the control as well as an absolute value that the ThumbWheel will either ignore, bound to or wrap around.

ThumbWheel can also return relative values via its -relativeIntValue & -relativeFloatValue methods. A snap back option allows mouse loops to start from and return to a base value.

TWCell (ThumbWheelCell) is the actual guts behind ThumbWheel (using the Control/Cell paradigm).

Written by: Jeff Martin (jmartin@reportmill.com)
You may freely copy, distribute and reuse the code in this example.  
Don't even talk to me about warranties.
******************************************************************************/
// A mod function for floating values
#define MOD(x,y) ((x) - (y)*(int)((float)(x)/(y)))
// Keep 'a' between x and y
#define CLAMP(a,x,y) (MAX((x), MIN((y), (a))))
// Keep 'a' between 'x' and 'y' by wrapping it to the other side
#define CLAMP_WITH_WRAP(a,x,y) \
( ((a) < (x)) ? ((y) - MOD(((x)-(a)),((y)-(x)))) : ( ((a) > (y)) ? ((x) + MOD(((a)-(y)),((y)-(x)))) : (a) ) )
// Is a between x and y
#define ISBETWEEN(a,x,y) (((a)>=(x))&&((a)<=(y)))
#define EQUAL(a,b) (ABS((a)-(b))<0.00001)

@implementation TWCell : NSActionCell

+ (void)initialize { [self setVersion:1]; }

// Set reasonable defaults
- init
{
    self = [super init];
    [self setType:NSTextCellType];
    [self setHorizontal]; [self setLinear]; [self setUnbounded];
    [self setVisibleMax:1.0]; [self setVisibleMin:-1.0]; [self setAbsoluteMax:2.0]; [self setAbsoluteMin:-2.0];
    [self setSnapsBack:NO]; [self setSnapBackValue:0.0]; [self setDashInterval:10]; [self setShowMainDash:YES];
    [self setColor:[NSColor lightGrayColor]];
    [self sendActionOn:NSLeftMouseDownMask|NSLeftMouseDraggedMask|NSLeftMouseUpMask];
    return self;
}

// Override trackMouse to provide an infinitely large area(continuous tracking)
- (BOOL)trackMouse:(NSEvent *)theEvent inRect:(NSRect)cellFrame ofView:(NSView *)controlView untilMouseUp:(BOOL)flag
{ _cellFrame = cellFrame; return [super trackMouse:theEvent inRect:cellFrame ofView:controlView untilMouseUp:YES]; }

// Override to return YES (always track mouse)
- (BOOL)startTrackingAt:(NSPoint)startPoint inView:(NSView *)controlView { return YES;}

// Override from cell to track mouse
- (BOOL)continueTracking:(NSPoint)lastPoint at:(NSPoint)currentPoint inView:(NSView *)controlView
{
    float currPointVal =[self floatValueAtPoint:currentPoint forFrame:_cellFrame];
    float lastPointVal =[self floatValueAtPoint:lastPoint forFrame:_cellFrame];
    float mouseScale = ([[NSApp currentEvent] modifierFlags] & NSAlternateKeyMask)? .1 : 1;
    
    // If TW is bounded and at absolute bound, only track when mouse comes back
    if([self isBounded]) {

        // Return if we are already at absoluteMax and currentPoint is greater return YES
        if((EQUAL([self floatValue], [self absoluteMax])) && (currPointVal > [self absoluteMax])) return YES;

        // Return if we are already at absoluteMin and currPoint is less return YES
        if((EQUAL([self floatValue], [self absoluteMin])) && (currPointVal < [self absoluteMin])) return YES;
    }
    
    // Set the float value relative to last point
    [self setFloatValue:[self floatValue] + (currPointVal - lastPointVal)*mouseScale];
    
    // Redraw control
    [controlView setNeedsDisplay:YES];
    return YES;    // Always continue to track mouse
}


/******************************************************************************
    - (float)floatValueAtPoint:(NXPoint)point forFrame:(NXRect)frame
    
    This method gives the value that corresponds to a point with respect to the given frame and the visible range. When in radial mode, the point on the thumbwheel is approximated with a power series for arcCos to get legal values for points outside of the frame.
******************************************************************************/
- (float)floatValueAtPoint:(NSPoint)point forFrame:(NSRect)frame
{
    float pos = [self isVertical] ? point.y : point.x;
    float base = [self isVertical] ? NSMinY(frame) : NSMinX(frame);
    float width = [self isVertical] ? NSHeight(frame) : NSWidth(frame);
    float value;
    
    if([self isLinear])
        value = [self visibleMin] + (pos - base)/width*[self visibleRange];
    else {
        float radius = width/2, midP = base + radius, x = (midP - pos)/radius;
        // Get degrees by pwr series approximation of ArcCos (Pi/2 - x - x^3/6)
        float alpha = (M_PI/2 - x - x*x*x/6);
        // Convert degrees to TW coords
        value = [self visibleMin] + alpha/M_PI*[self visibleRange];
    }
    return value;
}
    
// Override from cell to snapback if neccessary
- (void)stopTracking:(NSPoint)lastPoint at:(NSPoint)stopPoint inView:(NSView *)controlView mouseIsUp:(BOOL)flag
{
    [super stopTracking:lastPoint at:stopPoint inView:controlView mouseIsUp:flag];
    
    // If the ThumbWheel is in snap back mode, snap it back and redraw
    if([self snapsBack]) {
        [self setFloatValue:[self snapBackValue]]; [self resetRelativeValue]; [controlView setNeedsDisplay:YES];
    }
}

// Direction is either DIRECTION_HORIZONTAL or DIRECTION_VERTICAL
- (int)direction { return _direction; }
- (void)setDirection:(int)dir { if(_direction!=dir) { _direction = dir; [_image release]; _image = NULL; } }
- (BOOL)isVertical { return _direction == DIRECTION_VERTICAL; }
- (void)setVertical { [self setDirection:DIRECTION_VERTICAL]; }
- (BOOL)isHorizontal { return _direction == DIRECTION_HORIZONTAL; }
- (void)setHorizontal { [self setDirection:DIRECTION_HORIZONTAL]; }

/******************************************************************************
    - (int)displayMode, setDisplayMode:(int)mode
    
    The displayMode is either DISPLAY_MODE_LINEAR or DISPLAY_MODE_RADIAL. Linear displays a flat ruler type control whereas radial displays a 3D thumbwheel that actually looks curved .
******************************************************************************/
- (int)displayMode { return _displayMode; }
- (void)setDisplayMode:(int)mode { _displayMode = mode; }
- (BOOL)isRadial { return _displayMode == DISPLAY_MODE_RADIAL; }
- (void)setRadial { [self setDisplayMode:DISPLAY_MODE_RADIAL]; }
- (BOOL)isLinear { return _displayMode == DISPLAY_MODE_LINEAR; }
- (void)setLinear { [self setDisplayMode:DISPLAY_MODE_LINEAR]; }

/******************************************************************************
    - setIntValue:(int)value, - setFloatValue:(float)value
    
    These methods are overridden to allow us to calculate relative values and to constrain the value with respect to the absolute mode and absolute values.
******************************************************************************/
- (void)setIntValue:(int)val { [self setFloatValue:val]; }
- (void)setFloatValue:(float)val
{ 
    // Clamp or Wrap newValue wrt the absoluteMode
    if(!ISBETWEEN(val, [self absoluteMin], [self absoluteMax])) {
        if([self isBounded]) val = CLAMP(val, [self absoluteMin], [self absoluteMax]);
        else if([self isWrapped]) val = CLAMP_WITH_WRAP(val, [self absoluteMin], [self absoluteMax]);
    }

    // Store last float value and set float value
    _lastFloatValue = [self floatValue]; [super setFloatValue:val];
}

/******************************************************************************
    - (float)visibleMax, - (float)visibleMin
    - (float)visibleRange, - (float)middleValue
    
    VisibleMax and visibleMin are the values of the thumbwheel at either end; max value is at right/top, min is at left/bottom. middleValue is the value of the TW at the center (wrt visibleMin and visibleMax).
******************************************************************************/
- (float)visibleMax { return _visMax; }
- (void)setVisibleMax:(float)max { _visMax = max; }
- (float)visibleMin { return _visMin; }
- (void)setVisibleMin:(float)min { _visMin = min; }
- (float)visibleRange { return [self visibleMax] - [self visibleMin]; }
- (float)middleValue { return [self visibleMin] + [self visibleRange]/2; }

/******************************************************************************
    - (int)absoluteMode, - (float)absoluteMax, - (float)absoluteMin
    
    The absolute mode refers to ThumbWheel values that exceed the visible range. ABSOLUTE_UNBOUNDED means that the TW can be dragged as high or low as desired. ABSOLUTE_BOUNDED means that the TW will be clamped to some arbitrarily large value. ABSOLUTE_WRAPPED means that the TW will wrap from the absoluteMax to the absoluteMin (and vise-versa) when applicable. AbsoluteMax is the value off to the right and up. AbsoluteMin us the value off to the left and down.
******************************************************************************/
- (int)absoluteMode { return _absMode; }
- (void)setAbsoluteMode:(int)mode { _absMode = mode; }
- (BOOL)isUnbounded { return _absMode == ABSOLUTE_UNBOUNDED;  }
- (void)setUnbounded { [self setAbsoluteMode:ABSOLUTE_UNBOUNDED]; }
- (BOOL)isBounded { return _absMode == ABSOLUTE_BOUNDED; }
- (void)setBounded { [self setAbsoluteMode:ABSOLUTE_BOUNDED]; }
- (BOOL)isWrapped { return _absMode == ABSOLUTE_WRAPPED; }
- (void)setWrapped { [self setAbsoluteMode:ABSOLUTE_WRAPPED]; }
- (float)absoluteMax { return _absMax; }
- (void)setAbsoluteMax:(float)value { _absMax = value; }
- (float)absoluteMin { return _absMin; }
- (void)setAbsoluteMin:(float)value { _absMin = value; }
- (float)absoluteRange { return [self absoluteMax] - [self absoluteMin]; }

/******************************************************************************
    - (int)relativeIntValue, - (float)relativeFloatValue, - resetRelativeValue
    
    These two methods return the change of the value since the last iteration. This is useful for a relative method call ( rotateBy: as opposed to rotateTo:). resetRelativeValue sets the relative change to zero (typically only called internally when snapping back).
******************************************************************************/
- (int)relativeIntValue { return [self intValue] - (int)[self lastFloatValue];}
- (float)relativeFloatValue { return [self floatValue] - [self lastFloatValue]; }
- (void)resetRelativeValue { [self setLastFloatValue:[self floatValue]];}

/******************************************************************************
    - (float)lastFloatValue, - setLastFloatValue:(float)value
    
    These two methods query and set the lastFloatValue. You should never need to call the methods - they are used internally to calculate the relative int and float value. The lastFloatValue is simply the value that the ThumbWheel was previously set to.
******************************************************************************/
- (float)lastFloatValue { return _lastFloatValue; }
- (void)setLastFloatValue:(float)value { _lastFloatValue = value; }

/******************************************************************************
    - (BOOL)snapsBack, - (float)snapBackValue
    
    It is sometimes useful to have the thumbwheel snap back to some value (zero by default) so that it can be used for relative modification of values (rotate by as opposed to rotateTo:).
******************************************************************************/
- (BOOL)snapsBack { return _snapsBack; }
- (void)setSnapsBack:(BOOL)flag { _snapsBack = flag; }
- (float)snapBackValue { return _snapBackValue; }
- (void)setSnapBackValue:(float)value { _snapBackValue = value; }

/******************************************************************************
    - (float)dashInterval, setDashInterval:(float)value

     The dash interval is either in degrees (DISPLAY_MODE_RADIAL) or PostScript 
points (DISPLAY_MODE_LINEAR). The default is 10 of each.
******************************************************************************/
- (float)dashInterval { return _dashInterval; }
- (void)setDashInterval:(float)val { _dashInterval = val; }

/******************************************************************************
    - (BOOL)showMainDash, - setShowMainDash:(BOOL)flag

     The main dash is the dash in the center of the control and gives feedback as to the absolute value of the control. This should be set to NO for TW that only provide relative values.
******************************************************************************/
- (BOOL)showMainDash { return _showMainDash;}
- (void)setShowMainDash:(BOOL)flag { _showMainDash = flag; }

/******************************************************************************
    - (NXColor)color, - setColor:(NXColor)color

     These methods manipulate the predominant color of the ThumbWheel. Radial ThumbWheel are of course shades of these. The default is NX_COLORLTGRAY. 
******************************************************************************/
- (NSColor *)color {return _color;}
- (void)setColor:(NSColor *)newColor
{ if(![_color isEqual:newColor]) { [_color release]; _color = [newColor retain]; [_image release]; _image = NULL; } }

/******************************************************************************
    - (int)shift
    
     The shift is how much the dashes are shifted by to achieve the animation of motion it is in points. It is calculated from the visibleRange and the physicalRange (frame).
******************************************************************************/
- (int)shift:(NSRect)frame
{    
    if([self isLinear]) {
        if([self isHorizontal]) return ([self floatValue] - [self visibleMin])/[self visibleRange]*NSWidth(frame) + .5;
        else return ([self floatValue] - [self visibleMin])/[self visibleRange]*NSHeight(frame) +.5;
    }
    else return ([self floatValue] - [self visibleMin])/[self visibleRange]*180 + .5;
}

// Override highlight:withFrame:inView: so that it does nothing
- (void)highlight:(BOOL)flag withFrame:(NSRect)cellFrame inView:(NSView *)controlView { }

// Override this so that we track mouse whether or not it is on top of us.
+ (BOOL)prefersTrackingUntilMouseUp { return YES; }

/******************************************************************************
     The Read and Write Methods are for archival.
******************************************************************************/
- (void)encodeWithCoder:(NSCoder *)archiver
{
    [super encodeWithCoder:archiver];
    [archiver encodeValueOfObjCType:"i" at:&_displayMode]; [archiver encodeValueOfObjCType:"i" at:&_direction];
    [archiver encodeValueOfObjCType:"f" at:&_floatValue]; [archiver encodeValueOfObjCType:"f" at:&_lastFloatValue];
    [archiver encodeValueOfObjCType:"f" at:&_visMax]; [archiver encodeValueOfObjCType:"f" at:&_visMin];
    [archiver encodeValueOfObjCType:"i" at:&_absMode];
    [archiver encodeValueOfObjCType:"f" at:&_absMax]; [archiver encodeValueOfObjCType:"f" at:&_absMin];
    [archiver encodeValueOfObjCType:"c" at:&_snapsBack]; [archiver encodeValueOfObjCType:"f" at:&_snapBackValue];
    [archiver encodeValueOfObjCType:"i" at:&_dashInterval]; [archiver encodeValueOfObjCType:"c" at:&_showMainDash];
    [archiver encodeObject:_color];
}

- initWithCoder:(NSCoder *)archiver
{
    self = [super initWithCoder:archiver];
    [archiver decodeValueOfObjCType:"i" at:&_displayMode]; [archiver decodeValueOfObjCType:"i" at:&_direction];
    [archiver decodeValueOfObjCType:"f" at:&_floatValue]; [archiver decodeValueOfObjCType:"f" at:&_lastFloatValue];
    [archiver decodeValueOfObjCType:"f" at:&_visMax]; [archiver decodeValueOfObjCType:"f" at:&_visMin];
    [archiver decodeValueOfObjCType:"i" at:&_absMode];
    [archiver decodeValueOfObjCType:"f" at:&_absMax]; [archiver decodeValueOfObjCType:"f" at:&_absMin];
    [archiver decodeValueOfObjCType:"c" at:&_snapsBack]; [archiver decodeValueOfObjCType:"f" at:&_snapBackValue];
    [archiver decodeValueOfObjCType:"i" at:&_dashInterval]; [archiver decodeValueOfObjCType:"c" at:&_showMainDash];
    if([archiver versionForClassName:@"TWCell"] == 0) _color = [[archiver decodeNXColor] retain];
    else _color = [[archiver decodeObject] retain];
    _image = nil; return self;
}

- (void)dealloc { [_image release]; [super dealloc]; }

@end
