/**
 * Zephyr Message Browser
 **
 * Wilfredo Sanchez | wsanchez@apple.com
 * Copyright 1998 Apple Computer, Inc.
 * All rights reserved.
 **/

#include "NSZephyr.h"

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>

#import "ZephyrApplication.h"
#import "ZephyrMessage.h"

#import "MessageBrowser.h"

/**
 * myMessageDict is of the form:
 *
 * {
 *   "class" = {
 *      "instance = (
 *        <ZephyrMessage>,
 *        ...
 *      );
 *      ...
 *   };
 *   ...
 * }
 **/

/**
 * We're multithreaded, so all display events need to aquire and release the event lock.
 * dispatchMessage: is the entry point for the message thread. Other such events are the IB
 * targets. Finer grain locks are probably not useful; you want an event to complete before
 * processing the next one: the only possible contention for the event lock is between
 * a user event and a message event, and I don't think we want both diddling the UI at the
 * same time.
 **/

#if DEBUG_LOCKING
#define   LockUI [myEventLock   lock]; NSLog(@"Lock retained by %s", sel_getName(_cmd))
#define UnLockUI [myEventLock unlock]; NSLog(@"Lock released by %s", sel_getName(_cmd))
#else
#define   LockUI [myEventLock   lock]
#define UnLockUI [myEventLock unlock]
#endif

enum _columns
{
    CLASS_COLUMN = 0,
    INSTANCE_COLUMN,
    RECIPIENT_COLUMN
};

#define FIRST_COLUMN CLASS_COLUMN
#define LAST_COLUMN  INSTANCE_COLUMN

@interface MessageBrowser (NSBrowserDelegate)

- (int)        browser: (NSBrowser*) aBrowser
  numberOfRowsInColumn: (int       ) aColumn;

- (void)  browser: (NSBrowser*    ) aBrowser
  willDisplayCell: (NSBrowserCell*) aCell
            atRow: (int           ) aRow
           column: (int           ) aColumn;

- (BOOL) browser: (NSBrowser*) aBrowser
       selectRow: (int       ) aRow
        inColumn: (int       ) aColumn;

@end

@interface MessageBrowser (Private)

- (NSMutableArray*) aggregateMessagesInDictionary: (NSDictionary*) aMessageDict
                                       forClasses: (NSArray*     ) aClasses
                                         inColumn: (int          ) aColumn;

- (void) updateMessageList;

- (int) showMessagesInList: (NSArray*) aMessageList
             startingAtRow: (int     ) aRow;

- (BOOL) selectClass: (NSString*) aClass
            inColumn: (int      ) aColumn;

@end

@implementation MessageBrowser

/********************
 * Instance Methods *
 ********************/

////
// Inits
////

- (id) init
{
    if ((self = [super init]))
      {
        // Load interface
        if (! [NSBundle loadNibNamed: @"MessageBrowser"
                               owner: self])
          {
            NSLog(@"Failed to open MessageBrowser interface.");
            [self release];
            return nil;
          }

        // Verify that the interface loaded properly
        if ( ! window                ||
             ! classBrowser          ||
             ! messageListMatrix     ||
             ! messageView           ||
             ! cleanupClassesButton  ||
             ! locateButton          ||
             ! previousMessageButton ||
             ! nextMessageButton     ||
             ! deleteMessageButton   ||
             ! composeMessageButton  ||
             ! authenticButton       ||
             ! encryptedButton       )
          {
            NSLog(@"WARNING: MessageBrowser interface failed to load properly.");
          }

        // Init window
        [window makeKeyAndOrderFront: self];

        // Init class browser
        [classBrowser setTakesTitleFromPreviousColumn: NO];
        [classBrowser setTitle:@"Class"    ofColumn:CLASS_COLUMN   ];
        [classBrowser setTitle:@"Instance" ofColumn:INSTANCE_COLUMN];

        // Init message list and message view
        [self setMessageList: nil];

        // Init data
        myClassesDict   = [[NSMutableDictionary dictionary] retain];
        myInstancesDict = [[NSMutableDictionary dictionary] retain];
        myMessageDict   = [[NSMutableDictionary dictionary] retain];
        myMessageList   = nil;

        // Init preferences
        myAggregateMessagesOption = YES;
        myAutoUnhideOption        = YES;
        myAutoBringToFrontOption  = YES;
        myAutoCleanupOption       = NO;

        // Init Mutex
        myEventLock = [[NSLock alloc] init];
      }
    return self;
}

- (void) dealloc
{
    [ myClassesDict   release ];
    [ myInstancesDict release ];
    [ myMessageDict   release ];
    [ myMessageList   release ];
    [ myEventLock     release ];

    [super dealloc];
}

////
// Accessors
////

- (NSString*) selectedClass    { return [[classBrowser selectedCellInColumn: CLASS_COLUMN   ] stringValue]; }
- (NSString*) selectedInstance { return [[classBrowser selectedCellInColumn: INSTANCE_COLUMN] stringValue]; }

- (ZephyrMessage*) selectedMessage { return [[messageListMatrix selectedCell] representedObject]; }

- (NSArray*) selectedMessages
{
    NSMutableArray* aMessages       = [NSMutableArray array];
    NSEnumerator*   aCellEnumerator = [[messageListMatrix selectedCells] objectEnumerator];
    NSCell*         aCell;

    while ((aCell = [aCellEnumerator nextObject]))
        [aMessages addObject: [aCell representedObject]];

    return aMessages;
}

- (void) setMessageList: (NSMutableArray*) aList
{
    if (myMessageList != aList)
      {
        [myMessageList autorelease];

        if (aList)
            myMessageList = [aList retain];
        else
            myMessageList = [[NSMutableArray array] retain];

      }

    [self showMessageList];
}

////
// Actions
////

- (void) selectInstance: (NSString*) anInstance
                ofClass: (NSString*) aClass
{
    if ([self selectClass:aClass inColumn:CLASS_COLUMN])
      {
        [self selectClass:anInstance inColumn:INSTANCE_COLUMN];
      }

    [self updateMessageList];
}

- (void) selectNextMessage: (int) aDelta
{
    if ([[messageListMatrix selectedCells] count] != -1)
      {
        int aRowCount    = [messageListMatrix numberOfRows];
        int aSelectedRow = [messageListMatrix selectedRow];
        int aNewRow      = aSelectedRow + aDelta;

        if (aNewRow < 0 || aNewRow > aRowCount-1)
          {
            NSLog(@"ERROR: Out of bounds while selecting next message.");
            return;
          }

        [messageListMatrix selectCellAtRow:aNewRow column:0];

        [self showSelectedMessage];
      }
    else
        NSLog(@"ERROR: No selection while selecting next message.");
}

- (void) showClasses
{
    NSString* aClass     = [self selectedClass   ];
    NSString* anInstance = [self selectedInstance];

    [classBrowser loadColumnZero];

    [self selectInstance: anInstance
                 ofClass: aClass];
}

- (void) showMessageList
{
    int aSelectedRow = [messageListMatrix selectedRow];

    if (aSelectedRow < 0) aSelectedRow = 0;

    {
        int aRowCount = [messageListMatrix numberOfRows];

        while (aRowCount--) [messageListMatrix removeRow: 0];
    }

    [self showMessagesInList: myMessageList
               startingAtRow: 0];

    if ([messageListMatrix numberOfRows] || ! [self selectedInstance])
      {
        [cleanupClassesButton setEnabled: NO];
      }
    else
      {
        [cleanupClassesButton setEnabled: YES];
      }

    [messageListMatrix selectCellAtRow:aSelectedRow column:0];

    ////
    // *** FIX ME ***
    // Work around an AppKit bug
    // Resizing a matrix to a smaller size doesn't clear the now unused area.
    ////
    [messageListMatrix display];

    [messageListMatrix sizeToCells];

    [self showSelectedMessage];
}

- (void) showSelectedMessage
{
    ZephyrMessage*      aMessage     = [self selectedMessage];
    NSAttributedString* aMessageText = [aMessage text];

    [ deleteMessageButton   setEnabled: NO ];
    [ previousMessageButton setEnabled: NO ];
    [ nextMessageButton     setEnabled: NO ];
    [ authenticButton       setEnabled: NO ];

    [messageView setString: @""];

    if (aMessage)
      {
        int aSelectedRow = [messageListMatrix selectedRow ];
        int aRowCount    = [messageListMatrix numberOfRows];

        if (aSelectedRow == -1)
          {
            NSLog(@"ERROR: No selected rows in message list, but there's a selected message?!?!");
          }
        else if ([[messageListMatrix selectedCells] count] == 1)
          {
            if (aSelectedRow != 0          ) [ previousMessageButton setEnabled: YES ];
            if (aSelectedRow != aRowCount-1) [ nextMessageButton     setEnabled: YES ];
          }

        [deleteMessageButton setEnabled: YES];

        {
            NSString* anAuth = [aMessage auth];

            if (anAuth && [[anAuth lowercaseString] isEqualToString: @"yes"])
                [authenticButton setEnabled: YES];
        }

        if (aMessageText)
          {
            [[messageView textStorage] setAttributedString: aMessageText];
            [messageView setNeedsDisplay: YES];
          }
      }
}

- (void) deleteSelectedMessages
{
    NSEnumerator*  aMessageEnumerator = [[self selectedMessages] objectEnumerator];
    ZephyrMessage* aMessage;

    while ((aMessage = [aMessageEnumerator nextObject]))
        [self deleteMessage: aMessage];

    [self showClasses];
}

- (void) deleteMessage: (ZephyrMessage*) aMessage
{
    if (aMessage)
      {
        NSString*            aClass         = [[aMessage zephyrClass   ] lowercaseString];
        NSString*            anInstance     = [[aMessage zephyrInstance] lowercaseString];
        NSMutableDictionary* aClassDict     = [myMessageDict objectForKey: aClass    ];
        NSMutableArray*      anInstanceList = [aClassDict    objectForKey: anInstance];

        [anInstanceList removeObject: aMessage];

        if (myAutoCleanupOption)
          {
            if (! [anInstanceList count])
              {
                [aClassDict      removeObjectForKey: anInstance];
                [myInstancesDict removeObjectForKey: anInstance];

                if (! [aClassDict count])
                  {
                    [myMessageDict removeObjectForKey: aClass];
                    [myClassesDict removeObjectForKey: aClass];
                  }
              }
          }
        else
          {
            if (! [anInstanceList count]) [cleanupClassesButton setEnabled: YES];
          }
      }
}

- (void) cleanupClasses
{
    NSString*            aSelectedClass    = [[self selectedClass   ] lowercaseString];
    NSString*            aSelectedInstance = [[self selectedInstance] lowercaseString];
    NSMutableDictionary* aClassDict        = [myMessageDict objectForKey: aSelectedClass];

    if (aSelectedInstance)
      {
        NSMutableArray* anInstanceList = [aClassDict objectForKey: aSelectedInstance];

        if (! [anInstanceList count])
          {
            [aClassDict      removeObjectForKey: aSelectedInstance];
            [myInstancesDict removeObjectForKey: aSelectedInstance];
          }
      }
    else if (aSelectedClass)
      {
        //NSEnumerator* anInstanceEnumerator = [[aClassDict allKeys] objectEnumerator];

      }

    if (aSelectedClass)
      {
        if (! [aClassDict count])
          {
            [myMessageDict removeObjectForKey: aSelectedClass];
            [myClassesDict removeObjectForKey: aSelectedClass];
          }
      }
    else
        NSLog(@"ERROR: No selected class while cleaning up classes.");

    [self showClasses];
}

- (void) dispatchMessage: (ZephyrMessage*) aMessage
{
    NSMutableDictionary* aClassDict;
    NSMutableArray*      anInstanceList;

    NSString* aClass     = [[aMessage info] objectForKey: @"class"   ];
    NSString* anInstance = [[aMessage info] objectForKey: @"instance"];

    NSString* aLowerClass    = [aClass     lowercaseString];
    NSString* aLowerInstance = [anInstance lowercaseString];

    if (! aClass || ! anInstance)
      {
        NSLog(@"ERROR: Class and/or instance unset. Tossing message.");
        return;
      }

    LockUI;

    if ([aClass     isEqualToString: NS_CLIENT_CONTROL_CLASS   ] &&
        [anInstance isEqualToString: NS_CLIENT_CONTROL_INSTANCE] )
      {
        NSAttributedString* aMessageText = [aMessage text];

        if (aMessageText)
          {
            [[messageView textStorage] setAttributedString: aMessageText];

            [messageView setNeedsDisplay: YES];
          }
        else
            [messageView setString: @""];

        UnLockUI;
        return;
      }

    if ([aClass     isEqualToString: @"MESSAGE" ]) aClass     = @"Message";
    if ([anInstance isEqualToString: @"PERSONAL"]) anInstance = @"Personal";
    if ([anInstance isEqualToString: @"URGENT"  ]) anInstance = @"Urgent";

    if (! [myClassesDict objectForKey: aLowerClass])
        [myClassesDict setObject: aClass
                          forKey: aLowerClass];

    if (! [myInstancesDict objectForKey: aLowerInstance])
        [myInstancesDict setObject: anInstance
                            forKey: aLowerInstance];

    aClassDict = [myMessageDict objectForKey: aLowerClass];

    if (! aClassDict)
      {
        aClassDict = [NSMutableDictionary dictionary];

        [myMessageDict setObject:aClassDict forKey:aLowerClass];
      }

    anInstanceList = [aClassDict objectForKey: aLowerInstance];

    if (! anInstanceList)
      {
        anInstanceList = [NSMutableArray array];

        [aClassDict setObject:anInstanceList forKey:aLowerInstance];
      }

    [anInstanceList addObject: aMessage];

    [self showClasses];

    if ([NSApp isHidden])
      {
        if (myAutoUnhideOption)
          {
            [NSApp unhide: self];

            [self selectInstance: anInstance
                         ofClass: aClass];
          }
      }
    else
      {
        if (myAutoBringToFrontOption) [window makeKeyAndOrderFront: self];
      }

    [window display];
    [window flushWindow];
    PSWait();

    UnLockUI;
}

- (void) composeMessage
{
    [(ZephyrApplication*)NSApp composeReplyToMessage: [self selectedMessage]];
}

- (void) locateUsers
{
    [(ZephyrApplication*)NSApp locateUsers];
}

////
// IB Targets
////

- (void) selectClassTarget     : (id) aSender { LockUI; [ self updateMessageList      ]; UnLockUI; }
- (void) cleanupClassesTarget  : (id) aSender { LockUI; [ self cleanupClasses         ]; UnLockUI; }
- (void) selectMessageTarget   : (id) aSender { LockUI; [ self showSelectedMessage    ]; UnLockUI; }
- (void) nextMessageTarget     : (id) aSender { LockUI; [ self selectNextMessage:  1  ]; UnLockUI; }
- (void) previousMessageTarget : (id) aSender { LockUI; [ self selectNextMessage: -1  ]; UnLockUI; }
- (void) deleteMessageTarget   : (id) aSender { LockUI; [ self deleteSelectedMessages ]; UnLockUI; }
- (void) composeMessageTarget  : (id) aSender { LockUI; [ self composeMessage         ]; UnLockUI; }
- (void) locateUsersTarget     : (id) aSender { LockUI; [ self locateUsers            ]; UnLockUI; }

////
// classBrowser delegate methods
////

- (int)        browser: (NSBrowser*) aBrowser
  numberOfRowsInColumn: (int       ) aColumn
{
    if (aColumn == FIRST_COLUMN)
      {
        return [myMessageDict count];
      }
    else if (aColumn > FIRST_COLUMN && aColumn <= LAST_COLUMN)
      {
        NSBrowserCell*       aCell        = [aBrowser selectedCellInColumn: aColumn-1];
        NSMutableDictionary* aMessageDict = [aCell representedObject];

        if (aMessageDict)
            return [aMessageDict count];
        else
            NSLog(@"ERROR: Selected class (if any) in column %d has no messages.", aColumn-1);
      }
    else // Out of bounds
        NSLog(@"ERROR: Class browser is requesting a row count out of bounds (column %d).", aColumn);

    NSLog(@"ERROR: Returning zero rows for column %d.", aColumn);
    return 0;
}

- (void)  browser: (NSBrowser*    ) aBrowser
  willDisplayCell: (NSBrowserCell*) aCell
            atRow: (int           ) aRow
           column: (int           ) aColumn
{
    NSMutableDictionary* aMessageDict = nil;

    if (aColumn == FIRST_COLUMN)
      {
        aMessageDict = myMessageDict;
      }
    else if (aColumn > FIRST_COLUMN && aColumn <= LAST_COLUMN)
      {
        NSBrowserCell* aPreviousCell = [aBrowser selectedCellInColumn: aColumn-1];

        if (aPreviousCell)
            aMessageDict = [aPreviousCell representedObject];
        else
            NSLog(@"ERROR: No selected item in previous column (%d).", aColumn-1);
      }
    else // Out of bounds
        NSLog(@"ERROR: Class browser is drawing cells out of bounds (column %d).", aColumn);

    {
        NSArray*  aClasses    = [[aMessageDict allKeys] sortedArrayUsingSelector: @selector(compare:)];
        NSString* aLowerClass = [aClasses objectAtIndex: aRow];
        NSString* aClass      = nil;

        switch (aColumn)
          {
            case CLASS_COLUMN:
                aClass = [myClassesDict objectForKey: aLowerClass];
                break;

            case INSTANCE_COLUMN:
                aClass = [myInstancesDict objectForKey: aLowerClass];
                break;

            default:
                NSLog(@"ERROR: Displaying in unknown column.");
                aClass = aLowerClass;
                break;
          }

        if (aClass)
          {
            [aCell setStringValue: aClass];
            [aCell setRepresentedObject: [aMessageDict objectForKey: aLowerClass]];
            [aCell setLeaf: (aColumn == LAST_COLUMN) ? YES:NO];
          }
        else
          {
            NSLog(@"ERROR: Item in class browser at column %d, row %d has no data to display.", aColumn, aRow);

            [aCell setStringValue: @"***ERROR***"];
            [aCell setLeaf: YES];
          }
    }
}

- (NSString*) browser: (NSBrowser*) aBrowser
        titleOfColumn: (int       ) aColumn
{
    ////
    // *** FIX ME ***
    // This method should be unnecessary, because we set the title of the columns in
    // classBrowser during init.
    ////
    switch (aColumn)
      {
        case CLASS_COLUMN:     return @"Class"    ; break;
        case INSTANCE_COLUMN:  return @"Instance" ; break;
        case RECIPIENT_COLUMN: return @"Recipient"; break;
      }

    return @"**ERROR**";
}

////
// splitView delegate methods
////

#define MIN_SIZE_VIEW_0  60.0
#define MIN_SIZE_VIEW_1  70.0
#define MIN_SIZE_VIEW_2 190.0

- (void)       splitView: (NSSplitView*) aSplitView
  constrainMinCoordinate: (float*      ) aMinimumHeight
           maxCoordinate: (float*      ) aMaximumHeight
             ofSubviewAt: (int         ) anIndex
{
    NSArray* aSubViews        = [aSplitView subviews];
    float    aSplitViewHeight = [aSplitView frame].size.height;
    float    aDividerHeight   = [aSplitView dividerThickness];

    switch (anIndex)
      {
        case 0:
        {
            float aView2Height = [[aSubViews objectAtIndex:2] frame].size.height;

            *aMinimumHeight =                    MIN_SIZE_VIEW_0;
            *aMaximumHeight = aSplitViewHeight - MIN_SIZE_VIEW_1 - aDividerHeight - aView2Height - aDividerHeight;

            break;
        }
        case 1:
        {
            float aView0Height = [[aSubViews objectAtIndex:0] frame].size.height;

            *aMinimumHeight = aView0Height     + MIN_SIZE_VIEW_1 + aDividerHeight;
            *aMaximumHeight = aSplitViewHeight - MIN_SIZE_VIEW_2 - aDividerHeight;

            break;
        }
        default:
            NSLog(@"SplitView size request out of bounds (%d)", anIndex);
            break;
      }

    NSLog(@"Constraints #%d (%f,%f)", anIndex, *aMinimumHeight, *aMaximumHeight);
}

- (BOOL)   splitView: (NSSplitView*) aSplitView
  canCollapseSubview: (NSView*     ) aSubview
{
    return YES;
}

////
// Private methods
////

- (NSMutableArray*) aggregateMessagesInDictionary: (NSDictionary*) aMessageDict
                                       forClasses: (NSArray*     ) aClasses
                                         inColumn: (int          ) aColumn
{
    NSMutableArray* aResult = [NSMutableArray array];

    if (! aClasses) aClasses = [aMessageDict allKeys];

    if (! aClasses) return nil;

    {
        NSEnumerator* aClassEnumerator = [aClasses objectEnumerator];
        NSString*     aClass;

        while ((aClass = [aClassEnumerator nextObject]))
          {
            NSArray* aTempList = nil;

            if (aColumn >= FIRST_COLUMN && aColumn < LAST_COLUMN)
              {
                NSDictionary* aChild = [aMessageDict objectForKey: aClass];

                if (aChild)
                    aTempList = [self aggregateMessagesInDictionary: aChild
                                                         forClasses: nil
                                                           inColumn: aColumn+1];
                else
                    NSLog(@"ERROR: Invalid key (%@) in class browser at column %d.", aClass, aColumn);
              }
            else if (aColumn == LAST_COLUMN)
              {
                aTempList = [aMessageDict objectForKey: aClass];
              }
            else // Out of bounds
              {
                return nil;
              }

            if (aTempList)
                [aResult addObjectsFromArray: aTempList];
            else
                NSLog(@"ERROR: No messages for key (%@) in class browser at column %d.", aClass, aColumn);
          }
    }

    return aResult;
}

- (void) updateMessageList
{
    int aColumn = [classBrowser selectedColumn];

    if (aColumn == -1)
      {
        [self setMessageList: nil];
        return;
      }

    if (myAggregateMessagesOption)
      {
        NSDictionary* aMessageDict = nil;

        if (aColumn == FIRST_COLUMN)
          {
            aMessageDict = myMessageDict;
          }
        else if (aColumn > FIRST_COLUMN && aColumn <= LAST_COLUMN)
          {
            NSBrowserCell* aCell = [classBrowser selectedCellInColumn: aColumn-1];

            aMessageDict = [aCell representedObject];
          }

        if (! aMessageDict)
          {
            NSLog(@"ERROR: Class browser selection has no data.");
            return;
          }

        {
            NSMutableArray* aClasses        = [NSMutableArray array];
            NSArray*        aCells          = [classBrowser selectedCells];
            NSEnumerator*   aCellEnumerator = [aCells objectEnumerator];
            NSBrowserCell*  aCell;

            while ((aCell = [aCellEnumerator nextObject]))
              {
                NSString* aClass = [[aCell stringValue] lowercaseString];

                if (aClass)
                    [aClasses addObject: aClass];
                else
                    NSLog(@"ERROR: Browser cell has no string value.");
              }

            {
                NSMutableArray* aList = [self aggregateMessagesInDictionary: aMessageDict
                                                                 forClasses: aClasses
                                                                   inColumn: aColumn];

                if (! aList) NSLog(@"ERROR: Aggregate list for selection is nil.");

                [self setMessageList: aList];
            }
        }
      }
    else // if (! myAggregateMessagesOption)
      {
        if (aColumn >= FIRST_COLUMN && aColumn < LAST_COLUMN)
          {
            [classBrowser selectRow:0 inColumn:aColumn+1];
          }
        else if (aColumn == LAST_COLUMN)
          {
            NSArray*        aCells       = [classBrowser selectedCells];
            NSMutableArray* aList        = [NSMutableArray array];
            NSEnumerator*   anEnumerator = [aCells objectEnumerator];
            NSBrowserCell*  aCell;

            while ((aCell = [anEnumerator nextObject]))
              {
                NSArray* aTempList = [aCell representedObject];

                if (! aTempList)
                    NSLog(@"ERROR: Selected instance (%@) has no messages.", [aCell stringValue]);

                [aList addObjectsFromArray: aTempList];
              }

            [self setMessageList: aList];
          }
        else // Out of bounds
          {
            NSLog(@"ERROR: User selection is out of bounds (column %d).", aColumn);
          }
      }

    ////
    // *** FIX ME ***
    // This should be unnecessary, but for some reason, it's not.
    // Select and instance, and then a class, and the aggregate list is not properly
    //  displayed unless we call showMessageList. But setMessageList: calls
    //  showMessageList for us, and we should need to call it again. Whassupwidat?
    ////
    [self showMessageList];
}

- (int) showMessagesInList: (NSArray*) aMessageList
             startingAtRow: (int     ) aRow
{
    if (aMessageList)
      {
        NSEnumerator*  aMessageEnumerator = [aMessageList objectEnumerator];
        ZephyrMessage* aMessage;

        for (; (aMessage = [aMessageEnumerator nextObject]); aRow++)
          {
#ifdef ALLOW_NESTED_ARRAYS_IN_MESSAGE_LIST
            if ([aMessage isKindOfClass: [NSArray class]])
              {
                aRow = [self showMessagesInList: (NSArray*)aMessage
                                  startingAtRow: aRow];
              }
            else
#endif
              {
                NSTextFieldCell* aCell;

                [messageListMatrix addRow];

                aCell = [messageListMatrix cellAtRow:aRow column:0];

                [aCell setStringValue: [NSString stringWithFormat: @"%@ <%@>",
                                                                   [[aMessage info] objectForKey: @"signature"],
                                                                   [[aMessage info] objectForKey: @"sender"   ]]];
                [aCell setRepresentedObject: aMessage];
              }
          }
      }
    return aRow;
}

- (BOOL) selectClass: (NSString*) aClass
            inColumn: (int      ) aColumn
{
    NSMatrix* aClassMatrix = [classBrowser matrixInColumn: aColumn];
    int       aCount       = [aClassMatrix numberOfRows];
    int       aRow;

    if (aClass)
      {
        for (aRow = 0; aRow < aCount; aRow++)
          {
            NSBrowserCell* aCell      = [classBrowser loadedCellAtRow:aRow column:aColumn];
            NSString*      aCellClass = [aCell stringValue];

            if (aCellClass)
              {
                if ([[aClass lowercaseString] isEqualToString: [aCellClass lowercaseString]])
                  {
                    [classBrowser selectRow:aRow inColumn:aColumn];
                    return YES;
                  }
              }
            else
                NSLog(@"ERROR: Cell has no title.");
          }
      }

    return NO;
}

@end
