/* rdline.c
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * Copyright (c) 1992, 1993, 1994, 1995, 1996, 1997 by Mike Gleason, NCEMRSoft.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, without modification,
 * are permitted.
 * 
 */

#include "syshdrs.h"

#ifdef HAVE_LIBREADLINE
#	include <readline/readline.h>
#	include <readline/history.h>
#endif	/* HAVE_LIBREADLINE */

#include "shell.h"
#include "util.h"
#include "bookmark.h"
#include "cmds.h"
#include "ls.h"
#include "readln.h"

const char *tcap_normal = "";
const char *tcap_boldface = "";
const char *tcap_underline = "";
const char *tcap_reverse = "";

extern int gEventNumber;
extern LsCacheItem gLsCache[kLsCacheSize];
extern FTPConnectionInfo gConn;
extern char gRemoteCWD[512];
extern char gOurDirectoryPath[];
extern char gVersion[];

/* For a few selected terminal types, we'll print in boldface, etc.
 * This isn't too important, though.
 */
void
InitTermcap(void)
{
	char *term;
	
	if ((term = getenv("TERM")) == NULL) {
		tcap_normal = "";
		tcap_boldface = "";
		tcap_underline = "";
		tcap_reverse = "";
	} else if (
		(strcmp(term, "xterm") == 0) ||
		(strcmp(term, "vt100") == 0) ||
		(strcmp(term, "linux") == 0) ||
		(strcmp(term, "vt102") == 0)
	) {
		tcap_normal = "\033[0m";       /* Default ANSI escapes */
		tcap_boldface = "\033[1m";
		tcap_underline = "\033[4m";
		tcap_reverse = "\033[7m";
	} else {
		tcap_normal = "";
		tcap_boldface = "";
		tcap_underline = "";
		tcap_reverse = "";
	}
}	/* InitTermcap */



#ifdef HAVE_LIBREADLINE

static char *
FindStartOfCurrentCommand(void)
{
	char *scp;
	char *start;
	int qc;

	for (scp = rl_line_buffer;;) {
		start = scp;
		for (;;) {
			if (*scp == '\0')
				goto done;
			if (!isspace(*scp))
				break;
			scp++;
		}
		start = scp;

		for (;;) {
			if (*scp == '\0') {
				goto done;
			} else if ((*scp == '"') || (*scp == '\'')) {
				qc = *scp++;

				for (;;) {
					if (*scp == '\0') {
						goto done;
					} else if (*scp == '\\') {
						scp++;
						if (*scp == '\0')
							goto done;
						scp++;
					} else if (*scp == qc) {
						scp++;
						break;
					} else {
						scp++;
					}
				}
			} else if (*scp == '\\') {
				scp++;
				if (*scp == '\0')
					goto done;
				scp++;
			} else if ((*scp == ';') || (*scp == '\n')) {
				/* command ended */
				scp++;
				if (*scp == '\0')
					goto done;
				break;
			} else {
				scp++;
			}
		}
	}
done:
	return (start);
}	/* FindStartOfCurrentCommand */



static FileInfoListPtr
GetLsCacheFileList(const char *const item)
{
	int ci;
	int sortBy;
	int sortOrder;
	FileInfoListPtr filp;

	ci = LsCacheLookup(item);
	if (ci < 0) {
		/* This dir was not in the
		 * cache -- go get it.
		 */
		(void) rl_insert_text("...");
		rl_redisplay();
		Ls(item, 'l', "", NULL);
		(void) rl_do_undo();
		ci = LsCacheLookup(item);
		if (ci < 0)
			return NULL;
	}

	sortBy = 'n';		/* Sort by filename. */
	sortOrder = 'a';	/* Sort in ascending order. */
	filp = &gLsCache[ci].fil;
	SortFileInfoList(filp, sortBy, sortOrder);
	return filp;
}	/* GetLsCacheFileList */




static char *
RemoteCompletionFunction(char *text, int state, int fTypeFilter)
{
	char rpath[256];
	char *cp;
	char *cp2;
	char *textbasename;
	int fType;
	FileInfoPtr diritemp;
	FileInfoListPtr filp;
	int textdirlen;
	size_t tbnlen;
	size_t flen, mlen;
	static FileInfoVec diritemv;
	static int i;

	textbasename = strrchr(text, '/');
	if (textbasename == NULL) {
		textbasename = text;
		textdirlen = -1;
	} else {
		textdirlen = (int) (textbasename - text);
		textbasename++;
	}
	tbnlen = strlen(textbasename);

	if (state == 0) {
		PathCat(rpath, sizeof(rpath), gRemoteCWD, text);
		cp2 = strrchr(rpath, '/');
		if (cp2 == NULL) {
			return NULL;
		} else if (cp2 == rpath) {
			/* Item in root directory. */
			cp2++;
		}
		*cp2 = '\0';

		filp = GetLsCacheFileList(rpath);
		if (filp == NULL)
			return NULL;

		diritemv = filp->vec;
		if (diritemv == NULL)
			return NULL;

		i = 0;
	}

	for ( ; ; ) {
		diritemp = diritemv[i];
		if (diritemp == NULL)
			break;

		i++;
		fType = (int) diritemp->type;
		if ((fTypeFilter == 0) || (fType == fTypeFilter) || (fType == '@')) {
			if (strncmp(textbasename, diritemp->relname, tbnlen) == 0) {
				flen = strlen(diritemp->relname);
				if (textdirlen < 0) {
					mlen = flen + 2;
					cp = malloc(mlen);
					if (cp == NULL)
						return (NULL);
					(void) memcpy(cp, diritemp->relname, mlen);
				} else {
					mlen = textdirlen + 1 + flen + 2;
					cp = malloc(mlen);
					if (cp == NULL)
						return (NULL);
					(void) memcpy(cp, text, textdirlen);
					cp[textdirlen] = '/';
					(void) strcpy(cp + textdirlen + 1, diritemp->relname);
				}
				if (fType == 'd') {
#ifdef HAVE_LIBREADLINE
					rl_completion_append_character = '/';
#else
					(void) strcat(cp, "/");
#endif	/* Readline 2.1 or higher */
#ifdef HAVE_LIBREADLINE
				} else {
					rl_completion_append_character = ' ';
#endif	/* Readline 2.1 or higher */
				}
				return cp;
			}
		}
	}
	return NULL;
}	/* RemoteCompletionFunction */




static char *
RemoteFileCompletionFunction(char *text, int state)
{
	char *cp;

	cp = RemoteCompletionFunction(text, state, 0);
	return cp;
}	/* RemoteFileCompletionFunction */




static char *
RemoteDirCompletionFunction(char *text, int state)
{
	char *cp;

	cp = RemoteCompletionFunction(text, state, 'd');
	return cp;
}	/* RemoteDirCompletionFunction */




static char *
CompletionFunction(char *text, int state)
{
	char *cp;
	char *cmdstart;
	ArgvInfo ai;
	int bUsed;
	CommandPtr cmdp;
	static int flags;

	if (state == 0) {
		flags = -1;
		cmdstart = FindStartOfCurrentCommand();
		if (cmdstart == NULL)
			return NULL;
		(void) memset(&ai, 0, sizeof(ai));
		bUsed = MakeArgv(cmdstart, &ai.cargc, ai.cargv,
			(int) (sizeof(ai.cargv) / sizeof(char *)),
			ai.argbuf, sizeof(ai.argbuf),
			ai.noglobargv);
		if (bUsed <= 0)
			return NULL;
		if (ai.cargc == 0)
			return NULL;

		cmdp = GetCommandByName(ai.cargv[0], 0);
		if (cmdp == kAmbiguousCommand) {
			return NULL;
		} else if (cmdp == kNoCommand) {
			return NULL;
		}
		flags = cmdp->flags;
	}
	if (flags < 0)
		return NULL;
	if ((flags & kCompleteLocalFile) != 0) {
		cp = filename_completion_function(text, state);
		return cp;
	} else if ((flags & kCompleteRemoteFile) != 0) {
		rl_filename_quoting_desired = 1;
		rl_filename_completion_desired = 1;
		cp = RemoteFileCompletionFunction(text, state);
		return cp;
	} else if ((flags & kCompleteRemoteDir) != 0) {
		rl_filename_quoting_desired = 1;
		rl_filename_completion_desired = 1;
		cp = RemoteDirCompletionFunction(text, state);
		return cp;
	}	
	return NULL;
}	/* CompletionFunction */




void
LoadHistory(void)
{
	char pathName[256];

	if (gOurDirectoryPath[0] == '\0')
		return;
	(void) OurDirectoryPath(pathName, sizeof(pathName), kHistoryFileName);

	(void) read_history(pathName);
}	/* LoadHistory */




/* Save the commands they typed in a history file, then they can use
 * readline to go through them again next time.
 */
void
SaveHistory(void)
{
	char pathName[256];

	if (gOurDirectoryPath[0] == '\0')
		return;		/* Don't create in root directory. */
	(void) OurDirectoryPath(pathName, sizeof(pathName), kHistoryFileName);

	stifle_history(100);
	(void) write_history(pathName);
}	/* SaveHistory */




void
InitReadline(void)
{
	rl_readline_name = "ncftp";
	rl_completion_entry_function = (Function *) CompletionFunction;
	rl_filename_quote_characters = " \t\n\\\"'@<>=;|&()#$`?*[!:";
	rl_completer_quote_characters = "\"";
	using_history();
	LoadHistory();
}	/* InitReadline */




char *
Readline(char *prompt)
{
	return (readline(prompt));
}	/* Readline */



void
AddHistory(char *line)
{
	add_history(line);
}	/* AddHistory */



void
DisposeReadline(void)
{
	SaveHistory();
}	/* DisposeReadline */


#else	/* HAVE_LIBREADLINE */

char *
Readline(char *prompt)
{
	char buf[512];

	fputs(prompt, stdout);
	if (fgets(buf, sizeof(buf) - 1, stdin) == NULL)
		return NULL;
	return (StrDup(buf));
}	/* Readline */




void
LoadHistory(void)
{
	/* nothing */
}	/* LoadHistory */




void
SaveHistory(void)
{
	/* nothing */
}	/* SaveHistory */



void
AddHistory(char *line)
{
	/* nothing */
}	/* AddHistory */



void
InitReadline(void)
{
	/* nothing */
}	/* InitReadline */



void
DisposeReadline(void)
{
	/* nothing */
}	/* DisposeReadline */


#endif	/* HAVE_LIBREADLINE */




void
PrintStartupBanner(void)
{
	char v[80], *cp;
	char vdate[32];

	/* Print selected information from the version ID. */
	vdate[0] = '\0';
	(void) STRNCPY(v, gVersion + 5);
	cp = strchr(v, ',');
	if (cp != NULL) {
		*cp = '\0';
		cp[-5] = '\0';
		(void) STRNCPY(vdate, " (");
		(void) STRNCAT(vdate, v + 16);
		(void) STRNCAT(vdate, ", ");
		(void) STRNCAT(vdate, cp - 4);
		(void) STRNCAT(vdate, ")");
	}

#if defined(BETA) && (BETA > 0)
	(void) fprintf(stdout, "%s%.11s beta %d%s%s by Mike Gleason, NCEMRSoft.\n",
		tcap_boldface,
		gVersion + 5,
		BETA,
		tcap_normal,
		vdate
	);
#else
	(void) fprintf(stdout, "%s%.11s%s%s by Mike Gleason, NCEMRSoft.\n",
		tcap_boldface,
		gVersion + 5,
		tcap_normal,
		vdate
	);
#endif
	(void) fflush(stdout);
}	/* PrintStartupBanner */




/* Print the command shell's prompt. */
void
MakePrompt(char *dst, size_t dsize)
{
#ifdef HAVE_LIBREADLINE
	char acwd[64];

#	ifdef HAVE_SNPRINTF
	if (gConn.loggedIn != 0) {
		AbbrevStr(acwd, gRemoteCWD, 25, 0);
		snprintf(dst, dsize, "%d ncftp %s > ",
			gEventNumber, acwd);
	} else {
		snprintf(dst, dsize, "%d ncftp> ", gEventNumber);
	}
#	else	/* HAVE_SNPRINTF */
	(void) sprintf(dst, "%d ", gEventNumber);
	(void) Strncat(dst, "ncftp", dsize);
	if (gConn.loggedIn != 0) {
		AbbrevStr(acwd, gRemoteCWD, 25, 0);
		(void) Strncat(dst, " ", dsize);
		(void) Strncat(dst, acwd, dsize);
		(void) Strncat(dst, " ", dsize);
	}
	(void) Strncat(dst, ">", dsize);
	(void) Strncat(dst, " ", dsize);
#	endif	/* HAVE_SNPRINTF */
#else	/* HAVE_LIBREADLINE */
	char acwd[64];

#	ifdef HAVE_SNPRINTF
	if (gConn.loggedIn != 0) {
		AbbrevStr(acwd, gRemoteCWD, 25, 0);
		snprintf(dst, dsize, "%d %sncftp%s %s %s>%s ",
			gEventNumber, tcap_boldface, tcap_normal, acwd,
			tcap_boldface, tcap_normal);
	} else {
		snprintf(dst, dsize, "%d %sncftp%s> ",
			gEventNumber, tcap_boldface, tcap_normal);
	}
#	else	/* HAVE_SNPRINTF */
	(void) sprintf(dst, "%d ", gEventNumber);
	(void) Strncat(dst, tcap_boldface, dsize);
	(void) Strncat(dst, "ncftp", dsize);
	(void) Strncat(dst, tcap_normal, dsize);
	if (gConn.loggedIn != 0) {
		AbbrevStr(acwd, gRemoteCWD, 25, 0);
		(void) Strncat(dst, " ", dsize);
		(void) Strncat(dst, acwd, dsize);
		(void) Strncat(dst, " ", dsize);
	}
	(void) Strncat(dst, tcap_boldface, dsize);
	(void) Strncat(dst, ">", dsize);
	(void) Strncat(dst, tcap_normal, dsize);
	(void) Strncat(dst, " ", dsize);
#	endif	/* HAVE_SNPRINTF */
#endif	/* HAVE_LIBREADLINE */
}	/* MakePrompt */
