/*
 HexInputServer.m
 Copyright (c) 1998, Apple Computer, Inc.

 You may incorporate this sample code into your applications without
 restriction, though the sample code has been provided "AS IS" and the
 responsibility for its operation is 100% yours.  However, what you are
 not permitted to do is to redistribute the source as "DTS Sample Code"
 after having made changes. If you're going to re-distribute the source,
 we require that you make it clear in the source that the code was
 descended from Apple Sample Code, but that you've made changes.
*/

#import <AppKit/AppKit.h>
#import <AppKit/NSStatusBar.h>
#import <AppKit/NSStatusItem.h>
#import "HexInputServer.h"
#import "HexInputContext.h"

// Converts client object's address to NSNumber
#define KEY_FROM_CLIENTID(theClient) ([NSNumber numberWithUnsignedInt:(unsigned int)theClient])
// UI settings for inputFeedbackField.
#define FEEDBACK_FIELD_FONT_SIZE (12.0)
#define NSInputCandidateWindowLevel (NSModalPanelWindowLevel + 1) // It must work with modal panels

// Localized strings for status item title
#define ALPHA_STATUS_TITLE (NSLocalizedString(@"Alpha", "This message is displayed as status item title when alphabet mode"))
#define HEX_STATUS_TITLE (NSLocalizedString(@"Hexadecimal", "This message is displayed as stauts item title when hexadecimal mode"))

// Class variable holding sharedInstance;
static HexInputServer *_SharedHexInputServerInstance = nil;

// Class variable holding textfiled for hex value display
static NSTextField *_InputFeedbackField = nil;

// Class variable holding statusitem for mode menu
static NSStatusItem *_AuxMenuStatusItem = nil;

// Private NSStatusItem API that's provided only for input servers
#define	_NSInputMethodMenuPriority	4000	// any input method menu

@interface NSStatusItem (NSStatusItem_Private)
+ (id) _itemInStatusBar:(NSStatusBar*)statusBar withLength:(float)length withPriority:(int)priority;
@end

@implementation HexInputServer
/* Private methods
*/
// Updates status menu states according to isHexInput
+ (void)_updateAuxStatusItemWithState:(BOOL)isHexInput {
    NSStatusItem *statusItem = [self auxMenuStatusItem];
    NSMenu *menu = [statusItem menu];

    [statusItem setTitle:(isHexInput ? HEX_STATUS_TITLE : ALPHA_STATUS_TITLE)];

    [[menu itemWithTag:0] setState:!isHexInput]; // Alpha input menu item
    [[menu itemWithTag:1] setState:isHexInput]; // Hex input menu item
}

// Called from status menu items. Sends corresponding commands to current context via doCommandBySelector:client: method
- (void)_modeMenuItemSelected:(id)sender {
    [self doCommandBySelector:([sender tag] ? @selector(beginHexInput:) : @selector(insertNewline:)) client:[_currentClientContext client]];
}

/* Class methods
*/
// Shared instance
+ (id)sharedInstance {
    if (!_SharedHexInputServerInstance) {
        _SharedHexInputServerInstance = [[HexInputServer alloc] init];
    }

    return _SharedHexInputServerInstance;
}

// Conversion feedback textfield
+ (NSTextField *)inputFeedbackField {
    if (!_InputFeedbackField) {
        NSPanel *panel = [[NSPanel allocWithZone:[self zone]] initWithContentRect:NSMakeRect(0, 0, 100, 40) styleMask:NSBorderlessWindowMask|NSUtilityWindowMask backing:NSBackingStoreBuffered defer:YES];

        [panel setHidesOnDeactivate:NO];
        [panel setLevel:NSInputCandidateWindowLevel];
        [panel setBecomesKeyOnlyIfNeeded:YES];
        [panel setWorksWhenModal:YES];

        _InputFeedbackField = [[[NSTextField allocWithZone:[self zone]] initWithFrame:NSMakeRect(0, 0, 60, 24)] autorelease];
        [_InputFeedbackField setEditable:NO];
        [_InputFeedbackField setSelectable:NO];
        [_InputFeedbackField setFont:[NSFont messageFontOfSize:FEEDBACK_FIELD_FONT_SIZE]];
        [panel setContentView:_InputFeedbackField];
    }

    return _InputFeedbackField;
}

// NSStatusItem for auxiliary menu
+ (NSStatusItem *)auxMenuStatusItem {
#ifndef WIN32 // It makes no sense to have this on Windows UI
    if (!_AuxMenuStatusItem) {
        NSMenu *auxMenu = [[[NSMenu allocWithZone:[self zone]] initWithTitle:NSLocalizedString(@"HexInput Mode", "This message is displayed as mode menu's title when it's tearoff'ed.")] autorelease];
        NSMenuItem *anItem;

        anItem = [[[NSMenuItem allocWithZone:[self zone]] initWithTitle:NSLocalizedString(@"Alphabet input", "This message is displayed as menu item title for alphabet input mode") action:@selector(_modeMenuItemSelected:) keyEquivalent:@""] autorelease];
        [anItem setTag:0];
        [anItem setTarget:_SharedHexInputServerInstance];

        [auxMenu addItem:anItem];

        anItem = [[[NSMenuItem allocWithZone:[self zone]] initWithTitle:NSLocalizedString(@"Hexadecimal input", "This message is displayed as menu item title for hexadecimal input mode") action:@selector(_modeMenuItemSelected:) keyEquivalent:@""] autorelease];
        [anItem setTag:1];
        [anItem setTarget:_SharedHexInputServerInstance];

        [auxMenu addItem:anItem];

        _AuxMenuStatusItem = [[NSStatusItem _itemInStatusBar:[NSStatusBar systemStatusBar] withLength:NSVariableStatusItemLength withPriority:_NSInputMethodMenuPriority] retain];
        [_AuxMenuStatusItem setMenu:auxMenu];
    }

    return _AuxMenuStatusItem;
#else
    return nil;
#endif WIN32
}

/* Deallocation
*/
- (void)dealloc {
    [_clientTable release];

    [super dealloc];
}

/* NSInputServiceProivder protocol methods
*/
- (BOOL)canBeDisabled {
    return YES; // This server can be disabled (it just send back messages when disabled)
}

- (BOOL)wantsToInterpretAllKeystrokes {
    return NO; // This server is using key binding manager
}

- (BOOL)wantsToHandleMouseEvents {
    return NO; // This server handles single character input, so no mouse support needed
}

- (BOOL)wantsToDelayTextChangeNotifications {
    return NO; // This server handles simple single character input, so no need to delay the notification
}

// Pass aString to sender's corresponding context if it's, indeed, "active" conversation.
- (void) insertText:(id)aString client:(id)sender {
    HexInputContext *context = [_clientTable objectForKey:KEY_FROM_CLIENTID(sender)];

    if (!_isEnabled || (_currentClientContext != context) || ([sender conversationIdentifier] != [context currentConversation])) { // This means sender is either NSInputText non-conformant simple client or non keyboard-focused view. Just send it back.
        [sender insertText:aString];
    } else {
        [context insertText:aString];
    }
}

// Pass aSelector to sender's corresponding context if it's, indeed, "active" conversation.
- (void) doCommandBySelector:(SEL)aSelector client:(id)sender {
    HexInputContext *context = [_clientTable objectForKey:KEY_FROM_CLIENTID(sender)];

    if (!_isEnabled || (_currentClientContext != context) || ([sender conversationIdentifier] != [context currentConversation])) { // This means sender is either NSInputText non-conformant simple client or non keyboard-focused view. Just send it back.
        [sender doCommandBySelector:aSelector];
    } else {
        [context doCommandBySelector:aSelector];
        if ((aSelector == @selector(beginHexInput:)) || (aSelector == @selector(insertNewline:))) { // Those two commands switches input states (alphabet/hex).  Update the status menu states.
            [[self class] _updateAuxStatusItemWithState:[context isHexInputMode]];
        }
    }
}

// Send the message to sender's corresponding context.
- (void) markedTextAbandoned:(id)sender {
    [[_clientTable objectForKey:KEY_FROM_CLIENTID(sender)] abandonCurrentConversation];
}

// Unmark it & abandon it
- (void) markedTextSelectionChanged:(NSRange)newSel client:(id)sender {
    [sender unmarkText];
    [self markedTextAbandoned:sender];
}

// Client app is either exited or died
- (void) terminate:(id)sender {
    HexInputContext *context;
    id key = KEY_FROM_CLIENTID(sender);

    if ((context = [_clientTable objectForKey:key]) == _currentClientContext) { // if it's currect context, clear the state
        [[_InputFeedbackField window] orderOut:self];
        _currentClientContext = nil;
    }
    [_clientTable removeObjectForKey:key];
}

// sender became the active application using this server for input
- (void) inputClientBecomeActive:(id)sender {
    NSStatusItem *statusItem = [[self class] auxMenuStatusItem];
    HexInputContext *context;
    id key = KEY_FROM_CLIENTID(sender);

    if (!(context = [_clientTable objectForKey:key])) { // If first time, register it in _clientTable. (It will retain the client proxy)
        context = [HexInputContext hexInputContextWithClient:sender];

        if (!_clientTable) {
            _clientTable = [[NSMutableDictionary allocWithZone:[self zone]] initWithCapacity:10];
        }
        [_clientTable setObject:context forKey:key];
    }

    _currentClientContext = context;

    // Show the status item
    [statusItem setLength:NSVariableStatusItemLength];
    [statusItem setTitle:ALPHA_STATUS_TITLE];
}

// sender resigned the active application state or changed input manager
- (void) inputClientResignActive:(id)sender {
    _currentClientContext = nil;

    // Close the status item
    [[[self class] auxMenuStatusItem] setLength:0.1]; // Setting it to 0 length will hide the status item
}

// a NSView conforming to NSTextInput within sender now has keyboard focus
- (void) inputClientEnabled:(id)sender {
    HexInputContext *context = [_clientTable objectForKey:KEY_FROM_CLIENTID(sender)];

    _isEnabled = YES;
    [context updateUserFeedbackWithState:_isEnabled]; // orderFront & reposition aux window showing hex value

    // Update status item states & enable it
    [[self class] _updateAuxStatusItemWithState:[context isHexInputMode]];
    [[[self class] auxMenuStatusItem] setEnabled:YES];
}

// sender does not have valid (NSTextInput conforming) view as first responder
- (void) inputClientDisabled:(id)sender {
    _isEnabled = NO;
    [[_clientTable objectForKey:KEY_FROM_CLIENTID(sender)] updateUserFeedbackWithState:_isEnabled]; // orderOut aux window

    // Disable status item
    [[[self class] auxMenuStatusItem] setEnabled:NO];
}

// sender's keyboard focus is about to change
- (void) activeConversationWillChange:(id)sender fromOldConversation:(long)oldConversation {
    [[_InputFeedbackField window] orderOut:self];
}

// sender has new keyboard focus view
- (void) activeConversationChanged:(id)sender toNewConversation:(long)newConversation {
    HexInputContext *context = [_clientTable objectForKey:KEY_FROM_CLIENTID(sender)];

    [context setCurrentConversation:newConversation]; // Sets current conversation identifier & calls updateUserFeedbackWithState: internally
    [[self class] _updateAuxStatusItemWithState:[context isHexInputMode]]; // Update status item states
}

@end
