/*++
/* NAME
/*	dict_env 3
/* SUMMARY
/*	dictionary manager interface to environment variables
/* SYNOPSIS
/*	#include <dict_env.h>
/*
/*	DICT	*dict_env_open(name, flags)
/*	const char *name;
/*	int	flags;
/* DESCRIPTION
/*	dict_env_open() opens the environment variable array and
/*	makes it accessible via the generic operations documented
/*	in dict_open(3). The \fIname\fR and \fIflags\fR arguments
/*	are ignored.
/* SEE ALSO
/*	dict(3) generic dictionary manager
/*	safe_getenv(3) safe getenv() interface
/* LICENSE
/* .ad
/* .fi
/*	The Secure Mailer license must be distributed with this software.
/* AUTHOR(S)
/*	Wietse Venema
/*	IBM T.J. Watson Research
/*	P.O. Box 704
/*	Yorktown Heights, NY 10598, USA
/*--*/

/* System library. */

#include "sys_defs.h"
#include <stdio.h>			/* sprintf() prototype */
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

/* Utility library. */

#include "mymalloc.h"
#include "msg.h"
#include "safe.h"
#include "dict.h"
#include "dict_env.h"

/* dict_env_update - update environment array */

static void dict_env_update(DICT *unused_dict, const char *name, const char *value)
{
    if (setenv(name, value, 1))
	msg_fatal("setenv: %m");
}

/* dict_env_lookup - access environment array */

static const char *dict_env_lookup(DICT *unused_dict, const char *name)
{
    return (safe_getenv(name));
}

/* dict_env_close - close environment dictionary */

static void dict_env_close(DICT *dict)
{
    myfree((char *) dict);
}

/* dict_env_open - make association with environment array */

DICT   *dict_env_open(const char *unused_name, int unused_flags)
{
    DICT   *dict;

    dict = (DICT *) mymalloc(sizeof(*dict));
    dict->lookup = dict_env_lookup;
    dict->update = dict_env_update;
    dict->close = dict_env_close;
    dict->fd = -1;
    return (dict);
}
