#import "OWShelfView.h"

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>
#import <OmniAppKit/OmniAppKit.h>
#import <OWF/OWF.h>

RCS_ID("$Header: /Network/Developer/Source/CVS/OmniGroup/OmniLibrarian/OWShelfView.m,v 1.4 1998/03/19 12:06:23 andrew Exp $")

@interface OWShelfView (Private)
- (NSRect)rectForSpaceAtPoint:(NSPoint)point;
- (void)_declareDragging;
- (void)_resizeContents;
@end

@implementation OWShelfView

#define CONTENTS(x,y)	(contents[(spacesAcross * y) + x])
#define SELECTED(x,y)	(selected[(spacesAcross * y) + x])

- initWithFrame:(NSRect)frameRect
{
    if (![super initWithFrame:frameRect])
        return nil;

    spaceSize.width = 60;
    spaceSize.height = 80;
    contents = NSZoneCalloc(NULL, 1, sizeof(id));
    selected = NSZoneCalloc(NULL, 1, sizeof(BOOL));
    spacesAcross = 1;
    spacesDown = 1;
    totalSpaces = spacesAcross * spacesDown;
    [self _resizeContents];
    [self _declareDragging];

    return self;
}

- (BOOL)isFlipped;
{
    return YES;
}

- (void)awakeFromNib
{
    [self _declareDragging];
}

- (void)dealloc;
{
    [draggingObject release];
    NSZoneFree(NULL, contents);
    NSZoneFree(NULL, selected);
    [super dealloc];
}

// Setup

- (void)setSpaceSize:(NSSize)size
{
    spaceSize = size;
    [self _resizeContents];
    [self setNeedsDisplay: YES];
}

- (void)setDelegate:(id <OWShelfViewDelegate>)aDelegate;
{
    delegate = aDelegate;
}

- (void)setFormatter:(id <OWShelfViewFormatter>)aFormatter;
{
    if (formatter == aFormatter)
	return;
    [formatter release];
    formatter = [aFormatter retain];
    [self setNeedsDisplay:YES];
}

- (void)setDragSupport:(id <OWShelfViewDragSupport>)aDragSupport;
{
    dragSupport = [aDragSupport retain];
    [self _declareDragging];
}

- (void)setMoveOnDrag:(BOOL)newMoveOnDrag;
{
    flags.moveOnDrag = newMoveOnDrag;
}

- (void)addEntry:(id)anEntry selected:(BOOL)isSelected atRow:(unsigned int)row andColumn:(unsigned int)column;
{
    NSSize newSize = [self bounds].size;
    BOOL resize = NO;

    if (spacesAcross <= column) {
	newSize.width = (column+1)*spaceSize.width;
	resize = YES;
    }
    if (spacesDown <= row) {
	newSize.height = (row+1)*spaceSize.height;
	resize = YES;
    }
    if (resize)
	[self setFrameSize:newSize];
#warning AFA: hey - surely we should release any entry already here?
    CONTENTS(column,row) = [anEntry retain];
    SELECTED(column,row) = isSelected;
    [self setNeedsDisplay: YES];
}

- (BOOL)moveOnDrag;
{
    return flags.moveOnDrag;
}

- (NSSize)spaceSize;
{
    return spaceSize;
}

- (NSMutableArray *)selection;
{
    unsigned int across, down;
    NSMutableArray *selection;

    selection = [NSMutableArray array];
    for (across = 0; across < spacesAcross; across++)
	for (down = 0; down < spacesDown; down++)
	    if (SELECTED(across, down))
                [selection addObject:CONTENTS(across,down)];
    return selection;
}

- (id <OWShelfViewFormatter>)formatter;
{
    return formatter;
}

- (id <OWShelfViewDelegate>)delegate;
{
    return delegate;
}

// Target-action

- (IBAction)selectAll:(id)sender;
{
    unsigned int across, down;

    for (across = 0; across < spacesAcross; across++)
	for (down = 0; down < spacesDown; down++)
	    if (CONTENTS(across, down))
		SELECTED(across, down) = YES;
    [self setNeedsDisplay:YES];
}

// Get state out for saving

- (unsigned int)rows;
{
    return spacesDown;
}

- (unsigned int)columns;
{
    return spacesAcross;
}

- (id)entryAtRow:(unsigned int)aRow column:(unsigned int)aColumn;
{
    return CONTENTS(aColumn, aRow);
}

- (BOOL)selectedAtRow:(unsigned int)aRow column:(unsigned int)aColumn;
{
    return SELECTED(aColumn, aRow);
}

// NSView subclass

- (void)setFrameSize:(NSSize)_newSize
{
    [super setFrameSize:_newSize];
    [self _resizeContents];
    [self setNeedsDisplay: YES];
}

- (void)drawRect:(NSRect)rect;
{
    NSRect position, bounds, frame;
    int across, down, startDown;
    int lastAcross, lastDown;
    id object;

    [[NSColor controlColor] set];
    NSRectFill(rect);

    bounds = [self bounds];
    frame = [self frame];

    // Call formatter to draw neccesary spaces
    across = (NSMinX(rect) - NSMinX(bounds)) / spaceSize.width;
    lastAcross = (NSMaxX(rect) + spaceSize.width - 1 - NSMinX(bounds)) / spaceSize.width;
    if (lastAcross >= (int)spacesAcross)
	lastAcross = spacesAcross - 1;

    startDown = (NSMinY(rect) - NSMinY(bounds)) / spaceSize.height;
    lastDown = (NSMaxY(rect) + spaceSize.height - 1 - NSMinY(bounds)) / spaceSize.height;
    if (lastDown >= (int)spacesDown)
	lastDown = spacesDown - 1;
    position.size = spaceSize;

    while (across <= lastAcross) {
	position.origin.x = NSMinX(frame) + (across * spaceSize.width);
	down = startDown;
	while (down <= lastDown) {
	    position.origin.y = NSMinY(frame) + (down * spaceSize.height);
	    if ((object = CONTENTS(across, down))) {
		[formatter drawEntry:object inRect:position 
		    ofShelf:self selected:SELECTED(across,down) dragging:NO];
	    } else if (draggingObject && NSPointInRect(dragPoint, position)) {
                [formatter drawEntry:draggingObject inRect:position ofShelf:self selected:NO dragging:YES];
	    }
	    down++;
	}
	across++;
    }
}

- (BOOL)acceptsFirstMouse:(NSEvent *)theEvent;
{
    return YES;
}

- (BOOL)shouldDelayWindowOrderingForEvent:(NSEvent *)anEvent;
{
    return YES;
}

- (unsigned int)draggingSourceOperationMaskForLocal:(BOOL)isLocal;
{
    return NSDragOperationAll;
}

- (BOOL)ignoreModifierKeysWhileDragging;
{
    return YES;
}

- (void)mouseDown:(NSEvent *)event;
{
    // If dragging begins, drag the thing; otherwise select or deselect it
    NSRect bounds;
    NSPoint point;
    unsigned int across, down;
    id clickedEntry;

    point = [self convertPoint:[event locationInWindow] fromView:nil];
    bounds = [self bounds];
    across = (point.x - NSMinX(bounds)) / spaceSize.width;
    down = (point.y - NSMinY(bounds)) / spaceSize.height;

    if (across >= spacesAcross || down >= spacesDown || !(clickedEntry = CONTENTS(across, down)))
	return;

    if ([event type] != NSLeftMouseDown)
        return;

    if (!([event modifierFlags] & NSShiftKeyMask)) {
        // turn off every other selection
        unsigned int a, d;

        for (a = 0; a < spacesAcross; a++)
            for (d = 0; d < spacesDown; d++)
                SELECTED(a, d) = NO;
        SELECTED(across, down) = YES;
    } else
        SELECTED(across, down) = !SELECTED(across, down);
    [delegate shelfViewSelectionChanged:self];
    [self display];

    if ([self shouldStartDragFromMouseDownEvent:event dragSlop:10 finalEvent:NULL]) {
        OAPasteboardHelper *helper;
        NSImage *dragImage;
        NSSize imageSize;
        NSPoint where;
        BOOL disappear;

        dragImage = [dragSupport dragImageForEntry:clickedEntry];
        disappear = flags.moveOnDrag && !([event modifierFlags] & NSAlternateKeyMask) && !([event modifierFlags] & NSShiftKeyMask);

        where = [self convertPoint:[event locationInWindow] fromView:nil];

        imageSize = [dragImage size];
        where.x -= imageSize.width / 2;
        where.y += imageSize.height / 2;

        dragOutObject = clickedEntry;
        helper = [OAPasteboardHelper helperWithPasteboardNamed:NSDragPboard];
        [dragSupport declareTypesForEntry:clickedEntry pasteboardHelper:helper];
        if (disappear) {
            CONTENTS(across,down) = nil;
            SELECTED(across,down) = NO;
            [delegate shelfViewChanged:self];
        }
        [dragSupport startDragOnEntry:clickedEntry fromView:self image:dragImage atPoint:where event:event pasteboardHelper:helper];
        if (!disappear)
            [dragOutObject retain];
    } else {
        if ([event clickCount] > 1)
            [delegate shelfViewDoubleClick:self];
        else
            [delegate shelfViewClick:self];
    }
}

// NSDraggingSource informal protocol

- (void)draggedImage:(NSImage *)image endedAt:(NSPoint)screenPoint deposited:(BOOL)didDeposit;
{
    [dragOutObject release];
    dragOutObject = nil;
}

// NSPasteboardOwner informal protocol

- (void)pasteboard:(NSPasteboard *)pasteboard provideDataForType:(NSString *)type
{
}

// NSDraggingDestination informal protocol

- (unsigned int)draggingEntered:(id <NSDraggingInfo>)sender;
{
    NSPasteboard *pasteboard;
    NSArray *types;
    NSString *type;
    id propertyList;

    pasteboard = [sender draggingPasteboard];
    types = [dragSupport acceptedPasteboardTypes];
    type = [pasteboard availableTypeFromArray:types];
    propertyList = [pasteboard propertyListForType:type];
    if (!propertyList)
	return NSDragOperationNone;

    draggingObject = [[dragSupport entryFromPropertyList:propertyList ofType:type] retain];

    // The following are hacks specific to OWSearchEntry.  Someday, we should make a search shelf subclass and move these hacks there.
//    [(OWSearchEntry *)draggingObject setImage:[[sender draggedImage] copy]];
//    [(OWSearchEntry *)draggingObject setView:self];

    dragPoint = [sender draggingLocation];
    dragPoint = [self convertPoint:dragPoint fromView:nil];

    return [self draggingUpdated:sender];
}

- (unsigned int)draggingUpdated:(id <NSDraggingInfo>)sender;
{
    NSRect dragRect, newRect, bothRects;
    NSPoint newPoint;
    
    newPoint = [sender draggingLocation];

    newPoint = [self convertPoint:newPoint fromView:nil];
    dragRect = [self rectForSpaceAtPoint:dragPoint];
    newRect = [self rectForSpaceAtPoint:newPoint];
    bothRects = NSUnionRect(dragRect, newRect);
    dragPoint = newPoint;

    [self displayRect:bothRects];

    return NSDragOperationGeneric;
}

- (void)draggingExited:(id <NSDraggingInfo>)sender;
{
    [draggingObject release];
    draggingObject = nil;
    [self displayRect:[self rectForSpaceAtPoint:dragPoint]];
}

- (BOOL)prepareForDragOperation:(id <NSDraggingInfo>)sender;
{
    return YES;
}

- (BOOL)performDragOperation:(id <NSDraggingInfo>)sender;
{
    NSRect bounds;
    unsigned int dragAcross, dragDown;

    bounds = [self bounds];
    dragAcross = (dragPoint.x - NSMinX(bounds)) / spaceSize.width;
    dragDown = (dragPoint.y - NSMinX(bounds)) / spaceSize.height;
    if (CONTENTS(dragAcross, dragDown))
	return NO;

    CONTENTS(dragAcross, dragDown) = draggingObject;
    SELECTED(dragAcross, dragDown) = YES;
    draggingObject = nil;
    [self displayRect:[self rectForSpaceAtPoint:dragPoint]];

    return YES;
}

- (void)concludeDragOperation:(id <NSDraggingInfo>)sender;
{
    [delegate shelfViewChanged:self];
    [delegate shelfViewSelectionChanged:self];
}

@end

@implementation OWShelfView (Private)

- (NSRect)rectForSpaceAtPoint:(NSPoint)point;
{
    unsigned int across, down;
    NSRect bounds, spaceRect;

    bounds = [self bounds];
    across = (point.x - NSMinX(bounds)) / spaceSize.width;
    down = (point.y - NSMinY(bounds)) / spaceSize.height;
    spaceRect.size = spaceSize;
    spaceRect.origin.x = (across * spaceSize.width) + NSMinX(bounds);
    spaceRect.origin.y = (down * spaceSize.height) + NSMinY(bounds);
    return spaceRect;
}

- (void)_declareDragging;
{
    NSArray *types;

    types = [dragSupport acceptedPasteboardTypes];
    if ([types count])
	[self registerForDraggedTypes:types];
}

- (void)_resizeContents;
{
    NSRect bounds;
    unsigned int newAcross, newDown;
    unsigned int newLength, index;

    bounds = [self bounds];
    newAcross = NSWidth(bounds) / spaceSize.width;
    newDown = NSHeight(bounds) / spaceSize.height;
    spacesAcross = newAcross ? newAcross : 1;
    newLength = newAcross * newDown;
    spacesDown = (newLength + spacesAcross - 1) / spacesAcross;

    if (newLength > totalSpaces) {
	contents = NSZoneRealloc(NULL, contents, newLength * sizeof(id));
	for (index = totalSpaces; index < newLength; index++)
	    contents[index] = nil;
        selected = NSZoneRealloc(NULL, selected, newLength * sizeof(BOOL));
	for (index = totalSpaces; index < newLength; index++)
	    selected[index] = NO;
	totalSpaces = newLength;
    }
}

@end
