/*
 * Name: id_translate.c
 * Description: Functions for translation of UIDs and GIDs.
 * Author: Christian Starkjohann <cs@hal.kph.tuwien.ac.at>
 * Date: 1996-11-14
 * Copyright: GNU-GPL
 * Tabsize: 4
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include "my_defines.h"

#define	DPRINTF(arg)	if(debug_mode & DEBUG_TRANS)	dprintf arg

/* ------------------------------------------------------------------------- */

typedef struct idpair{
	int	local;
	int	disk;
}idpair_t;

typedef struct translator{
	idpair_t	*idlist;
	int			allocated;
	int			used;
	int			is_fixed;
	int			fixed_value;
}translator_t;

/* ------------------------------------------------------------------------- */

translator_t	translators[2];

/* ------------------------------------------------------------------------- */

static void	add_translation(translator_t *t, int local, int disk)
{
	if(t->idlist == NULL){
		t->allocated = 30;
		t->idlist = malloc(t->allocated * sizeof(idpair_t));
	}
	if(t->used >= t->allocated){
		t->allocated *= 2;
		t->idlist = realloc(t->idlist, t->allocated * sizeof(idpair_t));
	}
	t->idlist[t->used].local = local;
	t->idlist[t->used++].disk = disk;
}

/* ------------------------------------------------------------------------- */

static inline int	to_disk(translator_t *t, int local)
{
int		i;

	for(i=0;i<t->used;i++){
		if(t->idlist[i].local == local)
			return t->idlist[i].disk;
	}
	return local;
}

/* ------------------------------------------------------------------------- */

static inline int	to_local(translator_t *t, int disk)
{
int		i;

	if(t->is_fixed)
		return t->fixed_value;
	for(i=0;i<t->used;i++){
		if(t->idlist[i].disk == disk)
			return t->idlist[i].local;
	}
	return disk;
}

/* ------------------------------------------------------------------------- */

static void	read_file(translator_t *t, char *file)
{
char	linebuf[4096], *local, *disk;
FILE	*fp;

	fp = fopen(file, "r");
	if(fp == NULL){
		eprintf("error reading table file ->%s<-: ", file);
		perror("");
		return;
	}
	while(fgets(linebuf, sizeof(linebuf), fp) != NULL){
		if(linebuf[0] == '#')
			continue;
		if((local = strtok(linebuf, " \t\n\r=")) == NULL)
			continue;
		if(!isdigit(*local) && *local != '-'){
			eprintf("->%s<-: local id (%s) must be numeric!\n", file, local);
			continue;
		}
		if((disk = strtok(NULL, " \t\n\r=")) == NULL){
			eprintf("->%s<-: no disk id for local id %s given\n", file, local);
			continue;
		}
		if(!isdigit(*disk) && *disk != '-'){
			eprintf("->%s<-: disk id (%s) must be numeric!\n", file, disk);
			continue;
		}
		add_translation(t, atoi(local), atoi(disk));
	}
	fclose(fp);
}

/* ------------------------------------------------------------------------- */

void	load_translationfile(int id, char *name)
{
	read_file(&translators[id], name);
}

/* ------------------------------------------------------------------------- */

void	set_fixed_id(int id, int value)
{
	translators[id].is_fixed = 1;
	translators[id].fixed_value = value;
}

/* ------------------------------------------------------------------------- */

int	id_is_fixed(int id)
{
	return translators[id].is_fixed;
}

/* ------------------------------------------------------------------------- */

int	fixed_id(int id)
{
	if(translators[id].is_fixed)
		return translators[id].fixed_value;
	else
		return -1;
}

/* ------------------------------------------------------------------------- */

int	translate_to_disk(int id, int local)
{
int	disk = to_disk(&translators[id], local);

	DPRINTF(("translate_to_disk(id=%d, local=%d) = %d\n", id, local, disk));
	return disk;
}

/* ------------------------------------------------------------------------- */

int	translate_to_local(int id, int disk)
{
int	local = to_local(&translators[id], disk);

	DPRINTF(("translate_to_local(id=%d, disk=%d) = %d\n", id, disk, local));
	return local;
}

/* ------------------------------------------------------------------------- */
