
#import "ManDocumentController.h"
#import <Foundation/NSPathUtilities.h>
#import <Foundation/NSScanner.h>
#import <Foundation/NSCharacterSet.h>
#import <Foundation/NSArray.h>
#import <Foundation/NSConnection.h>
#import <Foundation/NSFileHandle.h>
#import <Foundation/NSFileManager.h>
#import <AppKit/NSPasteboard.h>
#import <AppKit/NSOpenPanel.h>
#import <AppKit/NSApplication.h>
#import <AppKit/NSTextField.h>
#import "ManDocument.h"
#import "NSData+Utils.h"
#import "FindPanelController.h"
#import "PrefPanelController.h"

/*
 * The FOUNDATION_STATIC_INLINE #define appeared in Rhapsody, so if it's
 * not there we're on OPENSTEP.
 */
#ifndef FOUNDATION_STATIC_INLINE
#define OPENSTEP_ONLY
#endif

/*
 * Private NSDocument methods that we need to use.  These really should
 * have been public; it's hard to do interesting stuff with the NSDocument
 * framework without them.
 */
@interface NSDocumentController (ApplePrivate)
- (void)_addDocument:(NSDocument *)document;
- (void)_removeDocument:(NSDocument *)document;
@end

@implementation ManDocumentController

- init
{
	NSConnection *connection = [NSConnection defaultConnection];
	
	[super init];

    /*
     * Set ourselves up for DO connections.  I do it here so it's done as
     * early as possible.  If the command-line tool still has problems
     * connecting, we may be able to do this whole thing in main()...
     */
	[connection setRootObject:self];
	[connection registerName:@"ManOpenApp"];

	return self;
}

- (void)registerDefaults
{
    NSDictionary *defaults = [NSDictionary dictionaryWithObjectsAndKeys:
        @"NO",                  @"QuitWhenLastClosed",
        @"NO",                  @"UseItalics",
        @"YES",                 @"UseBold",
#ifdef OPENSTEP_ONLY
        @"tbl '%@' | nroff -man", @"NroffCommand",
        @"/usr/local/share/man:/usr/local/man:/usr/man", @"ManPath",
#else
        @"nroff -mandoc '%@'",    @"NroffCommand",
        @"/usr/local/share/man:/usr/local/man:/usr/share/man", @"ManPath",
#endif
        nil];

    [[NSUserDefaults standardUserDefaults] registerDefaults:defaults];
}

- (void)applicationDidFinishLaunching:(NSNotification *)aNotification
{
	[NSApp setServicesProvider:self];
    [self registerDefaults];
}

/*
 * By default, NSApplication will want to open an untitled document at
 * startup. We don't want to do this.
 */
- (BOOL)applicationShouldOpenUntitledFile:(NSApplication *)sender
{
    return NO;
}
- (BOOL)applicationOpenUntitledFile:(NSApplication *)sender
{
    return NO;
}


- (void)_removeDocument:(NSDocument *)document
{
    BOOL autoQuit = [[NSUserDefaults standardUserDefaults] boolForKey:@"QuitWhenLastClosed"];

    [super _removeDocument:document];

    if ([[self documents] count] == 0 && autoQuit)
    {
        [[NSApplication sharedApplication] performSelector:@selector(terminate:)
                                                withObject:self afterDelay:0.0];
    }
}

- (NSString *)manPath
{
    return [[NSUserDefaults standardUserDefaults] stringForKey:@"ManPath"];
}

- (NSString *)typeFromFilename:(NSString *)filename
{
    NSFileHandle  *handle;
    NSFileManager *manager = [NSFileManager defaultManager];
    NSDictionary  *attributes = [manager fileAttributesAtPath:filename traverseLink:YES];
    unsigned      maxLength = MIN(50, (unsigned)[attributes fileSize]);
    NSData        *fileHeader;

    if (maxLength == 0) return nil;

    handle = [NSFileHandle fileHandleForReadingAtPath:filename];
    fileHeader = [handle readDataOfLength:maxLength];

	return [fileHeader isNroffData]? @"man" : @"cat";
}

/*" Override to always guess the type ourselves. "*/
- (id)makeDocumentWithContentsOfFile:(NSString *)fileName ofType:(NSString *)type
{
	NSString *overriddenType = [self typeFromFilename:fileName];
	return [super makeDocumentWithContentsOfFile:fileName ofType:overriddenType];
}

/*
 * NSDocument[Controller] doesn't standardize filenames, at least when
 * calling -initWithContentsOfFile:, meaning it will raise if it gets a
 * symlink rather then a regular file.  Therefore, we override to
 * standardize the path, so that symlinks are resolved and removed from the
 * equation.  This will also make document lookup always find the file if
 * it's open.
 */
- (id)openDocumentWithContentsOfFile:(NSString *)filename display:(BOOL)display
{
	filename = [filename stringByStandardizingPath];
	return [super openDocumentWithContentsOfFile:filename display:display];
}

/* Ignore the types; man/cat files can have any range of extensions. */
- (int)runModalOpenPanel:(NSOpenPanel *)openPanel forTypes:(NSArray *)openableFileExtensions
{
    return [openPanel runModal];
}

/*" A parallel for openDocumentWithContentsOfFile: for a specific man page "*/
- (id)openDocumentWithName:(NSString *)name
		section:(NSString *)section
		manPath:(NSString *)manPath
{
	ManDocument *document;
	NSString	*title = name;

	if (section && [section length] > 0)
	{
		title = [NSString stringWithFormat:@"%@(%@)",name, section];
	}
	
	if ((document = [self documentForFileName:title]) == nil)
	{	
		document = [[[ManDocument alloc]
						initWithName:name section:section
						manPath:manPath title:title] autorelease];
		[self _addDocument:document];
		[document makeWindowControllers];
	}
	
	[document showWindows];

	return document;
}


/*"
 * Parses word for stuff like "file(3)" to break out the section, then
 * calls openDocumentWithName:section:manPath: as appropriate.
"*/
- (id)openWord:(NSString *)word
{
	NSString *section = nil;
	NSString *base    = word;
	NSRange lparenRange = [word rangeOfString:@"("];
	NSRange rparenRange = [word rangeOfString:@")"];

	if (lparenRange.length != 0 && rparenRange.length != 0 &&
	      lparenRange.location < rparenRange.location)
    {
        NSRange sectionRange;

		sectionRange.location = NSMaxRange(lparenRange);
		sectionRange.length = rparenRange.location - sectionRange.location;

		base = [word substringToIndex:lparenRange.location];
		section = [word substringWithRange:sectionRange];
    }

	return [self openDocumentWithName:base section:section manPath:[self manPath]];
}

/*"
 * Breaks string up into words and calls -#openWord: on each one.
 * Essentially opens a man page for each word in string.
"*/
- (void)openString:(NSString *)string
{
	NSScanner		*scanner = [NSScanner scannerWithString:string];
	NSCharacterSet	*whitespaceSet = [NSCharacterSet whitespaceAndNewlineCharacterSet];
	NSCharacterSet	*nonwhitespaceSet = [whitespaceSet invertedSet];
	NSString		*aWord;

	[scanner setCharactersToBeSkipped:whitespaceSet];
	
	while (![scanner isAtEnd])
	{
		if ([scanner scanCharactersFromSet:nonwhitespaceSet intoString:&aWord])
		{
			[self openWord:aWord];
		}
	}
}

- (IBAction)openTextPanel:(id)sender
{
	[openTextField selectText:self];
    if ([NSApp runModalForWindow:openTextPanel] == NSOKButton)
    {
        [self openString:[openTextField stringValue]];
    }
}

- (IBAction)okText:(id)sender
{
	[openTextPanel orderOut:self];
    [NSApp stopModalWithCode:NSOKButton];
}

- (IBAction)cancelText:(id)sender
{
	[openTextPanel orderOut:self];
    [NSApp stopModalWithCode:NSCancelButton];
}

/*" A simple API method to open a file "*/
- (void)openFile:(NSString *)filename
{
	[self openDocumentWithContentsOfFile:filename display:YES];
}

/*" Simple API methods to open a named man page "*/

- (void)openName:(NSString *)name section:(NSString *)section manPath:(NSString *)manPath
{
	[self openDocumentWithName:name section:section manPath:manPath];
}

- (void)openName:(NSString *)name section:(NSString *)section
{
	[self openDocumentWithName:name section:section manPath:[self manPath]];
}

- (void)openName:(NSString *)name
{
	[self openDocumentWithName:name section:nil manPath:[self manPath]];
}

/*" Methods to do the services entries "*/
- (void)openFiles:(NSPasteboard *)pboard userData:(NSString *)data error:(NSString **)error
{
    NSArray *fileArray;
    NSArray *types = [pboard types];

    if ([types containsObject:NSFilenamesPboardType] &&
		(fileArray = [pboard propertyListForType:NSFilenamesPboardType]))
	{
		int i, count = [fileArray count];

		for (i=0; i<count; i++)
        {
            [self openDocumentWithContentsOfFile:[fileArray objectAtIndex:i] display:YES];
        }
    }
}

- (void)openSelection:(NSPasteboard *)pboard userData:(NSString *)data error:(NSString **)error
{
    NSString *pboardString;
    NSArray *types = [pboard types];

    if ([types containsObject:NSStringPboardType] &&
		(pboardString = [pboard stringForType:NSStringPboardType]))
	{
        [self openString:pboardString];
    }
}

- (IBAction)orderFrontFindPanel:(id)sender
{
    [[FindPanelController sharedInstance] showWindow:sender];
}

- (IBAction)orderFrontPreferencesPanel:(id)sender
{
    [[PrefPanelController sharedInstance] showWindow:sender];
}

@end

