	/*
		cat2rtf.l: cat to RTF converter specification
		(c) 1993 by Harald Schlangmann
		Permission is granted to use this code. Send additions
		and bug reports to my address below.
		
		v1.0	Harald Schlangmann, July 20 1993
			schlangm@informatik.uni-muenchen.de
		v1.1	Bold style x^H{x^H}* implemented.
        
        v2.0  Added blank line suppressing.  -Carl Lindberg
	 */

	#define yywrap() (1)

	#define BOLDFLAG	1
	#define ULINEFLAG	2

	int flags = 0, neededflags = 0;
	
	#define SETB	neededflags |= BOLDFLAG
	#define UNSETB	neededflags &= ~BOLDFLAG
	#define SETU	neededflags |= ULINEFLAG
	#define UNSETU	neededflags &= ~ULINEFLAG
	
	/*
	 *	Default settings, may be changed using options...
	 */
	
	char	*startBold = "\n\\b ";
	char	*stopBold = "\n\\b0 ";
	char	*startULine = "\n\\ul ";
	char	*stopULine = "\n\\ulnone ";
	
	void emitPreamble(void) {
	
		fputs("{\\rtf0\\ansi{\\fonttbl\\f0\\fmodern Courier;\\f1"
		"\\fswiss Helvetica;}\n" "\\paperw14640\n" "\\paperh9600\n"
		"\\margl120\n" "\\margr120\n"
		"{\\colortbl;\\red0\\green0\\blue0;}\n"
		"\\pard\\tx1140\\tx2300\\tx3440\\tx4600"
		"\\tx5760\\tx6900\\tx8060\\tx9200"
		"\\tx10360\\tx11520\\f0\\b0\\i0\\ulnone"
		"\\fs24\\fc1\\cf1 ",stdout);
	}
	
	void emitPostamble(void) {
	
		fputs("\n}\n",stdout);
	}

	#define adjust() if( neededflags!=flags ) _adjust()
	
	void _adjust(void) {
	
		if( (flags^neededflags)&ULINEFLAG )
			fputs(neededflags&ULINEFLAG?
				startULine:stopULine,stdout);
		if( (flags^neededflags)&BOLDFLAG )
			fputs(neededflags&BOLDFLAG?
				startBold:stopBold,stdout);
		flags = neededflags;
	}
	
    /*
     * 'lineCount' is here to squeeze multiple-blank lines (like 'more -s'
     * does with the traditional nroff output).  By default, a max of 3
     * consecutive blank lines are allowed. A value of -1 means to not
	 * squeeze lines at all.  -CEL
     */
	static int lineCount = 0;
	static int maxLineCount = 3;

	/*
	 *	emitChar escapes RTF-special characters...
	 */
	void emitChar(int ch) {		
		adjust();

		if (ch=='\n') {
			if (maxLineCount > 0 && lineCount > maxLineCount)
			{
				// Squeeze the blank line -- don't output it.
				return;
			}
			lineCount++;
		}
		else lineCount = 0;

		if( ch=='\n'||ch=='{'||ch=='}'||ch=='\\' )
			fputc('\\',stdout);
		fputc(ch,stdout);
	}


	/*
	 *	emitString doesn't!
	 */
	void emitString(char *string) {
		adjust();
		lineCount = 0;
		fputs(string,stdout);
	}

ALLBUTUL	[^\n_\010]
NEEDQUOTE	[\\{}]
VERBATIM	[^\n_\010\\{}]
UPPER		[A-Z]
UPPERCONT	[-/A-Z0-9 \t()]

%x FIRSTLINE

%%

	/*
	 *	Start state FIRSTLINE is used to treat the first non-empty
	 *	line special. (First line contains header).
	 */

<FIRSTLINE>.		SETB; emitChar(yytext[0]);

<FIRSTLINE>.\n		{
				SETB;
				emitChar(yytext[0]);
				emitChar('\n');
				BEGIN(INITIAL);
				UNSETB;
			}

<FIRSTLINE>\n		UNSETB; emitChar('\n');

	/*
	 *	Non-empty, all-uppercase lines are treated as
	 *	headers
	 */

^{UPPER}{UPPERCONT}*$	{
				SETB;
				emitString(yytext);
				emitString("\\\n");
				UNSETB;
			}
	
	/*
	 *	nroff +-
	 */

"+"\010_		emitChar('\321');

	/*
	 *	underline (part 1)
	 */

{ALLBUTUL}\010_		{
				SETU;
				emitChar(yytext[0]);
				UNSETU;
			}

	/*
	 *	nroff bullet
	 */

o\010"+"		emitChar('\267');

	/*
	 *	underline (part 2)
	 */

_\010{ALLBUTUL}		{
				SETU;
				emitChar(yytext[2]);
				UNSETU;
			}

	/*
	 *	handle further BS combinations
	 */

.\010.			{
				if( yytext[0]==yytext[2] ) {
					SETB;
					emitChar(yytext[0]);
					UNSETB;
				} else
					emitChar(yytext[2]);
			}

	/*
		group characters in VERBATIM to make this
		filter faster...
	 */

{VERBATIM}+/[^\010]	emitString(yytext);

	/*
		remaining specials
	 */

\n			emitChar('\n');
.			emitChar(yytext[0]);

%%

void usage() {

	fprintf(stderr,"Usage: cat2rtf [-g] [-i] [-s <num>| -S] [<filename>]\n"
		"\tTranslate output of (g)nroff to RTF. If no\n"
		"\t<filename> is given, cat2rtf reads stdin.\n"
		"\tOption -g uses gray for bold characters,\n"
		"\toption -i uses italic characters for underlining.\n"
		"\toption -s will allow only <num> consecutive blank lines,\n"
		"\toption -S will not do any squeezing of blank lines.\n"
		"\tRTF output is sent to stdout.\n");
	exit(1); 
}

extern int getopt (int argc, char **argv, char *optstring);
extern char *optarg;
extern int optind;

int main(int argc, char *argv[])
{
	int c;

	while ((c = getopt(argc, argv, "gGiISs:")) != EOF)
	{
		switch( c ) {
			case 'g':
			case 'G':
				startBold = "\n\\gray333 ";
				stopBold = "\n\\gray0 ";
				break;
			case 'i':
			case 'I':
				startULine = "\n\\i ";
				stopULine = "\n\\i0 ";
				break;
			case 's':
				maxLineCount = atoi(optarg);
				break;
			case 'S':
				maxLineCount = -1;
				break;
			case '?':
			default:
				usage();
		}
	}

	if( optind < argc )
		yyin = fopen(argv[optind], "r");
	else
		yyin = stdin;
	
	emitPreamble();
	BEGIN(FIRSTLINE);
	yylex();
	emitPostamble();
	
	/* Shuts up a compiler warning */
	if (0) unput('h');

	return 0;
}
