
#import <Foundation/NSPathUtilities.h>
#import <Foundation/NSAutoreleasePool.h>
#import <Foundation/NSUtilities.h>
#import <Foundation/NSConnection.h>
#import <AppKit/NSWorkspace.h>
#import <libc.h>  // for getopt()
#import "ManOpenProtocol.h"

/*
 * The FOUNDATION_STATIC_INLINE #define appeared in Rhapsody, so if it's
 * not there we're on OPENSTEP.
 */
#ifndef FOUNDATION_STATIC_INLINE
#define OPENSTEP_ONLY
#endif

NSString *MakeAbsolutePath(const char *filename)
{
    NSString *currFile = [NSString stringWithCString:filename];

    if (![currFile isAbsolutePath])
    {
        currFile = [[[NSFileManager defaultManager] currentDirectoryPath]
                     stringByAppendingPathComponent:currFile];
    }

    return currFile;
}

void usage(const char *progname)
{
#ifdef OPENSTEP_ONLY
    fprintf(stderr,"%s: [-M path] [-f file] [section] [title ...]\n", progname);
#else
    fprintf(stderr,"%s: [-M path] [-m path] [-f file] [section] [name ...]\n", progname);
#endif
}

void main (int argc, const char *argv[])
{
    NSAutoreleasePool   *pool = [[NSAutoreleasePool alloc] init];
    NSWorkspace         *workspace;
    NSString            *manPath = nil;
    NSString            *section = nil;
    NSMutableArray      *files = [NSMutableArray array];
    int                 i;
    char                c;
    NSDistantObject <ManOpen>  *server;

    while ((c = getopt(argc,argv,"hm:M:f:aCcw")) != EOF)
    {
        switch(c) {
            case 'm':
            case 'M':
                manPath = [NSString stringWithCString:optarg];
                break;
            case 'f':
                [files addObject:MakeAbsolutePath(optarg)];
                break;
            case 'a':
            case 'C':
            case 'c':
            case 'w':
                // MacOS X man(1) options; no-op here.
                break;
            case 'h':
            case '?':
            default:
                usage(argv[0]);
                [pool release];
                exit(0);
        }
    }

    if (optind >= argc && [files count] <= 0)
    {
        usage(argv[0]);
        [pool release];
        exit(0);
    }

    if (optind < argc)
    {
        NSString *tmp = [NSString stringWithCString:argv[optind]];

        if (isdigit(argv[optind][0])          ||
#ifdef OPENSTEP_ONLY
            [tmp isEqualToString:@"public"]   ||
            [tmp isEqualToString:@"new"]      ||
            [tmp isEqualToString:@"old"]      ||
#else
      /* These are configurable in /etc/man.conf; these are just the default strings. */
            [tmp isEqualToString:@"system"]   ||
            [tmp isEqualToString:@"commands"] ||
            [tmp isEqualToString:@"syscalls"] ||
            [tmp isEqualToString:@"libc"]     ||
            [tmp isEqualToString:@"special"]  ||
            [tmp isEqualToString:@"files"]    ||
            [tmp isEqualToString:@"games"]    ||
            [tmp isEqualToString:@"miscellaneous"] ||
            [tmp isEqualToString:@"misc"]     ||
            [tmp isEqualToString:@"admin"]    ||
#endif
            [tmp isEqualToString:@"local"])
        {
            section = tmp;
            optind++;
        }
    }

    if (optind >= argc)
    {
        if ([section length] > 0)
        {
#ifdef OPENSTEP_ONLY
            fprintf(stderr, "But what do you want from section %s?\n", [section cString]);
#else
            /* MacOS X assumes it's a man page name */
            section = nil;
            optind--;
#endif
        }

        if (optind >= argc && [files count] <= 0)
        {
            [pool release];
            exit(0);
        }
    }

    workspace = [NSWorkspace sharedWorkspace];

    if (![workspace launchApplication:@"ManOpen"] ||
        (server = [NSConnection rootProxyForConnectionWithRegisteredName:@"ManOpenApp" host:nil]) == nil)
    {
        fprintf(stderr,"Could not open connection to ManOpen\n");
        [pool release];
        exit(1);
    }

    [server setProtocolForProxy:@protocol(ManOpen)];

    for (i=0; i<[files count]; i++)
    {
        [server openFile:[files objectAtIndex:i]];
    }

    if (manPath == nil && getenv("MANPATH") != NULL)
        manPath = [NSString stringWithCString:getenv("MANPATH")];

    for (i = optind; i < argc; i++)
    {
        NSString *currFile = [NSString stringWithCString:argv[i]];
        [server openName:currFile section:section manPath:manPath];
    }

    [pool release];
    exit(0);       // ensure the process exit status is 0
}
