// NSTextView_TETextExtras.m
// TextExtras - Yellow Box
//
// Copyright 1996-1999, Mike Ferris.
// All rights reserved.

#import "NSTextView_TETextExtras.h"
#import "TEGotoPanelController.h"
#import "TEPipePanelController.h"
#import "TEPipeCommand.h"
#import "TEPreferencesController.h"
#import "TETextUtils.h"
#import "TETextWatcher.h"
#import "TESpecialCharactersController.h"
#import "TEOpenQuicklyController.h"
#import <MOKit/MOKit.h>

@interface NSLayoutManager (TEPrivatesOnParade)

// MF: Shhh... don't tell anyone about these private methods.  If they stop working, tough luck.
- (NSString *)_containerDescription;
- (NSString *)_lineFragmentDescription:(BOOL)verboseFlag;
- (NSString *)_glyphDescription;

@end

@implementation NSTextView (TETextExtras)

// MF: The ugliness of most of the method names in this category are because the base calass and all categories share a namespace for their methods.  All the methods in this category that are prefixed with TE_ are done that way to avoid conflicting with another method in another category (I mean, who else would have such ugly method names?)  There are exceptions.  Methods without TE_ prefixes are inherently unwise in a loadable bundle situation like this.  Where there are exceptions the overriding reasons for them are explained.

// ********************** Nest/Unnest feature **********************

- (void)TE_doUserIndentByNumberOfLevels:(int)levels {
    // Because of the way paragraph ranges work we will add spaces a final paragraph separator only if the selection is an insertion point at the end of the text.
    // We ask for rangeForUserTextChange and extend it to paragraph boundaries instead of asking rangeForUserParagraphAttributeChange because this is not an attribute change and we don't want it to be affected by the usesRuler setting.
    NSRange charRange = [[self string] lineRangeForRange:[self rangeForUserTextChange]];
    NSRange selRange = [self selectedRange];
    if (charRange.location != NSNotFound) {
        NSTextStorage *textStorage = [self textStorage];
        NSAttributedString *newText;
        unsigned tabWidth = [[TEPreferencesController sharedPreferencesController] tabWidth];
        unsigned indentWidth = [[TEPreferencesController sharedPreferencesController] indentWidth];

        selRange.location -= charRange.location;
        newText = TE_attributedStringByIndentingParagraphs([textStorage attributedSubstringFromRange:charRange], levels,  &selRange, [self typingAttributes], tabWidth, indentWidth);
        selRange.location += charRange.location;
        if ([self shouldChangeTextInRange:charRange replacementString:[newText string]]) {
            [textStorage replaceCharactersInRange:charRange withAttributedString:newText];
            [self setSelectedRange:selRange];
            [self didChangeText];
        }
    }
}

- (void)TE_indentRight:(id)sender {
    [self TE_doUserIndentByNumberOfLevels:1];
}

- (void)TE_indentLeft:(id)sender {
    [self TE_doUserIndentByNumberOfLevels:-1];
}

// ********************** Auto-indent feature **********************

static void autoIndent(NSTextView *tv, NSString *lineEnding) {
    if ([tv isFieldEditor]) {
        // Field editors needs to do something special with newlines, and we don't know how.  Let insertNewline handle it, because it does know how.
        if (lineEnding) {
            [tv insertNewline:tv];
        }
    } else {
        NSRange charRange = [tv rangeForUserTextChange];
        if (charRange.location != NSNotFound) {
            NSString *insertString = (lineEnding ? lineEnding : @"");
            NSString *string = [tv string];
            if (charRange.location > 0) {
                if (!lineEnding) {
                    // We were called from TE_insertNewlineAndIndent: and the newline has already been inserted.  Back up by one char.
                    charRange.location--;
                }
                if ((charRange.location > 0) && !TE_IsHardLineBreakUnichar([string characterAtIndex:(charRange.location - 1)], string, charRange.location - 1)) {
                    unsigned tabWidth = [[TEPreferencesController sharedPreferencesController] tabWidth];
                    NSRange paraRange = [string lineRangeForRange:NSMakeRange(charRange.location - 1, 1)];
                    unsigned leadingSpaces = TE_numberOfLeadingSpacesFromRangeInString(string, &paraRange, tabWidth);
                    insertString = [insertString stringByAppendingString:TE_tabbifiedStringWithNumberOfSpaces(leadingSpaces, tabWidth)];
                }
            }
            [tv insertText:insertString];
        }
    }
}

- (void)TE_insertNewlineAndIndent:(id)sender {
    NSRange selRange;
    // Some text system clients do special things when they see insertNewline:.  In order to preserve this behavior when we can, the TE_insertNewlineAndIndent: method actually calls insertNewline: and then does the indenting separately.
    [self insertNewline:sender];
    // Some apps may do their own auto-indenting (hi PB) or do other funky things.  We will check to see that the thing right before the insertion point is a newline, and if it's not we assume something we don't understand happened, and we won't auto-indent.
    selRange = [self selectedRange];
    if ((selRange.location > 0) && ([[self string] characterAtIndex:selRange.location-1] == NSNewlineCharacter)) {
        autoIndent(self, nil);
    }
}

- (void)TE_insertCRLFAndIndent:(id)sender {
    autoIndent(self, @"\r\n");
}

- (void)TE_insertCRAndIndent:(id)sender {
    autoIndent(self, @"\r");
}

- (void)TE_insertParagraphSeparatorAndIndent:(id)sender {
    unichar paragraphSeparator[1];

    paragraphSeparator[0] = NSParagraphSeparatorCharacter;

    autoIndent(self, [NSString stringWithCharacters:paragraphSeparator length:1]);
}

- (void)TE_insertLineSeparatorAndIndent:(id)sender {
    unichar lineSeparator[1];

    lineSeparator[0] = NSLineSeparatorCharacter;

    autoIndent(self, [NSString stringWithCharacters:lineSeparator length:1]);
}

// ********************** Panel actions **********************

- (void)TE_gotoPanel:(id)sender {
    [[TEGotoPanelController sharedGotoPanelController] showWindow:self];
    [[TEGotoPanelController sharedGotoPanelController] selectRangeText];
    return;
}

- (void)TE_preferencesPanel:(id)sender {
    [[TEPreferencesController sharedPreferencesController] showWindow:self];
}

- (void)TE_specialCharactersPanel:(id)sender {
    [[TESpecialCharactersController sharedSpecialCharactersController] showWindow:self];
}

- (void)TE_openQuickly:(id)sender {
    [[TEOpenQuicklyController sharedOpenQuicklyController] runOpenQuicklyPanel:self];
}

- (void)TE_executePipe:(id)sender {
    [[TEPipePanelController sharedPipePanelController] runPipePanelWithTextView:self];
}

- (void)TE_executeUserPipe:(id)sender {
    if ([sender respondsToSelector:@selector(representedObject)]) {
        [[TEPipePanelController sharedPipePanelController] runUserPipe:[sender representedObject] withTextView:self];
    }
}

// ********************** Escape-completion support **********************

- (NSString *)TE_completionStringForSearchStringFoundRange:(NSRange)foundRange inString:(NSString *)string nonCompletableCharSet:(NSCharacterSet *)nonCompletableSet {
    // foundRange is the range of the found searchString in string.
    // Find the end of the completion text.
    NSString *completionString = nil;
    unsigned completionIndex = NSMaxRange(foundRange);
    unsigned stringLength = [string length];
    
    foundRange = [string rangeOfCharacterFromSet:nonCompletableSet options:0 range:NSMakeRange(completionIndex, stringLength - completionIndex)];
    if (foundRange.length > 0) {
        if (foundRange.location > completionIndex) {
            completionString = [string substringWithRange:NSMakeRange(completionIndex, foundRange.location - completionIndex)];
        }
    } else {
        if (stringLength > completionIndex) {
            completionString = [string substringWithRange:NSMakeRange(completionIndex, stringLength - completionIndex)];
        }
    }
    return completionString;
}

- (void)complete:(id)sender {
    // MF: This method has no TE_ prefix.  This is a somewhat dangerous thing to add in a category.  Anybody who would do such a thing deserves what they get (ie their implementation might not win out in a conflict.)  The reason here is that we really want to be the default complete: binding.  I know that NSTextView in the current release does not implement this method.  If other bundles or applications have category overrides we might conflict and either override or be overridden.  This might not be good either way in different situations and there is no guaranteed way to know who wins.  Subclasses, on the other hand, always win.  They may call super and not get the implementation they expect, but they probably don't since there is no superclass implementation in the base class.  So, since this is a self-contained feature, I chose to implement this standard method in this category.
    static NSMutableCharacterSet *completableSet = nil;
    static NSCharacterSet *nonCompletableSet = nil;
    
    if ([self isEditable]) {
        NSString *string = [self string];
        unsigned stringLength = [string length];
        NSRange selectedRange = [self selectedRange];
        unsigned completionStartLoc = [TETextWatcher escapeCompletionStartLocation];
        NSRange searchRange, foundRange;
        NSString *searchString;
        BOOL searchBackward = YES;
        NSString *completionString = nil;
        BOOL cycling = [TETextWatcher isCyclingEscapeCompletions];
        
        // Set up the statics
        if (!completableSet) {
            completableSet = [[[NSCharacterSet alphanumericCharacterSet] mutableCopy] autorelease];
            [completableSet addCharactersInString:@"_@"];
            nonCompletableSet = [[completableSet invertedSet] retain];
        }

        // MF: We really don't want to search the document if we've already found all the possible completions.
        if (!cycling) {
            NSRange searchStringRange;
            
            // Find the search string.
            if (completionStartLoc == NSNotFound) {
                searchRange = NSMakeRange(0, selectedRange.location);
            } else {
                searchRange = NSMakeRange(0, completionStartLoc);
            }
            if (searchRange.length == 0) {
                NSBeep();
                return;
            }
            foundRange = [string rangeOfCharacterFromSet:nonCompletableSet options:NSBackwardsSearch range:searchRange];
            if (foundRange.length > 0) {
                searchStringRange = NSMakeRange(NSMaxRange(foundRange), NSMaxRange(searchRange) - NSMaxRange(foundRange));
                searchString = [string substringWithRange:searchStringRange];
                // Set up for backward search
                searchRange = NSMakeRange(0, foundRange.location);
            } else {
                searchStringRange = NSMakeRange(0, NSMaxRange(searchRange));
                searchString = [string substringWithRange:searchStringRange];
                // Set up for forward search
                searchRange = NSMakeRange(NSMaxRange(selectedRange), stringLength - NSMaxRange(selectedRange));
                searchBackward = NO;
            }

            {
                // Try the completion dictionary.
                NSDictionary *completionDict = [[TEPreferencesController sharedPreferencesController] completionDictionary];
                NSString *replaceString = [completionDict objectForKey:searchString];
                if (replaceString) {
                    [self setSelectedRange:searchStringRange];
                    // Just insert it.  Since we call a user-editing type method, it will take care of notification, delegations, and ensuring correct display.
                    [self insertText:replaceString];
                    return;
                }
            }

            // Do the backward search, if necessary.
            if (searchBackward) {
                // searchRange is the front part of the string.  We are searching for the searchString backwards through this range.
                while (searchRange.length > 0) {
                    foundRange = [string rangeOfString:searchString options:NSBackwardsSearch range:searchRange];
                    if (foundRange.length > 0) {
                        // We found it, but is it the beginning of a word?  If it isn't we'll keep looking.
                        searchRange = NSMakeRange(0, foundRange.location);
                        if ((searchRange.length == 0) || [string rangeOfCharacterFromSet:nonCompletableSet options:(NSBackwardsSearch | NSAnchoredSearch) range:searchRange].length > 0) {
                            completionString = [self TE_completionStringForSearchStringFoundRange:foundRange inString:string nonCompletableCharSet:nonCompletableSet];
                            if ([TETextWatcher alreadyFoundEscapeCompletion:completionString]) {
                                completionString = nil;
                            } else {
                                break;
                            }
                        }
                    } else {
                        // Set up for forward search
                        searchRange = NSMakeRange(NSMaxRange(selectedRange), stringLength - NSMaxRange(selectedRange));
                        break;
                    }
                }
                if (searchRange.length == 0) {
                    // Set up for forward search
                    searchRange = NSMakeRange(NSMaxRange(selectedRange), stringLength - NSMaxRange(selectedRange));
                }
            }

            // Do the forward search, if necessary.
            if (!completionString) {
                // searchRange is the back part of the string.  We are searching for the searchString forwards through this range.
                while (searchRange.length > 0) {
                    foundRange = [string rangeOfString:searchString options:0 range:searchRange];
                    if (foundRange.length > 0) {
                        // We found it, but is it the beginning of a word?  If it isn't we'll keep looking.
                        searchRange = NSMakeRange(NSMaxRange(foundRange), stringLength - NSMaxRange(foundRange));
                        if ([string rangeOfCharacterFromSet:nonCompletableSet options:NSAnchoredSearch range:NSMakeRange(foundRange.location - 1, 1)].length > 0) {
                            completionString = [self TE_completionStringForSearchStringFoundRange:foundRange inString:string nonCompletableCharSet:nonCompletableSet];
                            if ([TETextWatcher alreadyFoundEscapeCompletion:completionString]) {
                                completionString = nil;
                            } else {
                                break;
                            }
                        }
                    } else {
                        break;
                    }
                }
            }
        }

        if (!completionString && (completionStartLoc != NSNotFound)) {
            // We didn't find a new completion, but there were completions, so cycle back through the existing ones.
            NSString *oldCompletionString = [string substringWithRange:NSMakeRange(completionStartLoc, selectedRange.location - completionStartLoc)];
            completionString = [TETextWatcher escapeCompletionAfterOldCompletion:oldCompletionString];
            cycling = YES;
        }
        
        // If we found a completion, do it.
        if (completionString) {
            if (completionStartLoc != NSNotFound) {
                [self setSelectedRange:NSMakeRange(completionStartLoc, selectedRange.location - completionStartLoc)];
            }
            // Just insert it.  Since we call a user-editing type method, it will take care of notification, delegations, and ensuring correct display.
            [self insertText:completionString];
            // Tell the TETextWatcher (who is storing some state for us) the start location for the completion and add this completion.
            // MF: The TETextWatcher keeps track of the current set of previously found completions and where in the text the completions string starts.  It invalidates this information whenever the first responder changes or the selection in the text view changes.
            // MF: Since the selection just changed (as a result of insertText: above), the current escape completion state has been marked for invalidation.  We can rescue the old state though, by setting the completion start location.  We do this and then we add the completion to the list if we aren't just cycling.
            [TETextWatcher setEscapeCompletionStartLocation:((completionStartLoc != NSNotFound) ? completionStartLoc : selectedRange.location)];
            if (!cycling) {
                [TETextWatcher addFoundEscapeCompletion:completionString];
            }
        } else {
            NSBeep();
        }
    }
}

// ********************** Text Debugging utilities **********************

- (void)TE_logTextViewDescriptions:(id)sender {
    NSArray *textContainers = [[self layoutManager] textContainers];
    unsigned i, c = [textContainers count];
    NSMutableString *desc = [NSMutableString string];

    [desc appendFormat:@"%u TextViews.\n", c];
    for (i=0; i<c; i++) {
        [desc appendFormat:@"TextView %u: %@", i, [[textContainers objectAtIndex:i] textView]];
    }
    NSLog(@"%@", desc);
}

- (void)TE_logTextContainerDescriptions:(id)sender {
    NSArray *textContainers = [[self layoutManager] textContainers];
    unsigned i, c = [textContainers count];
    NSMutableString *desc = [NSMutableString string];

    [desc appendFormat:@"%u TextContainers.\n", c];
    for (i=0; i<c; i++) {
        [desc appendFormat:@"TextContainer %u: %@", i, [textContainers objectAtIndex:i]];
    }
    NSLog(@"%@", desc);
}

- (void)TE_logLayoutManagerDescription:(id)sender {
    NSLog(@"%@", [[self layoutManager] description]);
}

- (void)TE_logLayoutManagerContainerDescription:(id)sender {
    // MF: Shhh... don't tell anyone about this private method.
    NSLog(@"%@", [[self layoutManager] _containerDescription]);
}

- (void)TE_logLayoutManagerLineFragmentDescription:(id)sender {
    // MF: Shhh... don't tell anyone about this private method.
    NSLog(@"%@", [[self layoutManager] _lineFragmentDescription:NO]);
}

- (void)TE_logLayoutManagerVerboseLineFragmentDescription:(id)sender {
    // MF: Shhh... don't tell anyone about this private method.
    NSLog(@"%@", [[self layoutManager] _lineFragmentDescription:YES]);
}

- (void)TE_logLayoutManagerGlyphDescription:(id)sender {
    // MF: Shhh... don't tell anyone about this private method.
    NSLog(@"%@", [[self layoutManager] _glyphDescription]);
}

- (void)TE_logTextStorageDescription:(id)sender {
    NSLog(@"%@", [[self textStorage] description]);
}

// ********************** Toggle shows control characters **********************

- (void)TE_toggleShowsControlCharacters:(id)sender {
    NSLayoutManager *lm = [self layoutManager];
    BOOL showsControlChars = [lm showsControlCharacters];
    [lm setShowsControlCharacters:(showsControlChars ? NO : YES)];
}

- (void)TE_parseAsPropertyList:(id)sender {
    NSString *str = [self string];
    volatile NSString *exceptionDescription = nil;
    if ([str length] > 0) {
        NS_DURING
            [str propertyList];
        NS_HANDLER
            exceptionDescription = [[localException description] retain];
        NS_ENDHANDLER

        if (exceptionDescription) {
            NSRunAlertPanel(NSLocalizedStringFromTableInBundle(@"Property List Error", @"TextExtras", [NSBundle bundleForClass:[self class]], @"Title for property list parse failure alerts"), NSLocalizedStringFromTableInBundle(@"An error occured parsing the current text as a property list: %@.", @"TextExtras", [NSBundle bundleForClass:[self class]], @"Error string"), NSLocalizedStringFromTableInBundle(@"OK", @"TextExtras", [NSBundle bundleForClass:[self class]], @"OK button title."), nil, nil, exceptionDescription);
        } else {
            NSRunAlertPanel(NSLocalizedStringFromTableInBundle(@"Property List Parsed", @"TextExtras", [NSBundle bundleForClass:[self class]], @"Title for property list parse success alerts"), NSLocalizedStringFromTableInBundle(@"The current text is a valid property list.", @"TextExtras", [NSBundle bundleForClass:[self class]], @"Success message"), NSLocalizedStringFromTableInBundle(@"OK", @"TextExtras", [NSBundle bundleForClass:[self class]], @"OK button title."), nil, nil);
        }
    }
}

- (void)TE_standardizeEndOfLineToLF:(id)sender {
    NSTextStorage *textStorage = [self textStorage];
    NSMutableString *str = [textStorage mutableString];
    NSRange charRange = NSMakeRange(0, [str length]);
    
    if ([self shouldChangeTextInRange:charRange replacementString:nil]) {
        // -MO_standardizeEndOfLineToLF might do many separate edits, so turn on the batching in textStorage here.
        [textStorage beginEditing];
        [str MO_standardizeEndOfLineToLF];
        [textStorage endEditing];
        [self didChangeText];
    }
}

- (void)TE_standardizeEndOfLineToCRLF:(id)sender {
    NSTextStorage *textStorage = [self textStorage];
    NSMutableString *str = [textStorage mutableString];
    NSRange charRange = NSMakeRange(0, [str length]);

    if ([self shouldChangeTextInRange:charRange replacementString:nil]) {
        // -MO_standardizeEndOfLineToCRLF might do many separate edits, so turn on the batching in textStorage here.
        [textStorage beginEditing];
        [str MO_standardizeEndOfLineToCRLF];
        [textStorage endEditing];
        [self didChangeText];
    }
}

- (void)TE_standardizeEndOfLineToCR:(id)sender {
    NSTextStorage *textStorage = [self textStorage];
    NSMutableString *str = [textStorage mutableString];
    NSRange charRange = NSMakeRange(0, [str length]);

    if ([self shouldChangeTextInRange:charRange replacementString:nil]) {
        // -MO_standardizeEndOfLineToCR might do many separate edits, so turn on the batching in textStorage here.
        [textStorage beginEditing];
        [str MO_standardizeEndOfLineToCR];
        [textStorage endEditing];
        [self didChangeText];
    }
}

- (void)TE_standardizeEndOfLineToParagraphSeparator:(id)sender {
    NSTextStorage *textStorage = [self textStorage];
    NSMutableString *str = [textStorage mutableString];
    NSRange charRange = NSMakeRange(0, [str length]);

    if ([self shouldChangeTextInRange:charRange replacementString:nil]) {
        // -MO_standardizeEndOfLineToParagraphSeparator might do many separate edits, so turn on the batching in textStorage here.
        [textStorage beginEditing];
        [str MO_standardizeEndOfLineToParagraphSeparator];
        [textStorage endEditing];
        [self didChangeText];
    }
}

- (void)TE_standardizeEndOfLineToLineSeparator:(id)sender {
    NSTextStorage *textStorage = [self textStorage];
    NSMutableString *str = [textStorage mutableString];
    NSRange charRange = NSMakeRange(0, [str length]);

    if ([self shouldChangeTextInRange:charRange replacementString:nil]) {
        // -MO_standardizeEndOfLineToLineSeparator might do many separate edits, so turn on the batching in textStorage here.
        [textStorage beginEditing];
        [str MO_standardizeEndOfLineToLineSeparator];
        [textStorage endEditing];
        [self didChangeText];
    }
}

- (void)TE_insertCRLF:(id)sender {
    if ([self isFieldEditor]) {
        // Field editors needs to do something special with newlines, and we don't know how.  Let insertNewline handle it, because it does know how.
        [self insertNewline:self];
    } else {
        [self insertText:@"\r\n"];
    }
}

- (void)TE_insertCR:(id)sender {
    if ([self isFieldEditor]) {
        // Field editors needs to do something special with newlines, and we don't know how.  Let insertNewline handle it, because it does know how.
        [self insertNewline:self];
    } else {
        [self insertText:@"\r"];
    }
}

- (void)TE_insertLineSeparator:(id)sender {
    if ([self isFieldEditor]) {
        // Field editors needs to do something special with newlines, and we don't know how.  Let insertNewline handle it, because it does know how.
        [self insertNewline:self];
    } else {
        unichar lineSeparator[1];

        lineSeparator[0] = NSLineSeparatorCharacter;

        [self insertText:[NSString stringWithCharacters:lineSeparator length:1]];
    }
}

- (void)TE_indentFriendlyDeleteBackward:(id)sender {
    if ([self isRichText]) {
        // This is not appropriate for rich text.
        [self deleteBackward:sender];
    } else {
        NSRange charRange = [self rangeForUserTextChange];
        if (charRange.location != NSNotFound) {
            if (charRange.length > 0) {
                // Non-zero selection.  Delete normally.
                [self deleteBackward:sender];
            } else {
                if (charRange.location == 0) {
                    // At beginning of text.  Delete normally.
                    [self deleteBackward:sender];
                } else {
                    NSString *string = [self string];
                    NSRange paraRange = [string lineRangeForRange:NSMakeRange(charRange.location - 1, 1)];
                    if (paraRange.location == charRange.location) {
                        // At beginning of line.  Delete normally.
                        [self deleteBackward:sender];
                    } else {
                        unsigned tabWidth = [[TEPreferencesController sharedPreferencesController] tabWidth];
                        unsigned indentWidth = [[TEPreferencesController sharedPreferencesController] indentWidth];
                        NSRange leadingSpaceRange = paraRange;
                        unsigned leadingSpaces = TE_numberOfLeadingSpacesFromRangeInString(string, &leadingSpaceRange, tabWidth);

                        if (charRange.location > NSMaxRange(leadingSpaceRange)) {
                            // Not in leading whitespace.  Delete normally.
                            [self deleteBackward:sender];
                        } else {
                            NSTextStorage *text = [self textStorage];
                            unsigned leadingIndents = leadingSpaces / indentWidth;
                            NSString *replaceString;

                            // If we were indented to an fractional level just go back to the last even multiple of indentWidth, if we were exactly on, go back a full level.
                            if (leadingSpaces % indentWidth == 0) {
                                leadingIndents--;
                            }
                            leadingSpaces = leadingIndents * indentWidth;
                            replaceString = ((leadingSpaces > 0) ? TE_tabbifiedStringWithNumberOfSpaces(leadingSpaces, tabWidth) : @"");
                            if ([self shouldChangeTextInRange:leadingSpaceRange replacementString:replaceString]) {
                                NSDictionary *newTypingAttributes;
                                if (charRange.location < [string length]) {
                                    newTypingAttributes = [[text attributesAtIndex:charRange.location effectiveRange:NULL] retain];
                                } else {
                                    newTypingAttributes = [[text attributesAtIndex:(charRange.location - 1) effectiveRange:NULL] retain];
                                }

                                [text replaceCharactersInRange:leadingSpaceRange withString:replaceString];

                                [self setTypingAttributes:newTypingAttributes];
                                [newTypingAttributes release];

                                [self didChangeText];
                            }
                        }
                    }
                }
            }
        }
    }
}

- (void)TE_reindentWrappedLines:(id)sender {
    NSTextStorage *text = [self textStorage];
    unsigned textLength = ((text != nil) ? [text length] : 0);

    if ((textLength == 0) || [self isRichText] || [self isFieldEditor]) {
        // Forget it.
        return;
    }
    // Just tickle the text storage to trick it into fixing attributes and notifying everybody.
    [text beginEditing];
    [text edited:NSTextStorageEditedCharacters range:NSMakeRange(0, textLength) changeInLength:0];
    [text endEditing];
}

@end
